/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkLagrangeQuadrilateral.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkLagrangeQuadrilateral.h"

#include "vtkDoubleArray.h"
#include "vtkQuad.h"
#include "vtkIdList.h"
#include "vtkLagrangeCurve.h"
#include "vtkLagrangeInterpolation.h"
#include "vtkMath.h"
#include "vtkObjectFactory.h"
#include "vtkPoints.h"
#include "vtkTriangle.h"
#include "vtkVector.h"
#include "vtkVectorOperators.h"

vtkStandardNewMacro(vtkLagrangeQuadrilateral);

vtkLagrangeQuadrilateral::vtkLagrangeQuadrilateral()
{
  this->Approx = NULL;
  this->Order[0] = this->Order[1] = this->Order[2] = 1;
  this->Points->SetNumberOfPoints(4);
  this->PointIds->SetNumberOfIds(4);
  for (int i = 0; i < 4; i++)
    {
    this->Points->SetPoint(i, 0.0, 0.0, 0.0);
    this->PointIds->SetId(i,-1);
    }
}

vtkLagrangeQuadrilateral::~vtkLagrangeQuadrilateral()
{
}

void vtkLagrangeQuadrilateral::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os,indent);
  os << indent << "Order: " << this->GetOrder(2) << "\n";
  os
    << indent << "PointParametricCoordinates: "
    << this->PointParametricCoordinates->GetNumberOfPoints()
    << " entries\n";
  os << indent << "Approx: " << this->Approx << "\n";
}

vtkCell* vtkLagrangeQuadrilateral::GetEdge(int edgeId)
{
  vtkLagrangeCurve* result = vtkLagrangeCurve::New();
  const int* order = this->GetOrder();
  vtkIdType npts = ((edgeId % 2 == 0) ? order[0] : order[1]) + 1;
  std::vector<vtkIdType> econn(npts);
  bool iVaries = (edgeId % 2 == 0 ? true : false);
  bool jVaries = !iVaries;
  for (int aa = 0; aa < npts; ++aa)
    {
    int pid = this->PointIndexFromIJK(
      iVaries ? aa : (edgeId == 1 ? order[0] : 0),
      jVaries ? aa : (edgeId == 2 ? order[1] : 0),
      0);
    econn[aa] = pid;
    }
  result->vtkCell::Initialize(npts, &econn[0], this->Points);
  return result;
}

void vtkLagrangeQuadrilateral::Initialize()
{
}

int vtkLagrangeQuadrilateral::CellBoundary(
  int subId, double pcoords[3], vtkIdList* pts)
{
  // TODO: Fill me in!
  return 0;
}

int vtkLagrangeQuadrilateral::EvaluatePosition(
  double* x,
  double* closestPoint,
  int& subId,
  double pcoords[3],
  double& minDist2,
  double* weights)
{
  int result = 0;

  int dummySubId;
  double linearWeights[4];
  double tmpDist2;
  vtkVector3d params;
  vtkVector3d tmpClosestPt;

  minDist2 = VTK_DOUBLE_MAX;
  vtkIdType nquad = vtkLagrangeInterpolation::NumberOfIntervals<2>(this->GetOrder());
  for (int subCell = 0; subCell < nquad; ++subCell)
    {
    vtkQuad* approx = this->GetApproximateQuad(subCell, NULL, NULL);
    int stat = approx->EvaluatePosition(x, tmpClosestPt.GetData(), dummySubId, params.GetData(), tmpDist2, linearWeights);
    if (stat != -1 && tmpDist2 < minDist2)
      {
      result = stat;
      subId = subCell;
      minDist2 = tmpDist2;
      for (int ii = 0; ii < 3; ++ii)
        {
        pcoords[ii] = params[ii]; // We will translate the winning parameter values later.
        if (closestPoint)
          {
          closestPoint[ii] = tmpClosestPt[ii];
          }
        }
      }
    }

  if (result != -1)
    {
    this->TransformApproxToCellParams(subId, pcoords);
    if (closestPoint)
      {
      this->EvaluateLocation(dummySubId, pcoords, closestPoint, weights);
      }
    else
      {
      this->InterpolateFunctions(pcoords, weights);
      }
    }

  return result;
}

void vtkLagrangeQuadrilateral::EvaluateLocation(
  int& subId,
  double pcoords[3],
  double x[3], double* weights)
{
  subId = 0; // TODO: Should this be -1?
  this->InterpolateFunctions(pcoords, weights);

  double p[3];
  x[0] = x[1] = x[2] = 0.;
  vtkIdType nPoints = this->GetPoints()->GetNumberOfPoints();
  for (vtkIdType idx = 0; idx < nPoints; ++idx)
    {
    this->Points->GetPoint(idx, p);
    for (vtkIdType jdx = 0; jdx < 3; ++jdx)
      {
      x[jdx] += p[jdx] * weights[idx];
      }
    }
}

void vtkLagrangeQuadrilateral::Contour(
  double value,
  vtkDataArray* cellScalars,
  vtkIncrementalPointLocator* locator,
  vtkCellArray* verts,
  vtkCellArray* lines,
  vtkCellArray* polys,
  vtkPointData* inPd,
  vtkPointData* outPd,
  vtkCellData* inCd,
  vtkIdType cellId,
  vtkCellData* outCd)
{
  vtkIdType nquad = vtkLagrangeInterpolation::NumberOfIntervals<2>(this->GetOrder());
  for (int i = 0; i < nquad; ++i)
    {
    vtkQuad* approx = this->GetApproximateQuad(i, cellScalars, this->Scalars.GetPointer());
    approx->Contour(value, this->Scalars.GetPointer(), locator, verts, lines, polys, inPd, outPd, inCd, cellId, outCd);
    }
}

void vtkLagrangeQuadrilateral::Clip(
  double value,
  vtkDataArray* cellScalars,
  vtkIncrementalPointLocator* locator,
  vtkCellArray* polys,
  vtkPointData* inPd,
  vtkPointData* outPd,
  vtkCellData* inCd,
  vtkIdType cellId,
  vtkCellData* outCd,
  int insideOut)
{
  vtkIdType nquad = vtkLagrangeInterpolation::NumberOfIntervals<2>(this->GetOrder());
  for (int i = 0; i < nquad; ++i)
    {
    vtkQuad* approx = this->GetApproximateQuad(i, cellScalars, this->Scalars.GetPointer());
    approx->Clip(value, this->Scalars.GetPointer(), locator, polys, inPd, outPd, inCd, cellId, outCd, insideOut);
    }
}

int vtkLagrangeQuadrilateral::IntersectWithLine(
  double* p1,
  double* p2,
  double tol,
  double& t,
  double* x,
  double* pcoords,
  int& subId)
{
  vtkIdType nquad = vtkLagrangeInterpolation::NumberOfIntervals<2>(this->GetOrder());
  double tFirst = VTK_DOUBLE_MAX;
  bool intersection = false;
  vtkVector3d tmpX;
  vtkVector3d tmpP;
  int tmpId;
  for (int i = 0; i < nquad; ++i)
    {
    vtkQuad* approx = this->GetApproximateQuad(i);
    if (approx->IntersectWithLine(p1, p2, tol, t, tmpX.GetData(), tmpP.GetData(), tmpId))
      {
      intersection = true;
      if (t < tFirst)
        {
        tFirst = t;
        for (int ii = 0; ii < 3; ++ii)
          {
          x[ii] = tmpX[ii];
          pcoords[ii] = tmpP[ii]; // Translate this after we're sure it's the closest hit.
          subId = tmpId;
          }
        }
      }
    }
  if (intersection)
    {
    this->TransformApproxToCellParams(subId, pcoords);
    }
  return intersection ? 1 : 0;
}

int vtkLagrangeQuadrilateral::Triangulate(
  int vtkNotUsed(index),
  vtkIdList* ptIds,
  vtkPoints* pts)
{
  ptIds->Reset();
  pts->Reset();

  vtkIdType nquad = vtkLagrangeInterpolation::NumberOfIntervals<2>(this->GetOrder());
  vtkVector3i ijk;
  for (int i = 0; i < nquad; ++i)
    {
    vtkQuad* approx = this->GetApproximateQuad(i);
    this->SubCellCoordinatesFromId(ijk, i);
    if (approx->Triangulate(
        (ijk[0] + ijk[1] + ijk[2]) % 2,
        this->TmpIds.GetPointer(),
        this->TmpPts.GetPointer()))
      {
      // Sigh. Triangulate methods all reset their points/ids
      // so we must copy them to our output.
      vtkIdType np = this->TmpPts->GetNumberOfPoints();
      vtkIdType ni = this->TmpIds->GetNumberOfIds();
      for (vtkIdType ii = 0; ii < np; ++ii)
        {
        pts->InsertNextPoint(this->TmpPts->GetPoint(ii));
        }
      for (vtkIdType ii = 0; ii < ni; ++ii)
        {
        ptIds->InsertNextId(this->TmpIds->GetId(ii));
        }
      }
    }
  return 1;
}

void vtkLagrangeQuadrilateral::Derivatives(
  int vtkNotUsed(subId),
  double vtkNotUsed(pcoords)[3],
  double* vtkNotUsed(values),
  int vtkNotUsed(dim),
  double* vtkNotUsed(derivs))
{
  // TODO: Fill me in?
  return;
}

double* vtkLagrangeQuadrilateral::GetParametricCoords()
{
  if (!this->PointParametricCoordinates)
    {
    this->PointParametricCoordinates = vtkSmartPointer<vtkPoints>::New();
    this->PointParametricCoordinates->SetDataTypeToDouble();
    }

  // Ensure Order is up-to-date and check that current point size matches:
  if (static_cast<int>(this->PointParametricCoordinates->GetNumberOfPoints()) != this->GetOrder(2))
    {
    this->PointParametricCoordinates->Initialize();
    vtkLagrangeInterpolation::AppendQuadrilateralCollocationPoints(
      this->PointParametricCoordinates, this->Order);
    }

  return
    vtkDoubleArray::SafeDownCast(
      this->PointParametricCoordinates->GetData())->GetPointer(0);
}

double vtkLagrangeQuadrilateral::GetParametricDistance(double pcoords[3])
{
  double pDist, pDistMax = 0.0;

  for (int ii = 0; ii < 2; ++ii)
    {
    pDist =
      (pcoords[ii] < 0. ? -pcoords[ii] :
       (pcoords[ii] > 1. ? pcoords[ii] - 1. :
        0.));
    if ( pDist > pDistMax )
      {
      pDistMax = pDist;
      }
    }

  // The quadrilateral's 3rd parametric coordinate should always be 0:
  if (pcoords[2] != 0.0 && (pDist = std::abs(pcoords[2])) > pDistMax)
    {
    pDistMax = pDist;
    }

  return pDistMax;
}

const int* vtkLagrangeQuadrilateral::GetOrder()
{
  // FIXME: The interpolation routines can handle different order along each axis
  //   but we cannot infer the order from the number of points in that case.
  //   This method currrently assumes quads are of the same order on each axis.
  //   We populate the Order array for use with the interpolation class.
  vtkIdType npts = this->Points->GetNumberOfPoints();
  if (this->Order[2] != npts)
    {
    int pointsPerAxis = static_cast<int>(ceil(pow(npts, 1./2.))); // number of points along each axis
    for (int i = 0; i < 2; ++i)
      {
      this->Order[i] = pointsPerAxis - 1; // order 1 is linear, 2 is quadratic, ...
      }
    this->Order[2] = static_cast<int>(npts);
    }
  return this->Order;
}

void vtkLagrangeQuadrilateral::InterpolateFunctions(
  double pcoords[3], double* weights)
{
  vtkLagrangeInterpolation::TensorShapeFunctions<2>(this->GetOrder(), pcoords, weights);
}

void vtkLagrangeQuadrilateral::InterpolateDerivs(
  double pcoords[3], double* derivs)
{
  vtkLagrangeInterpolation::TensorShapeDerivatives<2>(this->GetOrder(), pcoords, derivs);
}

/// Return a linear quadrilateral used to approximate a region of the nonlinear quadrilateral.
vtkQuad* vtkLagrangeQuadrilateral::GetApprox()
{
  if (!this->Approx)
    {
    this->Approx = vtkSmartPointer<vtkQuad>::New();
    }
  return this->Approx.GetPointer();
}

/**\brief Populate the linear quadrilateral returned by GetApprox() with point-data from one voxel-like interval of this cell.
  *
  * Ensure that you have called GetOrder() before calling this method
  * so that this->Order is up to date. This method does no checking
  * before using it to map connectivity-array offsets.
  */
vtkQuad* vtkLagrangeQuadrilateral::GetApproximateQuad(
  int subId, vtkDataArray* scalarsIn, vtkDataArray* scalarsOut)
{
  vtkQuad* approx = this->GetApprox();
  bool doScalars = (scalarsIn && scalarsOut);
  if (doScalars)
    {
    scalarsOut->SetNumberOfTuples(4);
    }
  int i, j, k;
  this->SubCellCoordinatesFromId(i, j, k, subId);
  // Get the point ids (and optionally scalars) for each of the 4 corners
  // in the approximating quadrilateral spanned by (i, i+1) x (j, j+1):
  vtkIdType aconn[4];
  for (int ic = 0; ic < 4; ++ic)
    {
    int corner = this->PointIndexFromIJK(
      i + (((ic + 1) / 2) % 2 ? 1 : 0),
      j + ((ic / 2) % 2 ? 1 : 0),
      0);
    aconn[ic] = this->PointIds->GetId(corner);
    if (doScalars)
      {
      scalarsOut->SetTuple(ic,
        scalarsIn->GetTuple(
          corner));
      }
    }
  approx->Initialize(4, &aconn[0], this->Points);
  return approx;
}

/// A convenience method; see the overloaded variant for more information.
bool vtkLagrangeQuadrilateral::SubCellCoordinatesFromId(vtkVector3i& ijk, int subId)
{
  return this->SubCellCoordinatesFromId(ijk[0], ijk[1], ijk[2], subId);
}

/**\brief Given an integer specifying an approximating linear quad, compute its IJK coordinate-position in this cell.
 *
 * The \a subId specifies the lower-, left-, front-most vertex of the approximating quad.
 * This sets the ijk coordinates of that point.
 *
 * You must have called this->GetOrder() **before** invoking this method so that the order will be up to date.
 */
bool vtkLagrangeQuadrilateral::SubCellCoordinatesFromId(int& i, int& j, int& k, int subId)
{
  if (subId < 0)
    {
    return false;
    }

  i = subId % this->Order[0];
  j = (subId / this->Order[0]) % this->Order[1];
  k = 0;
  return i + this->Order[0] * j == subId ? true : false;
}

/**\brief Given (i,j,k) coordinates within the Lagrange quad, return an offset into the local connectivity (PointIds) array.
  *
  * Ensure that you have called GetOrder() before calling this method
  * so that this->Order is up to date. This method does no checking
  * before using it to map connectivity-array offsets.
  */
int vtkLagrangeQuadrilateral::PointIndexFromIJK(int i, int j, int k)
{
  bool ibdy = (i == 0 || i == this->Order[0]);
  bool jbdy = (j == 0 || j == this->Order[1]);
  // How many boundaries do we lie on at once?
  int nbdy = (ibdy ? 1 : 0) + (jbdy ? 1 : 0);

  if (nbdy == 2) // Vertex DOF
    { // ijk is a corner node. Return the proper index (somewhere in [0,7]):
    return (i ? (j ? 2 : 1) : (j ? 3 : 0));
    }

  int offset = 4;
  if (nbdy == 1) // Edge DOF
    {
    if (!ibdy)
      { // On i axis
      return (i - 1) +
        (j ? this->Order[0] - 1 + this->Order[1] - 1 : 0) +
        offset;
      }
    if (!jbdy)
      { // On j axis
      return (j - 1) +
        (i ? this->Order[0] - 1 : 2 * (this->Order[0] - 1) + this->Order[1] - 1) +
        offset;
      }
    }

  offset += 2 * (this->Order[0] - 1 + this->Order[1] - 1);
  // nbdy == 0: Face DOF
  return offset +
    (i - 1) + (this->Order[0] - 1) * (
      (j - 1));
}

/**\brief Given the index, \a subCell, of a linear approximating-quad, translate pcoords from that quad into this nonlinear quad.
  *
  * You must call this->GetOrder() **before** invoking this method as it assumes
  * the order is up to date.
  */
bool vtkLagrangeQuadrilateral::TransformApproxToCellParams(int subCell, double* pcoords)
{
  vtkVector3i ijk;
  if (!this->SubCellCoordinatesFromId(ijk, subCell))
    {
    return false;
    }
  for (int pp = 0; pp < 2; ++pp)
    {
    pcoords[pp] = (pcoords[pp] + ijk[pp]) / this->Order[pp];
    }
  pcoords[2] = 0.;
  return true;
}
