/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkStyleData.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class   vtkStyleData
 * @brief   Hold collections of computed styles generated by parsing CSS.
 *
 * vtkStyleData is an object that holds the output state of a CSS parser.
 * It is templated on the document object model (DOM), which provides
 * methods to traverse the document so that CSS selectors can be applied.
 *
 * vtkStyleData consists of a collection of styles.
 * Each style in the collection has a unique set of CSS properties.
 * When a dataset in a vtkDataAssembly needs to be rendered, its style
 * is obtained from this object by hashing the classes, attributes and
 * element type of the assembly entry. The hash lookup returns a set of
 * properties and these are used to configure the low-level renderer that
 * handles the dataset.
 */

#ifndef vtkStyleData_h
#define vtkStyleData_h

#include "vtkObject.h"
#include "vtkRenderingCoreModule.h" // For export macro

#include <functional> // For std::function.
#include <map> // For Styles and Properties maps.
#include <set> // For set<Selector>.
#include <set> // For vector<Hashable>.

template<typename DOM>
class VTKRENDERINGCORE_EXPORT vtkStyleData : public vtkObject
{
public:
  vtkTypeMacro(vtkStyleData<DOM>, vtkObject);
  static vtkStyleData* New();
  void PrintSelf(ostream& os, vtkIndent indent) override;

  bool SetMarkup(const std::string& markup);
  const std::string& GetMarkup() const { return this->Markup; }

protected:
  vtkStyleData();
  ~vtkStyleData() override;

  using DOMNode = typename DOM::Node;

  bool Parse(const std::string& markup);

  struct Rule
  {
    using SelectorLambda = std::function<bool(DOMNode&)>;
    std::set<SelectorLambda> Selectors;
    // std::map<PropertyKey, PropertyValue> Properties;

    bool operator < (const Rule& other) const
    {
      // TODO: We must sort rules in cascading order:
      //       https://developer.mozilla.org/en-US/docs/Web/CSS/Cascade#cascading_order
      return this < &other; // FIXME
    }
  };

  using StyleHash = std::size_t;

  struct Style
  {
    /// Property "names" are string hashes.
    using PropertyKey = std::size_t;
    /// A property value is a string (for now).
    using PropertyValue = std::string; // Variant<bool, std::string, FPNumberWithUnits, IntNumberWithUnits>;
    /// A vector holding all the bytes to hash.
    ///
    /// The bytes include:
    /// + the Element type (as UTF8 characters)
    /// + the number of attributes (an unsigned int)
    ///   + each attribute name followed by the attribute's string value
    std::vector<std::uint8_t> Hashable;
    std::map<PropertyKey, PropertyValue> Properties;
  };

  std::string Markup; // utf-8 CSS text
  // Keep @imported sheets plus their hashes here?
  std::set<Rule> Rules; // must be ordered so cascading works.
  std::map<StyleHash, Style> Styles; // unordered map or set might be better

private:
  vtkStyleData(const vtkStyleData&) = delete;
  void operator=(const vtkStyleData&) = delete;
};

#endif
