/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkDataAssemblyMapper.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class   vtkDataAssemblyMapper
 * @brief   map vtkDataAssembly to graphics primitives
 *
 * vtkDataAssemblyMapper is a class that maps a data assembly (i.e.,
 * vtkPartitionedDataSetCollection organized into a hierarchy) to
 * graphics primitives. vtkDataAssemblyMapper serves as a superclass for
 * device-specific mappers that actually do the mapping to the
 * rendering/graphics hardware/software.
 */

#ifndef vtkDataAssemblyMapper_h
#define vtkDataAssemblyMapper_h

#include "vtkMapper.h"
#include "vtkRenderingCoreModule.h" // For export macro
//#include "vtkTexture.h" // used to include texture unit enum.
#include "vtkStyleData.h" // a templated class held as an ivar.
#include "vtkDataAssemblyDOM.h" // used as a vtkStyleData template param.

class vtkDataAssembly;
class vtkPartitionedDataSetCollection;
class vtkRenderer;
class vtkRenderWindow;
template<typename DOM> class vtkStyleData;

class VTKRENDERINGCORE_EXPORT vtkDataAssemblyMapper : public vtkMapper
{
public:
  static vtkDataAssemblyMapper* New();
  vtkTypeMacro(vtkDataAssemblyMapper, vtkMapper);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  /**
   * Render a piece of the input data using Styles computed from Markup.
   *
   * This method configures then invokes other renderers.
   */
  virtual void RenderPiece(vtkRenderer*, vtkActor*);

  /**
   * This calls RenderPiece (in a for loop if streaming is necessary).
   */
  void Render(vtkRenderer* ren, vtkActor* act) override;

  ///@{
  /**
   * Specify the input data to map.
   * You must also provide an input vtkDataAssembly and
   * you may also provide CSS markup.
   */
  void SetInputData(vtkPartitionedDataSetCollection* in);
  vtkPartitionedDataSetCollection* GetInput();
  ///@}

  ///@{
  /**
   * Specify the hierarchy formed from the input.
   */
  void SetInputAssembly(vtkDataAssembly* in);
  vtkDataAssembly* GetInputAssembly();
  ///@}

  ///@{
  /**
   * Specify the visual style to use when mapping.
   * This string should be valid CSS encoded as UTF-8 text.
   */
  void SetMarkup(const std::string& markup);
  const std::string& GetMarkup() const;
  ///@}

  ///@{
  /**
   * Bring this algorithm's outputs up-to-date.
   */
  void Update(int port) override;
  void Update() override;
  vtkTypeBool Update(int port, vtkInformationVector* requests) override;
  vtkTypeBool Update(vtkInformation* requests) override;
  ///@}

  ///@{
  /**
   * If you want only a part of the data, specify by setting the piece.
   */
  vtkSetMacro(Piece, int);
  vtkGetMacro(Piece, int);
  vtkSetMacro(NumberOfPieces, int);
  vtkGetMacro(NumberOfPieces, int);
  vtkSetMacro(NumberOfSubPieces, int);
  vtkGetMacro(NumberOfSubPieces, int);
  ///@}

  ///@{
  /**
   * Set the number of ghost cells to return.
   */
  vtkSetMacro(GhostLevel, int);
  vtkGetMacro(GhostLevel, int);
  ///@}

  /**
   * Return bounding box (array of six doubles) of data expressed as
   * (xmin,xmax, ymin,ymax, zmin,zmax).
   */
  double* GetBounds() VTK_SIZEHINT(6) override;
  void GetBounds(double bounds[6]) override { this->Superclass::GetBounds(bounds); }

  /**
   * Make a shallow copy of this mapper.
   */
  void ShallowCopy(vtkAbstractMapper* m) override;

  /**
   * see vtkAlgorithm for details
   */
  vtkTypeBool ProcessRequest(
    vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;

  /// Set the object used to hold parsed CSS styles.
  /// A styles object is created automatically; this method is only exposed
  /// for potential sharing of CSS markup among mappers.
  void SetStyles(vtkSmartPointer<vtkStyleData> styles);
  /// Return the object used to hold parsed CSS styles.
  vtkSmartPointer<vtkStyleData> GetStyles() const { return this->Styles; }
protected:
  vtkDataAssemblyMapper();
  ~vtkDataAssemblyMapper() override;

  /**
   * Called in GetBounds(). When this method is called, the consider the input
   * to be updated depending on whether this->Static is set or not. This method
   * simply obtains the bounds from the data-object and returns it.
   */
  virtual void ComputeBounds();

  int Piece;
  int NumberOfPieces;
  int NumberOfSubPieces;
  int GhostLevel;
  vtkDataAssembly* InputAssembly;
  vtkTimeStamp RenderTime; // Last time Render occurred.
  vtkSmartPointer<vtkStyleData> Styles;

  int FillInputPortInformation(int, vtkInformation*) override;

private:
  vtkDataAssemblyMapper(const vtkDataAssemblyMapper&) = delete;
  void operator=(const vtkDataAssemblyMapper&) = delete;
};

#endif
