/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkDataAssemblyMapper.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkDataAssemblyMapper.h"

#include "vtkExecutive.h"
#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkMath.h"
#include "vtkObjectFactory.h"
#include "vtkDataAssembly.h"
#include "vtkPartitionedDataSetCollection.h"
#include "vtkRenderWindow.h"
#include "vtkStreamingDemandDrivenPipeline.h"
#include "vtkStyleData.h"

vtkObjectFactoryNewMacro(vtkDataAssemblyMapper);
vtkCxxSetObjectMacro(vtkDataAssemblyMapper, InputAssembly, vtkDataAssembly);

vtkDataAssemblyMapper::vtkDataAssemblyMapper()
{
  this->Piece = 0;
  this->NumberOfPieces = 1;
  this->NumberOfSubPieces = 1;
  this->GhostLevel = 0;
  this->InputAssembly = nullptr;
  this->Styles = vtkSmartPointer<vtkStyleData>::New();
}

vtkDataAssemblyMapper::~vtkDataAssemblyMapper()
{
  this->SetInputAssembly(nullptr);
}

void vtkDataAssemblyMapper::RenderPiece(vtkRenderer*, vtkActor*)
{
  // TODO
  // Basic algorithm:
  //   + Loop over collections in assembly (depth-first search).
  //     + If collecion has piece, find style for collection
  //     + If SD = (style-hash, collection-pointer) have a sub-mapper Ms,
  //       + If MTime(SD) > MTime(Ms), then update Ms.
  //     + else (Ms doesn't exist for SD)
  //       + Create Ms
  //       + Update Ms with SD
  //     + Invoke Ms->Render()
}

void vtkDataAssemblyMapper::Render(vtkRenderer* ren, vtkActor* act)
{
  if (!this->InputAssembly)
  {
    return;
  }

  if (this->InputAssembly->GetMTime() > this->Styles->GetMTime())
  {
    // compute styles from rules using InputAssembly DOM.
  }

  if (this->Styles->GetMTime() > this->RenderTime)
  {
    std::cout << "Apply computed styles\n";
    // apply computed styles to sub-mappers.
  }

  if (this->Static)
  {
    this->RenderPiece(ren, act);
    this->RenderTime.Modified();
    return;
  }

  vtkInformation* inInfo = this->GetInputInformation();
  if (inInfo == nullptr)
  {
    vtkErrorMacro("Mapper has no input.");
    return;
  }

  int nPieces = this->NumberOfPieces * this->NumberOfSubPieces;
  for (int i = 0; i < this->NumberOfSubPieces; i++)
  {
    // If more than one piece, render in a loop.
    int currentPiece = this->NumberOfSubPieces * this->Piece + i;
    this->GetInputAlgorithm()->UpdateInformation();
    inInfo->Set(vtkStreamingDemandDrivenPipeline::UPDATE_PIECE_NUMBER(), currentPiece);
    inInfo->Set(vtkStreamingDemandDrivenPipeline::UPDATE_NUMBER_OF_PIECES(), nPieces);
    inInfo->Set(
      vtkStreamingDemandDrivenPipeline::UPDATE_NUMBER_OF_GHOST_LEVELS(), this->GhostLevel);
    this->RenderPiece(ren, act);
  }
  this->RenderTime.Modified();
}

void vtkDataAssemblyMapper::SetMarkup(const std::string& markup)
{
  this->Styles->SetMarkup(markup);
}

const std::string& vtkDataAssemblyMapper::GetMarkup() const
{
  return this->Styles->GetMarkup();
}

void vtkDataAssemblyMapper::SetInputData(vtkPartitionedDataSetCollection* input)
{
  this->SetInputDataInternal(0, input);
}

vtkPartitionedDataSetCollection* vtkDataAssemblyMapper::GetInput()
{
  return vtkPartitionedDataSetCollection::SafeDownCast(this->GetExecutive()->GetInputData(0, 0));
}

vtkTypeBool vtkDataAssemblyMapper::ProcessRequest(
  vtkInformation* request, vtkInformationVector** inputVector, vtkInformationVector*)
{
  if (request->Has(vtkStreamingDemandDrivenPipeline::REQUEST_UPDATE_EXTENT()))
  {
    vtkInformation* inInfo = inputVector[0]->GetInformationObject(0);
    int currentPiece = this->NumberOfSubPieces * this->Piece;
    inInfo->Set(vtkStreamingDemandDrivenPipeline::UPDATE_PIECE_NUMBER(), currentPiece);
    inInfo->Set(vtkStreamingDemandDrivenPipeline::UPDATE_NUMBER_OF_PIECES(),
      this->NumberOfSubPieces * this->NumberOfPieces);
    inInfo->Set(
      vtkStreamingDemandDrivenPipeline::UPDATE_NUMBER_OF_GHOST_LEVELS(), this->GhostLevel);
  }
  return 1;
}

double* vtkDataAssemblyMapper::GetBounds()
{
  // do we have an input
  if (!this->GetNumberOfInputConnections(0))
  {
    vtkMath::UninitializeBounds(this->Bounds);
    return this->Bounds;
  }
  else
  {
    if (!this->Static)
    {
      vtkInformation* inInfo = this->GetInputInformation();
      if (inInfo)
      {
        this->GetInputAlgorithm()->UpdateInformation();
        int currentPiece = this->NumberOfSubPieces * this->Piece;
        this->GetInputAlgorithm()->UpdatePiece(
          currentPiece, this->NumberOfSubPieces * this->NumberOfPieces, this->GhostLevel);
      }
    }
    this->ComputeBounds();

    // if the bounds indicate NAN and subpieces are being used then
    // return nullptr
    if (!vtkMath::AreBoundsInitialized(this->Bounds) && this->NumberOfSubPieces > 1)
    {
      return nullptr;
    }
    return this->Bounds;
  }
}

void vtkDataAssemblyMapper::SetStyles(vtkSmartPointer<vtkStyleData> styles)
{
  if (this->Styles == styles)
  {
    return;
  }
  this->Styles = styles;
  this->Modified(); // TODO: Should this really happen? It could cause expensive updates to happen.
}

void vtkDataAssemblyMapper::ComputeBounds()
{
  vtkPartitionedDataSetCollection* input = this->GetInput();
  if (input)
  {
    input->GetBounds(this->Bounds);
  }
  else
  {
    vtkMath::UninitializeBounds(this->Bounds);
  }
}

void vtkDataAssemblyMapper::ShallowCopy(vtkAbstractMapper* mapper)
{
  vtkDataAssemblyMapper* m = vtkDataAssemblyMapper::SafeDownCast(mapper);
  if (m != nullptr)
  {
    this->SetInputConnection(m->GetInputConnection(0, 0));
    this->SetGhostLevel(m->GetGhostLevel());
    this->SetNumberOfPieces(m->GetNumberOfPieces());
    this->SetNumberOfSubPieces(m->GetNumberOfSubPieces());
    this->SetStyles(m->GetStyles());
    this->SetMarkup(m->GetMarkup());
  }

  // Now do superclass
  this->vtkMapper::ShallowCopy(mapper);
}

void vtkDataAssemblyMapper::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);

  os << indent << "Piece : " << this->Piece << endl;
  os << indent << "NumberOfPieces : " << this->NumberOfPieces << endl;
  os << indent << "GhostLevel: " << this->GhostLevel << endl;
  os << indent << "Number of sub pieces: " << this->NumberOfSubPieces << endl;
}

int vtkDataAssemblyMapper::FillInputPortInformation(int vtkNotUsed(port), vtkInformation* info)
{
  info->Set(vtkAlgorithm::INPUT_REQUIRED_DATA_TYPE(), "vtkPartitionedDataSetCollection");
  return 1;
}

void vtkDataAssemblyMapper::Update(int port)
{
  if (this->Static)
  {
    return;
  }
  this->Superclass::Update(port);
}

void vtkDataAssemblyMapper::Update()
{
  if (this->Static)
  {
    return;
  }
  this->Superclass::Update();
}

vtkTypeBool vtkDataAssemblyMapper::Update(int port, vtkInformationVector* requests)
{
  if (this->Static)
  {
    return 1;
  }
  return this->Superclass::Update(port, requests);
}

vtkTypeBool vtkDataAssemblyMapper::Update(vtkInformation* requests)
{
  if (this->Static)
  {
    return 1;
  }
  return this->Superclass::Update(requests);
}
