/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkDataAssembly.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class vtkDataAssembly
 * @brief hierarchical representation to use with
 * vtkPartitionedDataSetCollection
 *
 * vtkDataAssembly is a mechanism to represent hierarchical organization of
 * items (or vtkPartitionedDataSet instances) in a vtkPartitionedDataSetCollection.
 * vtkPartitionedDataSetCollection is similar to a vtkMultiBlockDataSet since it
 * provides a means for putting together multiple non-composite datasets.
 * However, vtkPartitionedDataSetCollection itself doesn't provide any mechanism
 * to define relationships between items in the collections. That is done using
 * vtkDataAssembly.
 *
 * @section Overview Overview
 *
 * At its core, vtkDataAssembly is simply a tree comprising on nodes starting
 * with the root node. Each node has a unique id and a string name (names need not
 * be unique). On initialization, `vtkDataAssembly::Initialize` a empty tree
 * with a root node is created. The root node's id and name can be obtained
 * using `vtkDataAssembly::GetRootNode` and `vtkDataAssembly::GetRootNodeName`.
 * The root node's id is fixed (vtkDataAssembly::GetRootNode), however the name
 * can be changed using `vtkDataAssembly::SetRootNodeName`.
 *
 * Child nodes can be added using `vtkDataAssembly::AddNode` or
 * `vtkDataAssembly::AddNodes`, each of which returns the ids for every child
 * node added. A non-root node can be removed using `vtkDataAssembly::RemoveNode`.
 *
 * Each node in the tree (including the root node) can has associated dataset
 * indices. For a vtkDataAssembly associated with a
 * vtkPartitionedDataSetCollection, these indices refer to the item index, or
 * partioned-dataset-index for items in the collection. Dataset indices can be
 * specified using `vtkDataAssembly::AddDataSetIndex`,
 * `vtkDataAssembly::AddDataSetIndices` and removed using `vtkDataAssembly::RemoveDataSetIndex`,
 * `vtkDataAssembly::RemoveAllDataSetIndices`.
 * `vtkDataAssembly::GetDataSetIndices` provides a mechanism to get the
 * database indices associated with a node, and optionally, the entire subtree
 * rooted at the chosen node.
 *
 * @section Searching Searching
 *
 * Each node in the vtkDataAssembly is assigned a unique id.
 * `vtkDataAssembly::FindFirstNodeWithName` and
 * `vtkDataAssembly::FindNodesWithName` can be used to get the id(s) for
 * node(s) with given name.
 *
 * `vtkDataAssembly::SelectNodes` provides a
 * more flexible mechanism to find nodes using name-based queries. Section
 * @ref DataAssemblyPathQueries covers supported queries.
 *
 * @section Traversal Traversal
 *
 * `vtkDataAssemblyVisitor` defines a visitor API. An instance of concretized
 * `vtkDataAssemblyVisitor` subclass can be passed to `vtkDataAssembly::Visit`
 * to traverse the data-assembly hierarchy either in depth-first or
 * breadth-first order.
 *
 * @section DataAssemblyPathQueries Supported Path Queries
 *
 * `vtkDataAssembly::SelectNodes` can be used find nodes that match the
 * specified query (or queries) using XPath 1.0 syntax.
 *
 * For example:
 *
 * * '/' is used as the path separator. If a node name has a `/` it must be
 * escaped using `\\` in the query. Note, escaping is not necessary when using
 * `SetNodeName`/`GetNodeName`.
 *
 * * '/' selects the root node.
 *
 * * '/nodename' selects all child nodes of the root with the name 'nodename'.
 *
 * * '//nodename' selects all nodes with 'nodename' that are descendants of the
 *   root; thus, this this will traverse the entire tree.
 *
 * * '/nodename/' selects all child nodes of all nodes named 'nodename' under
 *   the root; thus, ending a query with '/' selects the children of the found
 *   nodes rather than the nodes themselves.
 *
 * * '/nodename1/nodename2' selects all nodes named 'nodename2' which are
 *   childen of nodes with name 'nodename1' that are themselves children of the
 *   root node.
 *
 * * '//nodename1/nodename2' finds all nodes in the tree named 'nodename1' and
 *   then selects all chidren of these found nodes that are named 'nodename2'.
 *
 * @section Applications Applications
 *
 * The separation of dataset storage (vtkPartitionedDataSetCollection) and
 * organization (vtkDataAssembly) enables development of algorithms that can
 * expose APIs that are not tightly coupled to dataset storage. Together,
 * vtkPartitionedDataSetCollection and vtkDataAssembly can be thought of as a
 * different way of organizing data that was previously organized as a
 * vtkMultiBlockDataSet. The advantage of the this newer approach is that
 * filters can support specifying parameters using paths or path queries
 * rather than composite indices. The composite indices suffered from the fact
 * that they made no sense except for the specific vtkMultiBlockDataSet they
 * were applied too. Thus, if the filters input was changed, the composite ids
 * rarely made any sense and needed to be updated. Paths and path queries,
 * however, do not suffer from this issue.
 *
 */

#ifndef vtkDataAssembly_h
#define vtkDataAssembly_h

#include "vtkCommonDataModelModule.h" // for export macros
#include "vtkObject.h"

#include <functional> // for std::function
#include <map>    // for std::map
#include <memory> // for std::unique_ptr
#include <set> // for std::set
#include <string> // for std::string
#include <vector> // for std::vector

class vtkDataAssemblyVisitor;

class VTKCOMMONDATAMODEL_EXPORT vtkDataAssembly : public vtkObject
{
public:
  static vtkDataAssembly* New();
  vtkTypeMacro(vtkDataAssembly, vtkObject);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  /// Types of events that can affect user data stored on nodes.
  enum NodeEvent
  {
    ANCESTOR_MODIFIED, //!< An ancestor of this node has been marked MODIFIED.
    MODIFIED,          //!< SetNodeName, SetAttribute, or class-modification methods have run.
    REMOVED,           //!< RemoveNode has run.
    CLONED             //!< AddSubtree has copied a node.
  };

  /// The signature for node events.
  ///
  /// The parameters passed are:
  ///
  /// + The user data.
  /// + The event type.
  /// + The source assembly (for all events).
  /// + The source node ID (for all events).
  /// + The destination assembly (for CLONED events).
  /// + The destination node ID (for CLONED events).
  ///
  /// Handlers should return a new or updated user-data pointer when
  /// the event type is MODIFIED or CLONED. The return value is ignored
  /// for REMOVED events.
  /// When MODIFIED, the handler's return value replaces the existing value.
  /// When CLONED, the handler's return value is added to the new assembly node.
  using UserDataHandler =
    std::function<void*(void*, NodeEvent, vtkDataAssembly*, int, vtkDataAssembly*, int)>;

  /**
   * Initializes the data-assembly. When a new vtkDataAssembly instance is
   * created, it is in initialized form and it is not required to call this
   * method to initialize it.
   */
  void Initialize();

  ///@{
  /**
   * Initializes a data-assembly using an XML representation of the assembly.
   * Returns true if the initialization was successful, otherwise the assembly
   * is set a clean state and returns false.
   */
  bool InitializeFromXML(const char* xmlcontents);
  ///@}

  ///@{
  /**
   * Saves the data-assembly as a XML.
   */
  std::string SerializeToXML(vtkIndent indent) const;
  ///@}

  /**
   * Returns the ID for the root node.
   * This always returns 0.
   */
  static int GetRootNode() { return 0; }

  ///@{
  /**
   * Get/Set root node name. Defaults to DataAssembly.
   */
  void SetRootNodeName(const char* name) { this->SetNodeName(this->GetRootNode(), name); }
  const char* GetRootNodeName() const { return this->GetNodeName(this->GetRootNode()); }
  ///@}

  /**
   * Adds a node to the assembly with the given name and returns its id. `parent` is the id for
   * the parent node which defaults to the root node id (i.e. `GetRootNode`).
   *
   * If `parent` is invalid, the add will fail.
   *
   * @returns id of the newly added node on success, else -1.
   */
  int AddNode(const char* name, int parent = 0);

  /**
   * Same as `AddNode` except allows adding multiple nodes in one go.
   *
   * If `parent` is invalid, the add will fail.
   *
   * @returns vectors of corresponding indices for each of the `names` added.
   */
  std::vector<int> AddNodes(const std::vector<std::string>& names, int parent = 0);

  /**
   * Add a subtree by copy the nodes from another tree starting with the
   * specified parent index.
   */
  int AddSubtree(int parent, vtkDataAssembly* other, int otherParent = 0);

  /**
   * Removes a node from the assembly. The node identified by the id and all its
   * children are removed.
   *
   * Root node cannot be removed.
   *
   * @returns true if removal was successful, false if the id is invalid or root
   * node id.
   */
  bool RemoveNode(int id);

  ///@{
  /**
   * Get/Set a node's name. If node id is invalid, `SetNodeName` will raise an
   * error; `GetNodeName` will also raise an error and return nullptr.
   *
   * `SetNodeName` will raise an error if the name is not valid. Name cannot be
   * empty or nullptr.
   */
  void SetNodeName(int id, const char* name);
  const char* GetNodeName(int id) const;
  ///@}

  /**
   * Returns the path for a node.
   */
  std::string GetNodePath(int id) const;

  /**
   * Return a node id given the path. Returns `-1` if path is not valid.
   */
  int GetFirstNodeByPath(const char* path) const;

  /**
   * Add a dataset index to a node. The node id can refer to any
   * valid node in the assembly, including the root.
   *
   * While the same dataset can be added multiple times in the assembly,
   * it cannot be added multiple times to the same node. Additional adds
   * will fail.
   *
   * @returns true if addition succeeded else false.
   */
  bool AddDataSetIndex(int id, unsigned int dataset_index);

  /**
   * Same as `AddDataSetIndex` except supports adding multiple dataset indices
   * in one go. Note, a dataset index only gets added once.
   *
   * @returns true if any dataset index was successfully added.
   */
  bool AddDataSetIndices(int id, const std::vector<unsigned int>& dataset_indices);

  /**
   * Same as `AddDataSetIndices` except this supports adding a contiguous range of dataset
   * indices in one go.
   *
   * @ returns true if any dataset index was successfully added.
   */
  bool AddDataSetIndexRange(int id, unsigned int index_start, int count);

  /**
   * Removes a dataset index from a node.
   *
   * @returns true if the removal was successful, else returns false.
   */
  bool RemoveDataSetIndex(int id, unsigned int dataset_index);

  /**
   * Clears all dataset indices from the node.
   *
   * If `traverse_subtree` is true (default), recursively removes all
   * dataset indices from all the child nodes.
   *
   * @returns true on success, else returns false.
   */
  bool RemoveAllDataSetIndices(int id, bool traverse_subtree = true);

  enum TraversalOrder
  {
    DepthFirst = 0,
    BreadthFirst
  };

  /**
   * Finds first node that is encountered in a breadth first traversal
   * of the assembly with the given name.
   *
   * @returns the if of the node if found, else -1.
   */
  int FindFirstNodeWithName(
    const char* name, int traversal_order = vtkDataAssembly::TraversalOrder::DepthFirst) const;

  /**
   * Finds all nodes with the given name. The nodes can be ordered
   * depth first or breadth first, based on the `sort_order` flag.
   */
  std::vector<int> FindNodesWithName(
    const char* name, int sort_order = vtkDataAssembly::TraversalOrder::DepthFirst) const;

  /**
   * Returns ids for all child nodes.
   *
   * If `traverse_subtree` is true (default), recursively builds the child node
   * list. The traversal order can be specified using `traversal_order` flag;
   * defaults to depth-first.
   *
   * @sa vtkDataAssembly::Visit, vtkDataAssemblyVisitor
   *
   */
  std::vector<int> GetChildNodes(int parent, bool traverse_subtree = true,
    int traversal_order = vtkDataAssembly::TraversalOrder::DepthFirst) const;

  /**
   * Returns the number of child nodes.
   *
   * @returns the number of child nodes for the parent node or 0 if the parent
   * is invalid.
   */
  int GetNumberOfChildren(int parent) const;

  /**
   * Returns the id for a child not at the given index, if valid, otherwise -1.
   */
  int GetChild(int parent, int index) const;

  /**
   * Returns the index for a child under a given. -1 if invalid.
   */
  int GetChildIndex(int parent, int child) const;

  /**
   * Returns the id for the parent node, if any.
   * Returns -1 if the node is invalid or has no parent (i.e. is the root node).
   */
  int GetParent(int id) const;

  /**
   * Returns true if attribute with the given name is present
   * on the chosen node.
   */
  bool HasAttribute(int id, const char* name) const;

  ///@{
  /**
   * Set an attribute. Will replace an existing attribute with the same name if
   * present.
   */
  void SetAttribute(int id, const char* name, const char* value);
  void SetAttribute(int id, const char* name, int value);
  void SetAttribute(int id, const char* name, unsigned int value);
#if VTK_ID_TYPE_IMPL != VTK_INT
  void SetAttribute(int id, const char* name, vtkIdType value);
#endif
  ///@}

  ///@{
  /**
   * Set an attribute. Will replace an existing attribute with the same name if
   * present.
   */
  void RemoveAttribute(int id, const char* name);
  ///@}

  ///@{
  /**
   * Get an attribute value. Returns true if a value was provided else false.
   */
  bool GetAttribute(int id, const char* name, const char*& value) const;
  bool GetAttribute(int id, const char* name, int& value) const;
  bool GetAttribute(int id, const char* name, unsigned int& value) const;
#if VTK_ID_TYPE_IMPL != VTK_INT
  bool GetAttribute(int id, const char* name, vtkIdType& value) const;
#endif
  ///@}

  ///@{
  /**
   * Get an attribute value. Returns the value associated with the node or the
   * provided default value.
   */
  const char* GetAttributeOrDefault(int id, const char* name, const char* default_value) const;
  int GetAttributeOrDefault(int id, const char* name, int default_value) const;
  unsigned int GetAttributeOrDefault(int id, const char* name, unsigned int default_value) const;
#if VTK_ID_TYPE_IMPL != VTK_INT
  vtkIdType GetAttributeOrDefault(int id, const char* name, vtkIdType default_value) const;
#endif
  ///@}

  /**
   * Returns true if the class string-attribute is present
   * on node \a id.
   */
  bool HasClasses(int id) const;

  /**
   * Return true if the class string-attribute is present
   * **and** contains the \a className identifier on node \a id.
   */
  bool HasClass(int id, const char* className) const;

  ///@{
  /**
   * Get the list of all classes assigned to a node.
   */
  bool GetClasses(int id, const char*& classes) const;
  bool GetClasses(int id, std::vector<std::string>& classes) const;
  bool GetClasses(int id, std::set<std::string>& classes) const;
  ///@}

  ///@{
  /**
   * Replace (set), append, or remove a list of classes to a
   * node's class-attribute. If the given classes are already
   * present when adding (or already absent when removing),
   * this method has no effect.
   *
   * If you pass an empty or null string to SetClasses, all classes will be removed.
   *
   * Some variants take a \a validate parameter.
   * This is used to verify that (1) class names are valid CSS identifiers, and
   * (2) that class names are listed alphabetically and separated minimally
   * (i.e., with a single space instead of other whitespace character(s)).
   * These checks are the default but they add overhead.
   *
   * Validation is especially important as a potential optimization in the future
   * is to hash attribute values for faster comparisons.
   * If class names are not properly ordered and separated,
   * the hash will not be unique.
   *
   * Some variants accept a single string, which is treated as a
   * space-separated list of classes.
   * Other variants accept a vector of strings, which are treated
   * as pre-tokenized (although they may be validated as conforming
   * CSS identifiers to prevent subtle errors).
   *
   * These methods return true on success and false otherwise.
   * Note that a return value of true does not indicate the object was modified.
   */
  bool SetClasses(int id, const char* classes, bool validate = true);
  bool SetClasses(int id, const std::vector<std::string>& classes, bool validate = true);
  bool SetClasses(int id, const std::set<std::string>& classes, bool validate = true);
  bool AddClasses(int id, const char* classes);
  bool AddClasses(int id, const std::vector<std::string>& classes, bool validate = true);
  bool AddClasses(int id, const std::set<std::string>& classes, bool validate = true);
  bool RemoveClasses(int id, const char* classes);
  bool RemoveClasses(int id, const std::vector<std::string>& classes);
  bool RemoveClasses(int id, const std::set<std::string>& classes);
  ///@}

  ///@{
  /**
   * Visit each node in the assembly for processing. The traversal order can be
   * specified using `traversal_order` which defaults to depth-first.
   */
  void Visit(vtkDataAssemblyVisitor* visitor,
    int traversal_order = vtkDataAssembly::TraversalOrder::DepthFirst) const
  {
    this->Visit(0, visitor, traversal_order);
  }
  void Visit(int id, vtkDataAssemblyVisitor* visitor,
    int traversal_order = vtkDataAssembly::TraversalOrder::DepthFirst) const;
  ///@}

  ///@{
  /**
   * Returns the dataset indices associated with the node.
   *
   * If `traverse_subtree` is true (default), recursively builds the dataset
   * indices list for the node and all its child nodes.
   * Note, a dataset index will only appear once in the output even if it is
   * encountered on multiple nodes in the subtree.
   *
   * When `traverse_subtree` is true, the traversal order can be specified
   * using `traversal_order`. Defaults to depth-first.
   *
   */
  std::vector<unsigned int> GetDataSetIndices(int id, bool traverse_subtree = true,
    int traversal_order = vtkDataAssembly::TraversalOrder::DepthFirst) const;
  std::vector<unsigned int> GetDataSetIndices(const std::vector<int>& ids,
    bool traverse_subtree = true,
    int traversal_order = vtkDataAssembly::TraversalOrder::DepthFirst) const;
  ///@}

  /**
   * Returns ids for nodes matching the path_queries. See Section
   * @ref DataAssemblyPathQueries for supported query expressions.
   *
   * Will return an empty vector is no nodes match the requested query.
   *
   * @returns node ids matching the query in traversal order chosen using
   *          `traversal_order`.
   */
  std::vector<int> SelectNodes(const std::vector<std::string>& path_queries,
    int traversal_order = vtkDataAssembly::TraversalOrder::DepthFirst) const;

  /**
   * Remap dataset indices. `mapping` is map where the key is the old index and
   * value is the new index. If `remove_unmapped` is true, then any dataset not
   * in the map will be removed.
   */
  bool RemapDataSetIndices(
    const std::map<unsigned int, unsigned int>& mapping, bool remove_unmapped);

  /**
   * Create a deep-copy of other by only passing the chosen branches. All other
   * branches of the tree will be pruned. Note this method never affects the
   * depth of the selected branches or dataset indices attached to any of the nodes
   * in pruned output.
   */
  void SubsetCopy(vtkDataAssembly* other, const std::vector<int>& selected_branches);

  /**
   * Deep copy the `other`.
   */
  void DeepCopy(vtkDataAssembly* other);

  /**
   * Validates a node name.
   */
  static bool IsNodeNameValid(const char* name);

  /**
   * Converts any string to a string that is a valid node name.
   * This is done by simply discarding any non-supported character.
   * Additionally, if the first character is not a "_" or an alphabet, then
   * the "_" is prepended.
   */
  static std::string MakeValidNodeName(const char* name);

  /**
   * Returns true for node names that are reserved.
   */
  static bool IsNodeNameReserved(const char* name);

  ///@{
  /**
   * Set/get user data on a node.
   *
   * You may specify a handler for how to:
   * + dispose of the data when RemoveNode is called.
   * + update the data when SetNodeName(), SetAttribute(), or class-modifying methods are called.
   * + clone the data when AddSubtree() is passed a tree whose nodes have your data.
   *
   * The purpose for these methods is to allow libcss to cache style data on nodes,
   * so the handler is not invoked for dataset-modification events.
   *
   * Calling SetNodeUserData with \a userData set to a null pointer will remove any existing data.
   *
   * Calling SetNodeUserData with \a handler set to null will use a default handler that
   * assumes the user data is managed externally (i.e., it simply always returns its input).
   */
  bool SetNodeUserData(int nodeId, void* userData, UserDataHandler handler = nullptr);
  void* GetNodeUserData(int nodeId);
  ///@}

protected:
  vtkDataAssembly();
  ~vtkDataAssembly() override;

  struct OffsetIdWalker;
  class vtkInternals;
  std::unique_ptr<vtkInternals> Internals;
private:
  vtkDataAssembly(const vtkDataAssembly&) = delete;
  void operator=(const vtkDataAssembly&) = delete;
};

#endif
