/*=========================================================================

  Program:   Visualization Toolkit

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class    vtkOpenGLCellGridRenderRequest
 * @brief    State used by vtkOpenGLCellGridMapper2 during rendering.
 *
 * This is a vtkCellGridQuery subclass that mappers can use to draw cells
 * into a renderer using an actor and, subsequently, to release resources.
 *
 * Note that this request has two modes: it will either instruct
 * responders to draw cells (IsReleasingResources == false) or instruct
 * responders to release OpenGL objects for a particular window
 * (when IsReleasingResources == true).
 * Responders must call GetIsReleasingResources() and only perform
 * one task or the other, depending on the returned value.
 */

#ifndef vtkOpenGLCellGridRenderRequest_h
#define vtkOpenGLCellGridRenderRequest_h
#include "vtkRenderingOpenGL2Module.h" // For export macro
#include "vtkCellGridQuery.h"
#include "vtkStringToken.h"

#include <unordered_map> // for this->State
#include <memory> // for unique_ptr

class vtkActor;
class vtkOpenGLCellGridMapper2;
class vtkRenderer;
class vtkWindow;

class VTKRENDERINGOPENGL2_EXPORT vtkOpenGLCellGridRenderRequest : public vtkCellGridQuery
{
public:
  /// An empty base class that responders should inherit to store state using GetState<Subclass>().
  class StateBase
  {
  public:
    virtual ~StateBase() { }
  };

  vtkTypeMacro(vtkOpenGLCellGridRenderRequest, vtkCellGridQuery);
  static vtkOpenGLCellGridRenderRequest* New();
  virtual void PrintSelf(std::ostream& os, vtkIndent indent);

  /// Set/get the mapper which owns this request (so responders can inspect its configuration).
  virtual void SetMapper(vtkOpenGLCellGridMapper2* mapper);
  vtkGetObjectMacro(Mapper, vtkOpenGLCellGridMapper2);

  /// Set/get the actor which responders should use to draw cells.
  virtual void SetActor(vtkActor* actor);
  vtkGetObjectMacro(Actor, vtkActor);

  /// Set/get the renderer responders should use to draw cells.
  virtual void SetRenderer(vtkRenderer* renderer);
  vtkGetObjectMacro(Renderer, vtkRenderer);

  /// Set/get a window (used when IsReleasingResources is true).
  virtual void SetWindow(vtkWindow* window);
  vtkGetObjectMacro(Window, vtkWindow);

  /// This is invoked before processing any cell types during a render.
  void Initialize() override;
  /// This is invoked after processing all cell types during a render.
  void Finalize() override;

  /**
   * Set/get whether the request should render (false) or release resources (true).
   * The latter should be performed as a separate query after rendering.
   *
   * Note that after a successful call to Query with IsReleasingResources set to
   * true, the Finalize() method will reset IsReleasingResources to false, which
   * results in the request being marked modified.
   */
  vtkGetMacro(IsReleasingResources, bool);
  vtkSetMacro(IsReleasingResources, bool);

  /**
   * Return a state object of the given type.
   * This method is intended for responders to store data with the request.
   */
  template<typename StateType>
  StateType* GetState(vtkStringToken cellType)
  {
    auto it = this->State.find(cellType);
    if (it == this->State.end())
    {
      it = this->State.insert(std::make_pair(cellType, std::unique_ptr<StateBase>(new StateType))).first;
    }
    return static_cast<StateType*>(it->second.get());
  }

protected:
  vtkOpenGLCellGridRenderRequest();
  ~vtkOpenGLCellGridRenderRequest() override;

  vtkOpenGLCellGridMapper2* Mapper{nullptr};
  vtkActor* Actor{nullptr};
  vtkRenderer* Renderer{nullptr};
  vtkWindow* Window{nullptr};
  bool IsReleasingResources{false};
  std::unordered_map<vtkStringToken, std::unique_ptr<StateBase>> State;

private:
  vtkOpenGLCellGridRenderRequest(const vtkOpenGLCellGridRenderRequest&) = delete;
  void operator = (const vtkOpenGLCellGridRenderRequest&) = delete;
};

#endif // vtkOpenGLCellGridRenderRequest_h
