/*=========================================================================

  Program:   Visualization Toolkit

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkOpenGLCellGridRenderRequest.h"

#include "vtkObjectFactory.h"
#include "vtkDGHexahedronOpenGLRenderer.h"
#include "vtkActor.h"
#include "vtkOpenGLCellGridMapper2.h"
#include "vtkRenderer.h"
#include "vtkWindow.h"
#include "vtkCellMetadata.h"
#include "vtkCellGridResponders.h"
#include "vtkDGHex.h"

vtkStandardNewMacro(vtkOpenGLCellGridRenderRequest);

vtkCxxSetObjectMacro(vtkOpenGLCellGridRenderRequest, Mapper, vtkOpenGLCellGridMapper2);
vtkCxxSetObjectMacro(vtkOpenGLCellGridRenderRequest, Actor, vtkActor);
vtkCxxSetObjectMacro(vtkOpenGLCellGridRenderRequest, Renderer, vtkRenderer);
vtkCxxSetObjectMacro(vtkOpenGLCellGridRenderRequest, Window, vtkWindow);

vtkOpenGLCellGridRenderRequest::vtkOpenGLCellGridRenderRequest()
{
  // Plugins are expected to register responders, but for the base functionality provided
  // by VTK itself, we use this object to register responders at construction.
  // Since the vtkOpenGLCellGridMapper2 owns an instance of this request, the registration
  // is guaranteed to occur in time for the first render of cell types supported by VTK.
  static bool initialized = false;
  if (!initialized)
  {
    vtkNew<vtkDGHexahedronOpenGLRenderer> dgHexResponder;
    vtkCellMetadata::GetResponders()->RegisterQueryResponder<vtkDGHex, vtkOpenGLCellGridRenderRequest>(
      dgHexResponder.GetPointer());
    initialized = true;
  }
}

vtkOpenGLCellGridRenderRequest::~vtkOpenGLCellGridRenderRequest()
{
  this->State.clear();
  this->SetActor(nullptr);
  this->SetMapper(nullptr);
  this->SetRenderer(nullptr);
  this->SetWindow(nullptr);
}

void vtkOpenGLCellGridRenderRequest::PrintSelf(std::ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
  os << indent << "Actor: " << this->Actor << "\n";
  os << indent << "Renderer: " << this->Renderer << "\n";
  os << indent << "Mapper: " << this->Mapper << "\n";
  os << indent << "Window: " << this->Window << "\n";
  os << indent << "IsReleasingResources: " << (this->IsReleasingResources ? "True" : "False") << "\n";
  os << indent << "State: (" << this->State.size() << " entries)\n";
  vtkIndent i2 = indent.GetNextIndent();
  for (const auto& entry : this->State)
  {
    os << i2 << entry.first.Data() << ": " << entry.second.get() << "\n";
  }
}

void vtkOpenGLCellGridRenderRequest::Initialize()
{
  std::cerr << "Start render\n";
}

void vtkOpenGLCellGridRenderRequest::Finalize()
{
  std::cerr << "End render\n";

  // Always reset the request after releasing resources (but
  // never assume we're going to release resources after a
  // render pass).
  if (this->IsReleasingResources)
  {
    this->SetIsReleasingResources(false);
  }
}
