/*=========================================================================

  Program:   Visualization Toolkit

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#include "vtkOpenGLCellGridMapper2.h"

#include "vtkOpenGLCellGridRenderRequest.h"
#include "vtkActor.h"
#include "vtkCellAttribute.h"
#include "vtkCellGrid.h"
#include "vtkCellGridMapper.h"
#include "vtkColorTransferFunction.h"
#include "vtkImageData.h"
#include "vtkInformation.h"
#include "vtkLogger.h"
#include "vtkNew.h"
#include "vtkObjectFactory.h"
#include "vtkOpenGLActor.h"
#include "vtkOpenGLError.h"
#include "vtkOpenGLRenderWindow.h"
#include "vtkOpenGLRenderer.h"
#include "vtkOpenGLResourceFreeCallback.h"
#include "vtkRenderer.h"
#include "vtkSetGet.h"
#include "vtkSmartPointer.h"
#include "vtkStringToken.h"

#include <algorithm>
#include <iostream>
#include <map>
#include <numeric>
#include <ostream>
#include <sstream>
#include <string>
#include <vector>

using namespace vtk::literals;

class vtkOpenGLCellGridMapper2::vtkInternals
{
public:
  explicit vtkInternals(vtkOpenGLCellGridMapper2* Mapper);
  ~vtkInternals();
  vtkInternals(const vtkInternals&) = delete;
  void operator=(const vtkInternals&) = delete;

  std::unique_ptr<vtkGenericOpenGLResourceFreeCallback> ResourceCallback;
  vtkOpenGLCellGridMapper2* OglCellGridMapper = nullptr;
  vtkNew<vtkOpenGLCellGridRenderRequest> RenderQuery;
};

vtkOpenGLCellGridMapper2::vtkInternals::vtkInternals(vtkOpenGLCellGridMapper2* mapper)
  : OglCellGridMapper(mapper)
{
  this->RenderQuery->SetMapper(mapper);
  using CallBackT = vtkOpenGLResourceFreeCallback<vtkOpenGLCellGridMapper2>;
  this->ResourceCallback = std::unique_ptr<CallBackT>(
    new CallBackT(this->OglCellGridMapper, &vtkOpenGLCellGridMapper2::ReleaseGraphicsResources));
}

vtkOpenGLCellGridMapper2::vtkInternals::~vtkInternals()
{
  this->ResourceCallback->Release();
}

vtkStandardNewMacro(vtkOpenGLCellGridMapper2);

vtkOpenGLCellGridMapper2::vtkOpenGLCellGridMapper2()
  : Internal(new vtkInternals(this))
{
#ifndef NDEBUG
  this->DebugOn();
#endif
}

void vtkOpenGLCellGridMapper2::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

void vtkOpenGLCellGridMapper2::ReleaseGraphicsResources(vtkWindow* window)
{
  if (!this->Internal->ResourceCallback->IsReleasing())
  {
    this->Internal->ResourceCallback->Release();
    return;
  }

  auto* cellGrid = this->GetInput();
  if (cellGrid)
  {
    this->Internal->RenderQuery->SetIsReleasingResources(true);
    this->Internal->RenderQuery->SetWindow(window);
    cellGrid->Query(this->Internal->RenderQuery);
  }
  this->Modified();
}

void vtkOpenGLCellGridMapper2::ShallowCopy(vtkAbstractMapper*)
{
}

void vtkOpenGLCellGridMapper2::Render(vtkRenderer* ren, vtkActor* act)
{
  vtkDebugMacro(<< __func__);
  if (ren->GetRenderWindow()->CheckAbortStatus())
  {
    return;
  }
  auto* cellGrid = this->GetInput();
  if (cellGrid)
  {
    // TODO: Here, we hardwire a colormap for the vtkCellAttribute used for coloring.
    //       In practice, we need another vtkCellQuery for choosing the colormap to
    //       be consistent across all cell types (i.e., to get the field range correct).
    auto* colorAttribute = cellGrid->GetCellAttributeByName(this->GetArrayName());
    if (colorAttribute)
    {
      auto cmap = colorAttribute->GetColormap();
      if (!cmap)
      {
        // Create a cool-to-warm (blue to red) diverging colormap by default:
        vtkNew<vtkColorTransferFunction> ctf;
        ctf->SetVectorModeToMagnitude();
        ctf->SetColorSpaceToDiverging();
        ctf->AddRGBPoint(0.0, 59. / 255., 76. / 255., 192. / 255.);
        ctf->AddRGBPoint(0.5, 221. / 255., 221. / 255., 221. / 255.);
        ctf->AddRGBPoint(1.0, 180. / 255., 4. / 255., 38. / 255.);
        ctf->Build();
        colorAttribute->SetColormap(ctf);
        cmap = ctf;
      }
      // Now, if there is no colormap texture, make one from the colormap
      if (!this->LookupTable || this->LookupTable->GetMTime() < cmap->GetMTime())
      {
        this->SetLookupTable(cmap);
      }
      if (!this->ColorTextureMap || this->ColorTextureMap->GetMTime() < this->LookupTable->GetMTime())
      {
        this->CreateColormapTexture(); // populate this->ColorTexture from this->LookupTable
      }
      // The RenderQuery responders can now call this->GetColorTextureMap()
      // and use it for color lookup.
    }

    // Render the cells using our render-query.
    this->Internal->RenderQuery->SetRenderer(ren);
    this->Internal->RenderQuery->SetActor(act);
    cellGrid->Query(this->Internal->RenderQuery);
  }
}
