/*=========================================================================

  Program:   Visualization Toolkit

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#include "vtkOpenGLCellGridMapper.h"
#include "vtkAbstractArray.h"
#include "vtkActor.h"
#include "vtkArrayDispatch.h"
#include "vtkCellGrid.h"
#include "vtkCellArray.h"
#include "vtkCellGridFS.h"
#include "vtkCellGridGS.h"
#include "vtkCellGridMapper.h"
#include "vtkCellGridVS.h"
#include "vtkCellType.h"
#include "vtkCommand.h"
#include "vtkDataArray.h"
#include "vtkDataArrayMeta.h"
#include "vtkDataArrayRange.h"
#include "vtkDataObject.h"
#include "vtkDataSet.h"
#include "vtkFieldData.h"
#include "vtkGenericCell.h"
#include "vtkHexahedron.h"
#include "vtkImageData.h"
#include "vtkInformation.h"
#include "vtkLightingMapPass.h"
#include "vtkLogger.h"
#include "vtkMatrix3x3.h"
#include "vtkMatrix4x4.h"
#include "vtkNew.h"
#include "vtkObjectFactory.h"
#include "vtkOpenGLActor.h"
#include "vtkOpenGLBufferObject.h"
#include "vtkOpenGLCamera.h"
#include "vtkOpenGLError.h"
#include "vtkOpenGLFramebufferObject.h"
#include "vtkOpenGLHelper.h"
#include "vtkOpenGLIndexBufferObject.h"
#include "vtkOpenGLRenderPass.h"
#include "vtkOpenGLRenderWindow.h"
#include "vtkOpenGLRenderer.h"
#include "vtkOpenGLResourceFreeCallback.h"
#include "vtkOpenGLShaderCache.h"
#include "vtkOpenGLState.h"
#include "vtkOpenGLTexture.h"
#include "vtkOpenGLUniforms.h"
#include "vtkOpenGLVertexArrayObject.h"
#include "vtkOpenGLVertexBufferObject.h"
#include "vtkOpenGLVertexBufferObjectGroup.h"
#include "vtkPBRFunctions.h"
#include "vtkPBRIrradianceTexture.h"
#include "vtkPBRLUTTexture.h"
#include "vtkPBRPrefilterTexture.h"
#include "vtkPointData.h"
#include "vtkPolyData.h"
#include "vtkProperty.h"
#include "vtkQuad.h"
#include "vtkRenderer.h"
#include "vtkSetGet.h"
#include "vtkShader.h"
#include "vtkShaderProgram.h"
#include "vtkShaderProperty.h"
#include "vtkSmartPointer.h"
#include "vtkStringToken.h"
#include "vtkTextureObject.h"
#include "vtkTriangle.h"
#include "vtkType.h"
#include "vtkTypeFloat32Array.h"
#include "vtkTypeInt32Array.h"
#include "vtkTypeUInt32Array.h"
#include "vtkUnsignedCharArray.h"
#include "vtkUnstructuredGrid.h"
#include <algorithm>
#include <iostream>
#include <map>
#include <numeric>
#include <ostream>
#include <sstream>
#include <string>
#include <vector>

using namespace vtk::literals;

namespace
{
// values we use to determine if we need to rebuild shaders
// stored in a map keyed on the vtkOpenGLHelper, so one
// typically entry per type of primitive we render which
// matches the shader programs we use
class PrimitiveInformation
{
public:
  int LastLightComplexity = -1;
  int LastLightCount;
  vtkTimeStamp LightComplexityChanged;

  // Caches the vtkOpenGLRenderPass::RenderPasses() information.
  // Note: Do not dereference the pointers held by this object. There is no
  // guarantee that they are still valid!
  vtkNew<vtkInformation> LastRenderPassInfo;
};

class UploadableTexBuffer
{
public:
  vtkNew<vtkTextureObject> Texture;
  vtkNew<vtkOpenGLBufferObject> Buffer;
};

using TexInfo = std::pair<vtkTexture*, std::string>;
}

class vtkOpenGLCellGridMapper::vtkInternals
{
public:
  explicit vtkInternals(vtkOpenGLCellGridMapper* Mapper);
  ~vtkInternals();
  vtkInternals(const vtkInternals&) = delete;
  void operator=(const vtkInternals&) = delete;

  std::unique_ptr<vtkGenericOpenGLResourceFreeCallback> ResourceCallback;
  vtkOpenGLCellGridMapper* OglCellGridMapper = nullptr;
  vtkCellGrid* CurrentInput = nullptr;
  int MaxNumSides = 0;
  bool DrawingVertices = false;
  bool HaveCellScalars = false;
  bool PointPicking = false;

  enum Primitives
  {
    PrimitiveStart = 0,
    PrimitivePoint = 0,
    PrimtiiveLine,
    PrimitiveTri,
    PrimitiveVert,
    PrimitiveEnd
  };

  vtkNew<vtkMatrix3x3> TempMatrix3;
  vtkNew<vtkMatrix4x4> TempMatrix4;

  vtkNew<vtkTypeFloat32Array> InputPoints;
  vtkNew<vtkTypeFloat32Array> InputFieldCoefficients;
  vtkNew<vtkTypeFloat32Array> InputCustomTCoords;
  vtkUnsignedCharArray* InputCellScalarColors = nullptr;  // for coloring arrays on CellData
  vtkUnsignedCharArray* InputPointScalarColors = nullptr; // for coloring arrays on PointData

  vtkNew<vtkTypeInt32Array> InputCells, InputCellOffsets, InputSides;
  vtkUnsignedCharArray* InputCellTypes = nullptr;

  vtkOpenGLHelper Primitives[PrimitiveEnd], *LastBoundBO = nullptr;
  std::map<const vtkOpenGLHelper*, PrimitiveInformation> PrimitiveInfo;

  std::vector<float> CellParametrics;
  std::vector<int> FaceConnectivity, FaceOffsets, EdgeConnectivity, EdgeOffsets;

  UploadableTexBuffer CellConnectivityTB;
  UploadableTexBuffer CellOffsetsTB;
  UploadableTexBuffer FaceConnectivityTB;
  UploadableTexBuffer FaceOffsetsTB;
  UploadableTexBuffer EdgeConnectivityTB;
  UploadableTexBuffer EdgeOffsetsTB;
  UploadableTexBuffer CellParametricsTB;
  UploadableTexBuffer SidesTB;
  UploadableTexBuffer PointCoordinatesTB;
  UploadableTexBuffer FieldCoefficientsTB;
  UploadableTexBuffer CustomTCoordsTB;
  UploadableTexBuffer ScalarColorsTB;

  vtkNew<vtkOpenGLTexture> ColorTextureGL;

  std::vector<::TexInfo> GetTextures(vtkActor* actor);
  unsigned int GetNumberOfTextures(vtkActor* actor);
  bool HaveTextures(vtkActor* actor) { return (this->GetNumberOfTextures(actor) > 0); }
  bool HaveTCoords(vtkDataObject* dobj);

  void PopulateInputs(vtkActor*);
  void RenderStart(vtkRenderer*, vtkActor*);
  void RenderDraw(vtkRenderer*, vtkActor*);
  void RenderFinish(vtkRenderer*, vtkActor*);

  void UpdateBufferObjects(vtkRenderer*, vtkActor*);
  void BuildVBO(vtkRenderer*);
  void BuildTBO(vtkRenderer*, vtkActor*);
  void BuildIBO();
  void UpdateShaders(vtkOpenGLHelper& cellBO, vtkRenderer*, vtkActor*);
  void BuildShaders(std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer*, vtkActor*);
  bool GetNeedToRebuildShaders(vtkOpenGLHelper& cellBO, vtkRenderer*, vtkActor*);

  void GetShaderTemplate(std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer*, vtkActor*);

  void SetCameraShaderParameters(vtkOpenGLHelper& cellBO, vtkRenderer*, vtkActor*);
  void SetMapperShaderParameters(vtkOpenGLHelper& cellBO, vtkRenderer*, vtkActor*);
  void SetPropertyShaderParameters(vtkOpenGLHelper& cellBO, vtkRenderer*, vtkActor*);
  void SetLightingShaderParameters(vtkOpenGLHelper& cellBO, vtkRenderer*, vtkActor*);

  // For now, all these return false.
  bool DrawingEdges(vtkRenderer* ren, vtkActor* act) { (void)ren; (void)act; return false; }
  bool DrawingSpheres(vtkOpenGLHelper& cellBO, vtkActor* act) { (void)cellBO; (void)act; return false; }
  bool DrawingTubes(vtkOpenGLHelper& cellBO, vtkActor* act) { (void)cellBO; (void)act; return false; }
  bool DrawingTubesOrSpheres(vtkOpenGLHelper& cellBO, vtkActor* actor) { (void)cellBO; (void)actor; return false; }

  void ReplaceShaderValues(std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer*, vtkActor*);
  void ReplaceShaderRenderPass(
    std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer* ren, vtkActor* act, bool prePass);
  void ReplaceShaderCustomUniforms(std::map<vtkShader::Type, vtkShader*> shaders, vtkActor* act);
  void ReplaceShaderColor(
    std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer* ren, vtkActor* act);
  void ReplaceShaderEdges(
    std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer* ren, vtkActor* act);
  void ReplaceShaderLight(
    std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer* ren, vtkActor* act);
  void ReplaceShaderTCoord(
    std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer* ren, vtkActor* act);
  void ReplaceShaderPicking(
    std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer* ren, vtkActor* act);
  void ReplaceShaderPrimID(
    std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer* ren, vtkActor* act);
  void ReplaceShaderNormal(
    std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer* ren, vtkActor* act);
  void ReplaceShaderClip(
    std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer* ren, vtkActor* act);
  void ReplaceShaderPositionVC(
    std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer* ren, vtkActor* act);
  void ReplaceShaderCoincidentOffset(
    std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer* ren, vtkActor* act);
  void ReplaceShaderDepth(
    std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer* ren, vtkActor* act);
};

//------------------------------------------------------------------------------
vtkOpenGLCellGridMapper::vtkInternals::vtkInternals(vtkOpenGLCellGridMapper* mapper)
  : OglCellGridMapper(mapper)
{
  using CallBackT = vtkOpenGLResourceFreeCallback<vtkOpenGLCellGridMapper>;
  this->ResourceCallback = std::unique_ptr<CallBackT>(
    new CallBackT(this->OglCellGridMapper, &vtkOpenGLCellGridMapper::ReleaseGraphicsResources));
}

//------------------------------------------------------------------------------
vtkOpenGLCellGridMapper::vtkInternals::~vtkInternals()
{
  this->ResourceCallback->Release();
}

//------------------------------------------------------------------------------
std::vector<TexInfo> vtkOpenGLCellGridMapper::vtkInternals::GetTextures(vtkActor* actor)
{
  std::vector<TexInfo> res;

  if (this->OglCellGridMapper->ColorTextureMap)
  {
    res.emplace_back(this->ColorTextureGL, "colorTexture");
  }
  if (actor->GetTexture())
  {
    res.emplace_back(actor->GetTexture(), "actorTexture");
  }
  auto textures = actor->GetProperty()->GetAllTextures();
  for (const auto& texInfo : textures)
  {
    res.emplace_back(texInfo.second, texInfo.first);
  }
  return res;
}

//------------------------------------------------------------------------------
unsigned int vtkOpenGLCellGridMapper::vtkInternals::GetNumberOfTextures(vtkActor* actor)
{
  unsigned int res = 0;
  if (this->OglCellGridMapper->ColorTextureMap)
  {
    res++;
  }
  if (actor->GetTexture())
  {
    res++;
  }
  res += actor->GetProperty()->GetNumberOfTextures();
  return res;
}

//------------------------------------------------------------------------------
bool vtkOpenGLCellGridMapper::vtkInternals::HaveTCoords(vtkDataObject* dobj)
{
  (void)dobj;
#if 0
  xxx
  // TODO: Avoid casting to vtkDataSet
  if (auto dset = vtkDataSet::SafeDownCast(dobj))
  {
    return this->OglCellGridMapper->ColorCoordinates || dset->GetPointData()->GetTCoords();
  }
#endif
  return false;
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::PopulateInputs(vtkActor* actor)
{
  (void)actor;
  this->CurrentInput = this->OglCellGridMapper->GetInput();

  /*
  if (this->CurrentInput)
  {
    this->
  }
  */

  this->InputCells->Reset();
  this->InputCellOffsets->Reset();
  this->InputCellTypes = nullptr;
  this->InputPoints->Reset();
  this->InputFieldCoefficients->Reset();
  this->InputCustomTCoords->Reset();
  this->InputPointScalarColors = nullptr;
  this->InputCellScalarColors = nullptr;
  this->InputSides->Reset();
  this->FaceConnectivity.clear();
  this->FaceOffsets.clear();
  this->EdgeConnectivity.clear();
  this->EdgeOffsets.clear();

  if (this->CurrentInput == nullptr)
  {
    vtkErrorWithObjectMacro(this->OglCellGridMapper, << "No input!");
    return;
  }

  // update our input data.
  this->OglCellGridMapper->InvokeEvent(vtkCommand::StartEvent, nullptr);
  if (!this->OglCellGridMapper->Static)
  {
    this->OglCellGridMapper->GetInputAlgorithm()->Update();
  }
  this->OglCellGridMapper->InvokeEvent(vtkCommand::EndEvent, nullptr);

  if (auto cgrid = vtkCellGrid::SafeDownCast(this->CurrentInput))
  {
    vtkDebugWithObjectMacro(this->OglCellGridMapper, << "Detected vtkCellGrid input");
    vtkDebugWithObjectMacro(this->OglCellGridMapper, << "Loading points");

    // have points and cells
    this->InputPoints->ShallowCopy(cgrid->GetAttributes("coordinates"_token)->GetArray("dg points"));
#ifndef NDEBUG
    {
      std::stringstream sst;
      this->InputPoints->Print(sst);
      vtkDebugWithObjectMacro(this->OglCellGridMapper, << sst.str());
    }
#endif

    vtkDebugWithObjectMacro(this->OglCellGridMapper, << "Loading cells");
    this->InputCells->ShallowCopy(cgrid->GetAttributes("dg:hex8")->GetArray("conn"));
    // this->InputCellOffsets->ShallowCopy(cgrid->GetCells()->GetOffsetsArray());
#ifndef NDEBUG
    {
      std::stringstream sst;
      sst << "Connectivity"
          << "\n";
      this->InputCells->Print(sst);
      sst << "Offsets"
          << "\n";
      // this->InputCellOffsets->Print(sst);
      vtkDebugWithObjectMacro(this->OglCellGridMapper, << sst.str());
    }
#endif

    // get sides.
    vtkDebugWithObjectMacro(this->OglCellGridMapper, << "Loading sides");
    vtkIdTypeArray* sides = vtkIdTypeArray::FastDownCast(cgrid->GetAttributes("dg:hex8")->GetArray("sides"));
    const auto numCells = cgrid->GetNumberOfCells();
    const auto numSides = sides->GetNumberOfTuples();
    unsigned char cellType = VTK_HEXAHEDRON; // cgrid->GetDistinctCellTypesArray()->GetValue(0);
    vtkNew<vtkGenericCell> genCell;
    genCell->SetCellType(cellType);
    this->MaxNumSides =
      genCell->GetCellDimension() == 3 ? genCell->GetNumberOfFaces() : genCell->GetNumberOfEdges();
    // this is a flat 1D array - 0: skip side, 1: render side.
    this->InputSides->SetNumberOfComponents(1);
    this->InputSides->SetNumberOfValues(numCells * this->MaxNumSides);
    this->InputSides->FillValue(0);
    for (vtkIdType i = 0; i < numSides; ++i)
    {
      const auto& cellId = sides->GetTypedComponent(i, 0);
      const auto& sideId = sides->GetTypedComponent(i, 1);
      vtkIdType loc = cellId * this->MaxNumSides + sideId;
      vtkDebugWithObjectMacro(this->OglCellGridMapper, << i << " " << cellId << " " << sideId);
      this->InputSides->SetValue(loc, 1);
    }
#ifndef NDEBUG
    {
      std::stringstream sst;
      this->InputSides->Print(sst);
      sst << "Side ID | present(1)/absent(0)"
          << "\n";
      for (int i = 0; i < this->InputSides->GetNumberOfValues(); ++i)
      {
        sst << i << " " << this->InputSides->GetValue(i) << "\n";
      }
      vtkDebugWithObjectMacro(this->OglCellGridMapper, << sst.str());
    }
#endif

    // get cell types
    vtkDebugWithObjectMacro(this->OglCellGridMapper, << "Loading cellTypes");
    // this->InputCellTypes = cgrid->GetCellTypesArray();
#ifndef NDEBUG
    {
      std::stringstream sst;
      // this->InputCellTypes->Print(sst);
      vtkDebugWithObjectMacro(this->OglCellGridMapper, << sst.str());
    }
#endif
    this->FaceConnectivity.clear();
    this->FaceOffsets.clear();
    this->CellParametrics.clear();
    switch (cellType)
    {
      case VTK_HEXAHEDRON:
      {
        for (int faceId = 0; faceId < 6; ++faceId)
        {
          this->FaceOffsets.push_back(this->FaceConnectivity.size());
          const vtkIdType* facePts = vtkHexahedron::GetFaceArray(faceId);
          for (int i = 0; i < 4; ++i)
          {
            this->FaceConnectivity.push_back(facePts[i]);
          }
        }
        double* pcoords = genCell->GetParametricCoords();
        this->CellParametrics.assign(pcoords, pcoords + 24);
        // need to shift center of element to (0,0,0)
        std::transform(this->CellParametrics.cbegin(), this->CellParametrics.cend(),
          this->CellParametrics.begin(),
          [](const double& val) -> double { return 2 * (val - 0.5); });
        break;
      }
      default:
        break;
    }

    // get field coefficients
    vtkDebugWithObjectMacro(this->OglCellGridMapper, << "Loading field coefficients");
    // int cellFlag = 0;
    vtkDataArray* scalars = nullptr; /* this->OglCellGridMapper->GetScalars(
      ugrid, VTK_SCALAR_MODE_USE_CELL_DATA, 0, 0, nullptr, cellFlag); */
    if (scalars != nullptr)
    {
      this->InputFieldCoefficients->ShallowCopy(scalars);
    }
#ifndef NDEBUG
    {
      std::stringstream sst;
      this->InputFieldCoefficients->Print(sst);
      sst << "ID | Value"
          << "\n";
      for (int i = 0; i < this->InputFieldCoefficients->GetNumberOfValues(); ++i)
      {
        sst << i << " " << this->InputFieldCoefficients->GetValue(i) << "\n";
      }
      vtkDebugWithObjectMacro(this->OglCellGridMapper, << sst.str());
    }
#endif
    // Let vtkMapper finish the scalar -> lookup table process.
    this->OglCellGridMapper->MapScalars(1.0);
    cout << this->OglCellGridMapper->ColorTextureMap << endl;

    this->HaveCellScalars = false;
    this->InputPointScalarColors = this->OglCellGridMapper->Colors;
    if (this->OglCellGridMapper->ScalarVisibility)
    {
      // We must figure out how the scalars should be mapped to the polydata.
      int mode = this->OglCellGridMapper->ScalarMode;
      // auto fdofRelation = this->OglCellGridMapper->FieldToDOFRelation;
      // using DofType = vtkCellGridMapper::FieldDegreeOfFreedomType;

      if ((mode == VTK_SCALAR_MODE_USE_CELL_DATA || mode == VTK_SCALAR_MODE_USE_CELL_FIELD_DATA ||
            mode == VTK_SCALAR_MODE_USE_FIELD_DATA /*|| !ugrid->GetPointData()->GetScalars() */) &&
        mode != VTK_SCALAR_MODE_USE_POINT_FIELD_DATA && this->OglCellGridMapper->Colors /* &&
        fdofRelation != DofType::VertexDiscontinuous && fdofRelation != DofType::Edge &&
        fdofRelation != DofType::Face */ && this->OglCellGridMapper->Colors->GetNumberOfTuples() > 0)
      {
        this->HaveCellScalars = true;
        // reset point scalar colors. coloring is done with a cell scalar array.
        this->InputCellScalarColors = this->OglCellGridMapper->Colors;
        this->InputPointScalarColors = nullptr;
      }
    }

    // Set the texture if we are going to use texture for coloring with point attribute.
    if (this->HaveTCoords(this->CurrentInput))
    {
      this->InputCustomTCoords->SetNumberOfTuples(0);
      if (this->OglCellGridMapper->InterpolateScalarsBeforeMapping &&
        this->OglCellGridMapper->ColorCoordinates)
      {
        this->InputCustomTCoords->ShallowCopy(this->OglCellGridMapper->ColorCoordinates);
      }
      else
      {
        auto tcoords = nullptr; // cgrid->GetPointData()->GetTCoords();
        if (tcoords != nullptr)
        {
          this->InputCustomTCoords->ShallowCopy(tcoords);
        }
      }
    }
  }
  // else if (auto cellgrid = vtkCellGrid::SafeDownCast(this->CurrentInput))
  // {
  // }
  else
  {
    vtkDebugWithObjectMacro(this->OglCellGridMapper, << "Detected vtkDataObject input");
    // TODO: just a vtkDataObject
    return;
  }

  // no points on the input.
  if (this->InputPoints == nullptr || this->InputPoints->GetNumberOfTuples() == 0)
  {
    vtkDebugWithObjectMacro(this->OglCellGridMapper, << "No input points");
    return;
  }
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::RenderStart(vtkRenderer* ren, vtkActor* act)
{
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  auto oglRenWin = vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow());
  this->ResourceCallback->RegisterGraphicsResources(oglRenWin);

  this->UpdateBufferObjects(ren, act);

  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Activating CellConnectivityTexture");
  this->CellConnectivityTB.Texture->Activate();

  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Activating CellOffsetsTexture");
  this->CellOffsetsTB.Texture->Activate();

  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Activating FaceConnectivityTexture");
  this->FaceConnectivityTB.Texture->Activate();

  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Activating FaceOffsetsTexture");
  this->FaceOffsetsTB.Texture->Activate();

  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Activating CellParametricsTexture");
  this->CellParametricsTB.Texture->Activate();

  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Activating SidesTexture");
  this->SidesTB.Texture->Activate();

  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Activating PointCoordinatesTexture");
  this->PointCoordinatesTB.Texture->Activate();

  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Activating FieldCoefficientsTexture");
  this->FieldCoefficientsTB.Texture->Activate();

  // If we are coloring by texture, then load the texture map.
  // Use Map as indicator, because texture hangs around.
  if (this->OglCellGridMapper->ColorTextureMap)
  {
    this->ColorTextureGL->Load(ren);
  }

  // vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
  //                                                  << "Loading CustomTCoordsTexture");
  // this->CustomTCoordsTB.Texture->Activate();

  // vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
  //                                                  << "Loading ScalarColorsTexture");
  // this->ScalarColorsTB.Texture->Activate();
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::UpdateBufferObjects(vtkRenderer* ren, vtkActor* actor)
{
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  this->BuildIBO();
  this->BuildVBO(ren);
  this->BuildTBO(ren, actor);
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::BuildIBO()
{
  const auto numCells = this->InputCellTypes->GetNumberOfValues();
  std::vector<unsigned int> cellIds(numCells);
  std::iota(cellIds.begin(), cellIds.end(), 0);
  for (int i = PrimitiveStart; i < PrimitiveEnd; ++i)
  {
    i = PrimitiveTri;
    this->Primitives[i].IBO->IndexCount = numCells;
    this->Primitives[i].IBO->Upload(cellIds, vtkOpenGLBufferObject::ElementArrayBuffer);
    break;
  }
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Will draw " << numCells << " cells");
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::BuildTBO(vtkRenderer* ren, vtkActor*)
{
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  auto oglRenWin = vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow());

  // load cell connectivity arrays into a texture buffer.
  // load cell offset arrays into a texture buffer.
  // load sides array into a texture buffer.
  // load cell types array into a texture buffer.
  // load position coordinates into a texture buffer.
  // load scalar field data into a texture buffer if scalar mapping is enabled.
  // load primitive to vtk cell map into a texture buffer.

  // opengl textures are limited to 32 bit floats. possible loss of precision for narrowing casts.
  // use ssbo instead? requires opengl 4
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Uploading CellConnectivityTB.Buffer");
  this->CellConnectivityTB.Buffer->SetType(vtkOpenGLBufferObject::TextureBuffer);
  this->CellConnectivityTB.Texture->SetContext(oglRenWin);
  this->CellConnectivityTB.Buffer->Upload(this->InputCells->GetPointer(0),
    this->InputCells->GetNumberOfValues(), vtkOpenGLBufferObject::TextureBuffer);
  // set int support
  this->CellConnectivityTB.Texture->SetRequireTextureInteger(true);
  this->CellConnectivityTB.Texture->GetInternalFormat(VTK_INT, 1, true);
  this->CellConnectivityTB.Texture->CreateTextureBuffer(this->InputCells->GetNumberOfValues(), 1,
    this->InputCells->GetDataType(), this->CellConnectivityTB.Buffer);
  vtkOpenGLStaticCheckErrorMacro("failed to upload texture");

  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Uploading CellOffsetsTB.Buffer");
  this->CellOffsetsTB.Buffer->SetType(vtkOpenGLBufferObject::TextureBuffer);
  this->CellOffsetsTB.Texture->SetContext(oglRenWin);
  this->CellOffsetsTB.Buffer->Upload(this->InputCellOffsets->GetPointer(0),
    this->InputCellOffsets->GetNumberOfValues(), vtkOpenGLBufferObject::TextureBuffer);
  // set int support
  this->CellOffsetsTB.Texture->SetRequireTextureInteger(true);
  this->CellOffsetsTB.Texture->GetInternalFormat(VTK_INT, 1, true);
  this->CellOffsetsTB.Texture->CreateTextureBuffer(this->InputCellOffsets->GetNumberOfValues(), 1,
    this->InputCellOffsets->GetDataType(), this->CellOffsetsTB.Buffer);
  vtkOpenGLStaticCheckErrorMacro("failed to upload texture");

  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Uploading FaceConnectivityTB.Buffer");
  this->FaceConnectivityTB.Buffer->SetType(vtkOpenGLBufferObject::TextureBuffer);
  this->FaceConnectivityTB.Texture->SetContext(oglRenWin);
  this->FaceConnectivityTB.Buffer->Upload(
    this->FaceConnectivity, vtkOpenGLBufferObject::TextureBuffer);
  // set int support
  this->FaceConnectivityTB.Texture->SetRequireTextureInteger(true);
  this->FaceConnectivityTB.Texture->GetInternalFormat(VTK_INT, 1, true);
  this->FaceConnectivityTB.Texture->CreateTextureBuffer(
    this->FaceConnectivity.size(), 1, VTK_INT, this->FaceConnectivityTB.Buffer);
  vtkOpenGLStaticCheckErrorMacro("failed to upload texture");

  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Uploading FaceOffsetsTB.Buffer");
  this->FaceOffsetsTB.Buffer->SetType(vtkOpenGLBufferObject::TextureBuffer);
  this->FaceOffsetsTB.Texture->SetContext(oglRenWin);
  this->FaceOffsetsTB.Buffer->Upload(this->FaceOffsets, vtkOpenGLBufferObject::TextureBuffer);
  // set int support
  this->FaceOffsetsTB.Texture->SetRequireTextureInteger(true);
  this->FaceOffsetsTB.Texture->GetInternalFormat(VTK_INT, 1, true);
  this->FaceOffsetsTB.Texture->CreateTextureBuffer(
    this->FaceOffsets.size(), 1, VTK_INT, this->FaceOffsetsTB.Buffer);
  vtkOpenGLStaticCheckErrorMacro("failed to upload texture");

  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Uploading CellParametricsTB.Buffer");
  this->CellParametricsTB.Buffer->SetType(vtkOpenGLBufferObject::TextureBuffer);
  this->CellParametricsTB.Texture->SetContext(oglRenWin);
  this->CellParametricsTB.Buffer->Upload(
    this->CellParametrics, vtkOpenGLBufferObject::TextureBuffer);
  this->CellParametricsTB.Texture->CreateTextureBuffer(
    8, 3, VTK_FLOAT, this->CellParametricsTB.Buffer);
  vtkOpenGLStaticCheckErrorMacro("failed to upload texture");

  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Uploading SidesTB.Buffer");
  this->SidesTB.Buffer->SetType(vtkOpenGLBufferObject::TextureBuffer);
  this->SidesTB.Texture->SetContext(oglRenWin);
  this->SidesTB.Buffer->Upload(this->InputSides->GetPointer(0),
    this->InputSides->GetNumberOfValues(), vtkOpenGLBufferObject::TextureBuffer);
  this->SidesTB.Texture->SetRequireTextureInteger(true);
  this->SidesTB.Texture->GetInternalFormat(VTK_INT, 1, true);
  this->SidesTB.Texture->CreateTextureBuffer(this->InputSides->GetNumberOfValues(), 1,
    this->InputSides->GetDataType(), this->SidesTB.Buffer);
  vtkOpenGLStaticCheckErrorMacro("failed to upload texture");

  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Uploading point coordinates");
  this->PointCoordinatesTB.Buffer->SetType(vtkOpenGLBufferObject::TextureBuffer);
  this->PointCoordinatesTB.Texture->SetContext(oglRenWin);
  this->PointCoordinatesTB.Buffer->Upload(this->InputPoints->GetPointer(0),
    this->InputPoints->GetNumberOfValues(), vtkOpenGLBufferObject::TextureBuffer);
  assert(this->InputPoints->GetNumberOfComponents() == 3);
  this->PointCoordinatesTB.Texture->CreateTextureBuffer(this->InputPoints->GetNumberOfTuples(), 3,
    this->InputPoints->GetDataType(), this->PointCoordinatesTB.Buffer);
  vtkOpenGLStaticCheckErrorMacro("failed to upload texture");

  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Uploading field coefficients");
  this->FieldCoefficientsTB.Buffer->SetType(vtkOpenGLBufferObject::TextureBuffer);
  this->FieldCoefficientsTB.Texture->SetContext(oglRenWin);
  this->FieldCoefficientsTB.Buffer->Upload(this->InputFieldCoefficients->GetPointer(0),
    this->InputFieldCoefficients->GetNumberOfValues(), vtkOpenGLBufferObject::TextureBuffer);
  this->FieldCoefficientsTB.Texture->CreateTextureBuffer(
    this->InputFieldCoefficients->GetNumberOfValues(), 1,
    this->InputFieldCoefficients->GetDataType(), this->FieldCoefficientsTB.Buffer);
  vtkOpenGLStaticCheckErrorMacro("failed to upload texture");

  // If we are coloring by texture, then load the texture map.
  if (this->OglCellGridMapper->ColorTextureMap)
  {
    this->ColorTextureGL->RepeatOff();
    this->ColorTextureGL->SetInputData(this->OglCellGridMapper->ColorTextureMap);
  }

  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
                                                   << "Uploading CustomTCoords");
  this->CustomTCoordsTB.Buffer->SetType(vtkOpenGLBufferObject::TextureBuffer);
  this->CustomTCoordsTB.Texture->SetContext(oglRenWin);
  this->CustomTCoordsTB.Buffer->Upload(this->InputCustomTCoords->GetPointer(0),
    this->InputCustomTCoords->GetNumberOfValues(), vtkOpenGLBufferObject::TextureBuffer);
  this->CustomTCoordsTB.Texture->CreateTextureBuffer(this->InputCustomTCoords->GetNumberOfValues(),
    this->InputCustomTCoords->GetNumberOfComponents(), this->InputCustomTCoords->GetDataType(),
    this->CustomTCoordsTB.Buffer);
  vtkOpenGLStaticCheckErrorMacro("failed to upload texture");

  // vtkUnsignedCharArray* scalarColorArr =
  //   this->HaveCellScalars ? this->InputCellScalarColors : this->InputPointScalarColors;
  // if (scalarColorArr)
  // {
  //   vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__ << " - "
  //                                                    << "Uploading CellScalarColors");
  //   this->ScalarColorsTB.Buffer->SetType(vtkOpenGLBufferObject::TextureBuffer);
  //   this->ScalarColorsTB.Texture->SetContext(oglRenWin);
  //   this->ScalarColorsTB.Buffer->Upload(scalarColorArr->GetPointer(0),
  //     scalarColorArr->GetNumberOfValues(), vtkOpenGLBufferObject::TextureBuffer);
  //   assert(scalarColorArr->GetNumberOfComponents() == 4);
  //   this->ScalarColorsTB.Texture->CreateTextureBuffer(scalarColorArr->GetNumberOfTuples(), 4,
  //     scalarColorArr->GetDataType(), this->ScalarColorsTB.Buffer);
  //   vtkOpenGLStaticCheckErrorMacro("failed to upload texture");
  // }
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::BuildVBO(vtkRenderer*)
{
  // we don't yet use vbos. modern opengl still requires a vao to exist.
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::RenderDraw(vtkRenderer* ren, vtkActor* act)
{
  // update shaders(primitives)
  // bind primitive.IBO
  // draw range elements using primitve.IBO
  // unbind primitive.IBO
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  auto oglRenWin = vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow());
  vtkOpenGLState* ostate = oglRenWin->GetState();

#ifndef GL_ES_VERSION_3_0
  // when using IBL, we need seamless cubemaps to avoid artifacts
  if (ren->GetUseImageBasedLighting() && ren->GetEnvironmentTexture())
  {
    ostate->vtkglEnable(GL_TEXTURE_CUBE_MAP_SEAMLESS);
  }
#endif

  vtkHardwareSelector* selector = ren->GetSelector();
  int representation = act->GetProperty()->GetRepresentation();
  bool draw_surface_with_edges =
    (act->GetProperty()->GetEdgeVisibility() && representation == VTK_SURFACE) && !selector;
  GLenum mode = GL_POINTS; // always pass GL_POINTS.

  for (int i = PrimitiveStart; i < (draw_surface_with_edges ? PrimitiveEnd : PrimitiveTri + 1); i++)
  {
    i = PrimitiveTri;
    this->UpdateShaders(this->Primitives[i], ren, act);
    this->Primitives[i].IBO->Bind();
    glDrawRangeElements(mode, 0, static_cast<GLuint>(this->Primitives[i].IBO->IndexCount - 1),
      static_cast<GLsizei>(this->Primitives[i].IBO->IndexCount), GL_UNSIGNED_INT, nullptr);
    vtkOpenGLStaticCheckErrorMacro("Failed after glDrawRangeElements");
    this->Primitives[i].IBO->Release();
    break;
  }
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::RenderFinish(vtkRenderer* ren, vtkActor*)
{
  // release last bound BO
  // deactivate textures.
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  if (this->LastBoundBO != nullptr)
  {
    this->LastBoundBO->VAO->Release();
  }
  this->CellConnectivityTB.Texture->Deactivate();
  this->CellOffsetsTB.Texture->Deactivate();
  this->FaceConnectivityTB.Texture->Deactivate();
  this->FaceOffsetsTB.Texture->Deactivate();
  this->CellParametricsTB.Texture->Deactivate();
  this->SidesTB.Texture->Deactivate();
  this->PointCoordinatesTB.Texture->Deactivate();
  this->FieldCoefficientsTB.Texture->Deactivate();
  if (this->OglCellGridMapper->ColorTextureMap)
  {
    this->ColorTextureGL->PostRender(ren);
  }
  // this->CustomTCoordsTB.Texture->Deactivate();
  // this->ScalarColorsTB.Texture->Deactivate();
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::UpdateShaders(
  vtkOpenGLHelper& cellBO, vtkRenderer* ren, vtkActor* actor)
{
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  auto oglRenWin = vtkOpenGLRenderWindow::SafeDownCast(ren->GetRenderWindow());
  cellBO.VAO->Bind();
  this->LastBoundBO = &cellBO;

  if (this->GetNeedToRebuildShaders(cellBO, ren, actor))
  {
    // build for the first time.
    std::map<vtkShader::Type, vtkShader*> shaders;

    vtkShader* vss = vtkShader::New();
    vss->SetType(vtkShader::Vertex);
    shaders[vtkShader::Vertex] = vss;

    vtkShader* fss = vtkShader::New();
    fss->SetType(vtkShader::Fragment);
    shaders[vtkShader::Fragment] = fss;

    vtkShader* gss = vtkShader::New();
    gss->SetType(vtkShader::Geometry);
    shaders[vtkShader::Geometry] = gss;

    this->BuildShaders(shaders, ren, actor);

    vtkDebugWithObjectMacro(this->OglCellGridMapper, "Readying shader program");
    vtkShaderProgram* program = oglRenWin->GetShaderCache()->ReadyShaderProgram(shaders);

    std::stringstream sst;
    sst << "VS: \n" << shaders[vtkShader::Vertex]->GetSource() << endl;
    sst << "GS: \n" << shaders[vtkShader::Geometry]->GetSource() << endl;
    sst << "FS: \n" << shaders[vtkShader::Fragment]->GetSource() << endl;
    vtkDebugWithObjectMacro(this->OglCellGridMapper, << sst.str());

    vss->Delete();
    fss->Delete();
    gss->Delete();

    cellBO.Program = program;
    cellBO.VAO->ReleaseGraphicsResources();

    cellBO.ShaderSourceTime.Modified();
  }
  else
  {
    oglRenWin->GetShaderCache()->ReadyShaderProgram(cellBO.Program);
    if (cellBO.Program->GetMTime() > cellBO.AttributeUpdateTime)
    {
      // reset VAO
      cellBO.VAO->ReleaseGraphicsResources();
    }
  }
  vtkOpenGLStaticCheckErrorMacro("failed after UpdateShader");
  if (cellBO.Program)
  {
    this->SetMapperShaderParameters(cellBO, ren, actor);
    vtkOpenGLStaticCheckErrorMacro("failed after SetMapperShaderParameters");
    this->SetPropertyShaderParameters(cellBO, ren, actor);
    vtkOpenGLStaticCheckErrorMacro("failed after SetPropertyShaderParameters");
    this->SetCameraShaderParameters(cellBO, ren, actor);
    vtkOpenGLStaticCheckErrorMacro("failed after SetCameraShaderParameters");
    this->SetLightingShaderParameters(cellBO, ren, actor);
    vtkOpenGLStaticCheckErrorMacro("failed after SetLightingShaderParameters");
    this->OglCellGridMapper->InvokeEvent(vtkCommand::UpdateShaderEvent, cellBO.Program);
  }
  vtkOpenGLStaticCheckErrorMacro("failed after UpdateShader");
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::BuildShaders(
  std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer* ren, vtkActor* actor)
{
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  this->GetShaderTemplate(shaders, ren, actor);
  this->ReplaceShaderValues(shaders, ren, actor);
}

//------------------------------------------------------------------------------
bool vtkOpenGLCellGridMapper::vtkInternals::GetNeedToRebuildShaders(
  vtkOpenGLHelper& cellBO, vtkRenderer* ren, vtkActor* actor)
{ // TODO: Implement GetNeedToRebuildShaders

  int lightComplexity = 0;
  int numberOfLights = 0;

  bool needLighting = false;
  if (actor->GetProperty()->GetRepresentation() == VTK_POINTS)
  {
    needLighting = (actor->GetProperty()->GetInterpolation() != VTK_FLAT);
  }
  else // wireframe or surface rep
  {
    bool isTris = cellBO.PrimitiveType == PrimitiveTri;
    needLighting = (isTris || (!isTris && actor->GetProperty()->GetInterpolation() != VTK_FLAT));
  }
  // we sphering or tubing? Yes I made sphere into a verb
  if (this->DrawingTubesOrSpheres(cellBO, actor))
  {
    needLighting = true;
  }
  // do we need lighting?
  if (actor->GetProperty()->GetLighting() && needLighting)
  {
    vtkOpenGLRenderer* oren = static_cast<vtkOpenGLRenderer*>(ren);
    lightComplexity = oren->GetLightingComplexity();
    numberOfLights = oren->GetLightingCount();
  }

  auto& primInfo = this->PrimitiveInfo[&cellBO];
  if (primInfo.LastLightComplexity != lightComplexity || primInfo.LastLightCount != numberOfLights)
  {
    primInfo.LightComplexityChanged.Modified();
    primInfo.LastLightComplexity = lightComplexity;
    primInfo.LastLightCount = numberOfLights;
  }

  return cellBO.Program == nullptr;
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::GetShaderTemplate(
  std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer*, vtkActor*)
{
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  shaders[vtkShader::Vertex]->SetType(vtkShader::Vertex);
  shaders[vtkShader::Fragment]->SetType(vtkShader::Fragment);
  shaders[vtkShader::Geometry]->SetType(vtkShader::Geometry);
  shaders[vtkShader::Vertex]->SetSource(vtkCellGridVS);
  shaders[vtkShader::Fragment]->SetSource(vtkCellGridFS);
  shaders[vtkShader::Geometry]->SetSource(vtkCellGridGS);
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::SetCameraShaderParameters(
  vtkOpenGLHelper& cellBO, vtkRenderer* ren, vtkActor* actor)
{
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  vtkShaderProgram* program = cellBO.Program;

  vtkOpenGLCamera* cam = (vtkOpenGLCamera*)(ren->GetActiveCamera());

  // [WMVD]C == {world, model, view, display} coordinates
  // E.g., WCDC == world to display coordinate transformation
  vtkMatrix4x4* wcdc;
  vtkMatrix4x4* wcvc;
  vtkMatrix3x3* norms;
  vtkMatrix4x4* vcdc;
  cam->GetKeyMatrices(ren, wcvc, norms, vcdc, wcdc);

  if (program->IsUniformUsed("ZCalcR"))
  {
    if (cam->GetParallelProjection())
    {
      program->SetUniformf("ZCalcS", vcdc->GetElement(2, 2));
    }
    else
    {
      program->SetUniformf("ZCalcS", -0.5 * vcdc->GetElement(2, 2) + 0.5);
    }
#if ENABLE_SUPPORT_SPHERE
    // if (this->DrawingSpheres(cellBO, actor))
    // {
    //   program->SetUniformf("ZCalcR",
    //     actor->GetProperty()->GetPointSize() / (ren->GetSize()[0] * vcdc->GetElement(0, 0)));
    // }
    // else
#endif
    {
      program->SetUniformf("ZCalcR",
        actor->GetProperty()->GetLineWidth() / (ren->GetSize()[0] * vcdc->GetElement(0, 0)));
    }
  }

#if ENABLE_HANDLE_COINCIDENT
  // handle coincident
  float factor = 0.0;
  float offset = 0.0;
  this->GetCoincidentParameters(ren, actor, factor, offset);
  if ((factor != 0.0 || offset != 0.0) && cellBO.Program->IsUniformUsed("cOffset") &&
    cellBO.Program->IsUniformUsed("cFactor"))
  {
    cellBO.Program->SetUniformf("cOffset", offset);
    cellBO.Program->SetUniformf("cFactor", factor);
  }
#endif

  vtkNew<vtkMatrix3x3> env;
  if (program->IsUniformUsed("envMatrix"))
  {
    double up[3];
    double right[3];
    double front[3];
    ren->GetEnvironmentUp(up);
    ren->GetEnvironmentRight(right);
    vtkMath::Cross(right, up, front);
    for (int i = 0; i < 3; i++)
    {
      env->SetElement(i, 0, right[i]);
      env->SetElement(i, 1, up[i]);
      env->SetElement(i, 2, front[i]);
    }
  }

  // If the VBO coordinates were shifted and scaled, apply the inverse transform
  // to the model->view matrix:
  // vtkOpenGLVertexBufferObject* vvbo = this->VBOs->GetVBO("vertexMC");
#if ENABLE_SHIFT_SCALE
  if (vvbo && vvbo->GetCoordShiftAndScaleEnabled())
  {
    if (!actor->GetIsIdentity())
    {
      vtkMatrix4x4* mcwc;
      vtkMatrix3x3* anorms;
      static_cast<vtkOpenGLActor*>(actor)->GetKeyMatrices(mcwc, anorms);
      vtkMatrix4x4::Multiply4x4(this->VBOShiftScale, mcwc, this->TempMatrix4);
      if (program->IsUniformUsed("MCWCMatrix"))
      {
        program->SetUniformMatrix("MCWCMatrix", this->TempMatrix4);
      }
      if (program->IsUniformUsed("MCWCNormalMatrix"))
      {
        program->SetUniformMatrix("MCWCNormalMatrix", anorms);
      }
      vtkMatrix4x4::Multiply4x4(this->TempMatrix4, wcdc, this->TempMatrix4);
      program->SetUniformMatrix("MCDCMatrix", this->TempMatrix4);
      if (program->IsUniformUsed("MCVCMatrix"))
      {
        vtkMatrix4x4::Multiply4x4(this->VBOShiftScale, mcwc, this->TempMatrix4);
        vtkMatrix4x4::Multiply4x4(this->TempMatrix4, wcvc, this->TempMatrix4);
        program->SetUniformMatrix("MCVCMatrix", this->TempMatrix4);
      }
      if (program->IsUniformUsed("normalMatrix"))
      {
        vtkMatrix3x3::Multiply3x3(anorms, norms, this->TempMatrix3);
        program->SetUniformMatrix("normalMatrix", this->TempMatrix3);
      }
    }
    else
    {
      vtkMatrix4x4::Multiply4x4(this->VBOShiftScale, wcdc, this->TempMatrix4);
      program->SetUniformMatrix("MCDCMatrix", this->TempMatrix4);
      if (program->IsUniformUsed("MCVCMatrix"))
      {
        vtkMatrix4x4::Multiply4x4(this->VBOShiftScale, wcvc, this->TempMatrix4);
        program->SetUniformMatrix("MCVCMatrix", this->TempMatrix4);
      }
      if (program->IsUniformUsed("normalMatrix"))
      {
        program->SetUniformMatrix("normalMatrix", norms);
      }
    }
  }
  else
#endif
  {
    if (!actor->GetIsIdentity())
    {
      vtkMatrix4x4* mcwc;
      vtkMatrix3x3* anorms;
      ((vtkOpenGLActor*)actor)->GetKeyMatrices(mcwc, anorms);
      if (program->IsUniformUsed("MCWCMatrix"))
      {
        program->SetUniformMatrix("MCWCMatrix", mcwc);
      }
      if (program->IsUniformUsed("MCWCNormalMatrix"))
      {
        program->SetUniformMatrix("MCWCNormalMatrix", anorms);
      }
      vtkMatrix4x4::Multiply4x4(mcwc, wcdc, this->TempMatrix4);
      program->SetUniformMatrix("MCDCMatrix", this->TempMatrix4);
      if (program->IsUniformUsed("MCVCMatrix"))
      {
        vtkMatrix4x4::Multiply4x4(mcwc, wcvc, this->TempMatrix4);
        program->SetUniformMatrix("MCVCMatrix", this->TempMatrix4);
      }
      if (program->IsUniformUsed("normalMatrix"))
      {
        vtkMatrix3x3::Multiply3x3(anorms, norms, this->TempMatrix3);
        program->SetUniformMatrix("normalMatrix", this->TempMatrix3);
      }
    }
    else
    {
      program->SetUniformMatrix("MCDCMatrix", wcdc);
      if (program->IsUniformUsed("MCVCMatrix"))
      {
        program->SetUniformMatrix("MCVCMatrix", wcvc);
      }
      if (program->IsUniformUsed("normalMatrix"))
      {
        program->SetUniformMatrix("normalMatrix", norms);
      }
    }
  }

  if (program->IsUniformUsed("envMatrix"))
  {
    vtkMatrix3x3::Invert(norms, this->TempMatrix3);
    vtkMatrix3x3::Multiply3x3(this->TempMatrix3, env, this->TempMatrix3);
    program->SetUniformMatrix("envMatrix", this->TempMatrix3);
  }

  if (program->IsUniformUsed("cameraParallel"))
  {
    program->SetUniformi("cameraParallel", cam->GetParallelProjection());
  }
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::SetMapperShaderParameters(
  vtkOpenGLHelper& cellBO, vtkRenderer*, vtkActor*)
{
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);

  // still gotta bind the VAO. else open gl will not render anything
  if (cellBO.IBO->IndexCount /*&&
    (this->VBOs->GetMTime() > cellBO.AttributeUpdateTime ||
      cellBO.ShaderSourceTime > cellBO.AttributeUpdateTime ||
      cellBO.VAO->GetMTime() > cellBO.AttributeUpdateTime)*/)
  {
    cellBO.VAO->Bind();
    cellBO.AttributeUpdateTime.Modified();
  }
  vtkOpenGLStaticCheckErrorMacro("failed after AddAllAttributesToVAO");

  int tunit = this->PointCoordinatesTB.Texture->GetTextureUnit();
  if (!cellBO.Program->SetUniformi("vertexPositions", tunit))
  {
    vtkWarningWithObjectMacro(this->OglCellGridMapper, << cellBO.Program->GetError());
  }

  tunit = this->FieldCoefficientsTB.Texture->GetTextureUnit();
  if (!cellBO.Program->SetUniformi("fieldCoefficients", tunit))
  {
    vtkWarningWithObjectMacro(this->OglCellGridMapper, << cellBO.Program->GetError());
  }

  tunit = this->CellConnectivityTB.Texture->GetTextureUnit();
  if (!cellBO.Program->SetUniformi("cellConnectivity", tunit))
  {
    vtkWarningWithObjectMacro(this->OglCellGridMapper, << cellBO.Program->GetError());
  }

  tunit = this->CellOffsetsTB.Texture->GetTextureUnit();
  if (!cellBO.Program->SetUniformi("cellOffsets", tunit))
  {
    vtkWarningWithObjectMacro(this->OglCellGridMapper, << cellBO.Program->GetError());
  }

  tunit = this->FaceConnectivityTB.Texture->GetTextureUnit();
  if (!cellBO.Program->SetUniformi("faceConnectivity", tunit))
  {
    vtkWarningWithObjectMacro(this->OglCellGridMapper, << cellBO.Program->GetError());
  }

  tunit = this->FaceOffsetsTB.Texture->GetTextureUnit();
  if (!cellBO.Program->SetUniformi("faceOffsets", tunit))
  {
    vtkWarningWithObjectMacro(this->OglCellGridMapper, << cellBO.Program->GetError());
  }

  tunit = this->CellParametricsTB.Texture->GetTextureUnit();
  if (!cellBO.Program->SetUniformi("cellParametrics", tunit))
  {
    vtkWarningWithObjectMacro(this->OglCellGridMapper, << cellBO.Program->GetError());
  }

  tunit = this->SidesTB.Texture->GetTextureUnit();
  if (!cellBO.Program->SetUniformi("sides", tunit))
  {
    vtkWarningWithObjectMacro(this->OglCellGridMapper, << cellBO.Program->GetError());
  }

  tunit = this->ColorTextureGL->GetTextureUnit();
  if (!cellBO.Program->SetUniformi("colorTexture", tunit))
  {
    vtkWarningWithObjectMacro(this->OglCellGridMapper, << cellBO.Program->GetError());
  }

  if (!cellBO.Program->SetUniformi("maxNumSides", this->MaxNumSides))
  {
    vtkWarningWithObjectMacro(this->OglCellGridMapper, << cellBO.Program->GetError());
  }
  if (!cellBO.Program->SetUniformi("cellType", this->InputCellTypes->GetValue(0)))
  {
    vtkWarningWithObjectMacro(this->OglCellGridMapper, << cellBO.Program->GetError());
  }
  if (!cellBO.Program->SetUniformi("visualizePCoord", this->OglCellGridMapper->VisualizePCoords))
  {
    vtkWarningWithObjectMacro(this->OglCellGridMapper, << cellBO.Program->GetError());
  }
  if (!cellBO.Program->SetUniformi(
        "visualizeBasisFunction", this->OglCellGridMapper->VisualizeBasisFunction))
  {
    vtkWarningWithObjectMacro(this->OglCellGridMapper, << cellBO.Program->GetError());
  }

  float range[2];
  auto minmax =
    std::minmax_element(this->InputFieldCoefficients->Begin(), this->InputFieldCoefficients->End());
  range[0] = *minmax.first;
  range[1] = *minmax.second;
  cellBO.Program->SetUniform2f("fieldRange", range);

  vtkOpenGLStaticCheckErrorMacro("failed after updating shader uniforms");
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::SetPropertyShaderParameters(
  vtkOpenGLHelper& cellBO, vtkRenderer*, vtkActor* actor)
{
  // This is a bit toned down from vtkOpenGLPolyDataMapepr::SetPropertyShaderParameters
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  vtkShaderProgram* program = cellBO.Program;

  vtkProperty* ppty = actor->GetProperty();

  // Query the property for some of the properties that can be applied.
  float opacity = ppty->GetOpacity();
  double* aColor = ppty->GetAmbientColor();
  double aIntensity = ppty->GetAmbient();

  double* dColor = ppty->GetDiffuseColor();
  double dIntensity = ppty->GetDiffuse();

  // double* sColor = ppty->GetSpecularColor();
  // double sIntensity = ppty->GetSpecular();
  // double specularPower = ppty->GetSpecularPower();

  // these are always set
  program->SetUniformf("opacityUniform", opacity);
  program->SetUniformf("ambientIntensity", aIntensity);
  program->SetUniformf("diffuseIntensity", dIntensity);
  program->SetUniform3f("ambientColorUniform", aColor);
  if (program->IsUniformUsed("diffuseColorUniform"))
  {
    program->SetUniform3f("diffuseColorUniform", dColor);
  }
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::SetLightingShaderParameters(
  vtkOpenGLHelper& cellBO, vtkRenderer* ren, vtkActor*)
{
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  vtkOpenGLRenderer* oglRen = vtkOpenGLRenderer::SafeDownCast(ren);
  if (oglRen)
  {
    vtkFloatArray* sh = oglRen->GetSphericalHarmonics();

    if (oglRen->GetUseSphericalHarmonics() && sh)
    {
      std::string uniforms[3] = { "shRed", "shGreen", "shBlue" };
      for (int i = 0; i < 3; i++)
      {
        float coeffs[9];
        sh->GetTypedTuple(i, coeffs);

        // predivide with pi for Lambertian diffuse
        coeffs[0] *= 0.282095f;
        coeffs[1] *= -0.488603f * (2.f / 3.f);
        coeffs[2] *= 0.488603f * (2.f / 3.f);
        coeffs[3] *= -0.488603f * (2.f / 3.f);
        coeffs[4] *= 1.092548f * 0.25f;
        coeffs[5] *= -1.092548f * 0.25f;
        coeffs[6] *= 0.315392f * 0.25f;
        coeffs[7] *= -1.092548f * 0.25f;
        coeffs[8] *= 0.546274f * 0.25f;

        cellBO.Program->SetUniform1fv(uniforms[i].c_str(), 9, coeffs);
      }
    }
    oglRen->UpdateLightingUniforms(cellBO.Program);
  }
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::ReplaceShaderValues(
  std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer* ren, vtkActor* actor)
{
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  this->ReplaceShaderRenderPass(shaders, ren, actor, true);
  // this->ReplaceShaderCustomUniforms(shaders, actor);
  this->ReplaceShaderColor(shaders, ren, actor);
  // this->ReplaceShaderEdges(shaders, ren, actor);
  this->ReplaceShaderNormal(shaders, ren, actor);
  this->ReplaceShaderLight(shaders, ren, actor);
  // this->ReplaceShaderTCoord(shaders, ren, actor);
  // this->ReplaceShaderPicking(shaders, ren, actor);
  // this->ReplaceShaderClip(shaders, ren, actor);
  this->ReplaceShaderPositionVC(shaders, ren, actor);
  // this->ReplaceShaderCoincidentOffset(shaders, ren, actor);
  // this->ReplaceShaderDepth(shaders, ren, actor);
  this->ReplaceShaderRenderPass(shaders, ren, actor, false);
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::ReplaceShaderRenderPass(
  std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer*, vtkActor* act, bool prePass)
{
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  std::string VSSource = shaders[vtkShader::Vertex]->GetSource();
  std::string GSSource = shaders[vtkShader::Geometry]->GetSource();
  std::string FSSource = shaders[vtkShader::Fragment]->GetSource();

  vtkInformation* info = act->GetPropertyKeys();
  if (info && info->Has(vtkOpenGLRenderPass::RenderPasses()))
  {
    int numRenderPasses = info->Length(vtkOpenGLRenderPass::RenderPasses());
    for (int i = 0; i < numRenderPasses; ++i)
    {
      vtkObjectBase* rpBase = info->Get(vtkOpenGLRenderPass::RenderPasses(), i);
      vtkOpenGLRenderPass* rp = static_cast<vtkOpenGLRenderPass*>(rpBase);
      if (prePass)
      {
        if (!rp->PreReplaceShaderValues(VSSource, GSSource, FSSource, this->OglCellGridMapper, act))
        {
          vtkErrorWithObjectMacro(this->OglCellGridMapper,
            "vtkOpenGLRenderPass::ReplaceShaderValues failed for " << rp->GetClassName());
        }
      }
      else
      {
        if (!rp->PostReplaceShaderValues(
              VSSource, GSSource, FSSource, this->OglCellGridMapper, act))
        {
          vtkErrorWithObjectMacro(this->OglCellGridMapper,
            "vtkOpenGLRenderPass::ReplaceShaderValues failed for " << rp->GetClassName());
        }
      }
    }
  }

  shaders[vtkShader::Vertex]->SetSource(VSSource);
  shaders[vtkShader::Geometry]->SetSource(GSSource);
  shaders[vtkShader::Fragment]->SetSource(FSSource);
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::ReplaceShaderCustomUniforms(
  std::map<vtkShader::Type, vtkShader*> shaders, vtkActor* actor)
{
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  vtkShaderProperty* sp = actor->GetShaderProperty();

  vtkShader* vertexShader = shaders[vtkShader::Vertex];
  vtkOpenGLUniforms* vu = static_cast<vtkOpenGLUniforms*>(sp->GetVertexCustomUniforms());
  vtkShaderProgram::Substitute(vertexShader, "//VTK::CustomUniforms::Dec", vu->GetDeclarations());

  vtkShader* fragmentShader = shaders[vtkShader::Fragment];
  vtkOpenGLUniforms* fu = static_cast<vtkOpenGLUniforms*>(sp->GetFragmentCustomUniforms());
  vtkShaderProgram::Substitute(fragmentShader, "//VTK::CustomUniforms::Dec", fu->GetDeclarations());

  vtkShader* geometryShader = shaders[vtkShader::Geometry];
  vtkOpenGLUniforms* gu = static_cast<vtkOpenGLUniforms*>(sp->GetGeometryCustomUniforms());
  vtkShaderProgram::Substitute(geometryShader, "//VTK::CustomUniforms::Dec", gu->GetDeclarations());
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::ReplaceShaderColor(
  std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer*, vtkActor* actor)
{
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  std::string GSSource = shaders[vtkShader::Geometry]->GetSource();
  std::string FSSource = shaders[vtkShader::Fragment]->GetSource();

  // these are always defined
  std::string colorDec = "uniform float ambientIntensity; // the material ambient\n"
                         "uniform float diffuseIntensity; // the material diffuse\n"
                         "uniform float opacityUniform; // the fragment opacity\n"
                         "uniform vec3 ambientColorUniform; // ambient color\n"
                         "uniform vec3 diffuseColorUniform; // diffuse color\n";

  std::string colorImpl;

  // specular lighting?
  if (this->PrimitiveInfo[this->LastBoundBO].LastLightComplexity)
  {
    colorDec += "uniform float specularIntensity; // the material specular intensity\n"
                "uniform vec3 specularColorUniform; // intensity weighted color\n"
                "uniform float specularPowerUniform;\n";
    colorImpl += "vec3 specularColor = specularIntensity * specularColorUniform;\n"
                 "  float specularPower = specularPowerUniform;\n";
  }

  // for point picking we render primitives as points
  // that means cell scalars will not have correct
  // primitiveIds to lookup into the texture map
  // so we must skip cell scalar coloring when point picking

  // handle color point attributes
  if (this->InputPointScalarColors && this->InputPointScalarColors->GetNumberOfComponents() != 0 &&
    !this->DrawingVertices)
  {
    vtkShaderProgram::Substitute(GSSource, "//VTK::Color::Dec",
      "out vec4 vertexColorGSOutput;\n"
      "uniform samplerBuffer pointScalarColorTexture;");
    vtkShaderProgram::Substitute(GSSource, "//VTK::Color::Impl",
      "vertexColorGSOutput = texelFetchBuffer(pointScalarColorTexture, ptId);");

    // set the color in FS appropriately.
    colorDec += "in vec4 vertexColorGSOutput;\n";
    colorImpl += "  vec3 ambientColor = ambientIntensity * vertexColorGSOutput.rgb;\n"
                 "  vec3 diffuseColor = diffuseIntensity * vertexColorGSOutput.rgb;\n"
                 "  float opacity = opacityUniform * vertexColorGSOutput.a;";
  }
  // handle point color texture map coloring
  else if (this->OglCellGridMapper->InterpolateScalarsBeforeMapping &&
    this->OglCellGridMapper->ColorCoordinates && !this->DrawingVertices)
  {
    vtkShaderProgram::Substitute(GSSource, "//VTK::Color::Dec",
      "out vec4 vertexColorGSOutput;\n"
      "uniform sampler2D colorTexture;");
    vtkShaderProgram::Substitute(GSSource, "//VTK::Color::Impl",
      "vertexColorGSOutput = texture(colorTexture, tcoordVCGSOutput.st;");
    colorDec += "in vec4 vertexColorGSOutput;\n";
    colorImpl += "  vec3 ambientColor = ambientIntensity * vertexColorGSOutput.rgb;\n"
                 "  vec3 diffuseColor = diffuseIntensity * vertexColorGSOutput.rgb;\n"
                 "  float opacity = opacityUniform * vertexColorGSOutput.a;";
  }
  // discontinuous fields?
  if (this->OglCellGridMapper->InterpolateScalarsBeforeMapping &&
    !this->OglCellGridMapper->ColorCoordinates && !this->DrawingVertices)
  {
    colorDec += "uniform sampler2D colorTexture;";
    colorImpl += "  vec4 texColor = texture(colorTexture, texCoord.st);\n"
                 "  vec3 ambientColor = ambientIntensity * texColor.rgb;\n"
                 "  vec3 diffuseColor = diffuseIntensity * texColor.rgb;\n"
                 "  float opacity = opacityUniform * texColor.a;";
  }
  // are we doing cell scalar coloring by texture?
  else if (this->HaveCellScalars && !this->DrawingVertices && !this->PointPicking)
  {
    vtkShaderProgram::Substitute(GSSource, "//VTK::Color::Dec",
      "out vec4 vertexColorGSOutput;\n"
      "uniform samplerBuffer cellScalarColorTexture;");
    vtkShaderProgram::Substitute(GSSource, "//VTK::Color::Impl",
      "vertexColorGSOutput = texelFetchBuffer(cellScalarColorTexture, cellId);");

    colorDec += "in vec4 vertexColorGSOutput;\n";
    colorImpl += "  vec3 ambientColor = ambientIntensity * vertexColorGSOutput.rgb;\n"
                 "  vec3 diffuseColor = diffuseIntensity * vertexColorGSOutput.rgb;\n"
                 "  float opacity = opacityUniform * vertexColorGSOutput.a;";
  }
  // just material but handle backfaceproperties
  else
  {
    colorImpl += "  vec3 ambientColor = ambientIntensity * ambientColorUniform;\n"
                 "  vec3 diffuseColor = diffuseIntensity * diffuseColorUniform;\n"
                 "  float opacity = opacityUniform;\n";

    if (actor->GetBackfaceProperty() && !this->DrawingVertices)
    {
      colorDec += "uniform float opacityUniformBF; // the fragment opacity\n"
                  "uniform float ambientIntensityBF; // the material ambient\n"
                  "uniform float diffuseIntensityBF; // the material diffuse\n"
                  "uniform vec3 ambientColorUniformBF; // ambient material color\n"
                  "uniform vec3 diffuseColorUniformBF; // diffuse material color\n";
      if (this->PrimitiveInfo[this->LastBoundBO].LastLightComplexity)
      {
        colorDec += "uniform float specularIntensityBF; // the material specular intensity\n"
                    "uniform vec3 specularColorUniformBF; // intensity weighted color\n"
                    "uniform float specularPowerUniformBF;\n";
        colorImpl += "  if (gl_FrontFacing == false) {\n"
                     "    ambientColor = ambientIntensityBF * ambientColorUniformBF;\n"
                     "    diffuseColor = diffuseIntensityBF * diffuseColorUniformBF;\n"
                     "    specularColor = specularIntensityBF * specularColorUniformBF;\n"
                     "    specularPower = specularPowerUniformBF;\n"
                     "    opacity = opacityUniformBF; }\n";
      }
      else
      {
        colorImpl += "  if (gl_FrontFacing == false) {\n"
                     "    ambientColor = ambientIntensityBF * ambientColorUniformBF;\n"
                     "    diffuseColor = diffuseIntensityBF * diffuseColorUniformBF;\n"
                     "    opacity = opacityUniformBF; }\n";
      }
    }
  }

  if (this->HaveCellScalars && !this->DrawingVertices)
  {
    colorDec += "uniform samplerBuffer textureC;\n";
  }

  vtkShaderProgram::Substitute(FSSource, "//VTK::Color::Dec", colorDec);
  vtkShaderProgram::Substitute(FSSource, "//VTK::Color::Impl", colorImpl);

  shaders[vtkShader::Geometry]->SetSource(GSSource);
  shaders[vtkShader::Fragment]->SetSource(FSSource);
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::ReplaceShaderEdges(
  std::map<vtkShader::Type, vtkShader*>, vtkRenderer*, vtkActor*)
{
  // TODO: Port from vtkOpenGLPolyDataMapper
  vtkWarningWithObjectMacro(this->OglCellGridMapper,
    << "vtkOpenGLCellGridMapper::vtkInternals::" << __func__ << "is not implemented");
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::ReplaceShaderNormal(
  std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer*, vtkActor*)
{
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);

  std::string GSSource = shaders[vtkShader::Geometry]->GetSource();
  std::string FSSource = shaders[vtkShader::Fragment]->GetSource();

  vtkShaderProgram::Substitute(GSSource, "//VTK::Normal::Dec",
    "out vec3 normalVCGSOutput;"
    "uniform mat3 normalMatrix;\n");
  vtkShaderProgram::Substitute(
    GSSource, "//VTK::Normal::Impl", "normalVCGSOutput = normalMatrix * vec3(n.x, n.y, n.z);");

  vtkShaderProgram::Substitute(FSSource, "//VTK::Normal::Dec", "in vec3 normalVCGSOutput;");
  vtkShaderProgram::Substitute(FSSource, "//VTK::Normal::Impl",
    "vec3 normalVCGSOutput = normalize(normalVCGSOutput);\n"
    "  if (gl_FrontFacing == false) { normalVCGSOutput = -normalVCGSOutput; }\n");
  shaders[vtkShader::Geometry]->SetSource(GSSource);
  shaders[vtkShader::Fragment]->SetSource(FSSource);
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::ReplaceShaderLight(
  std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer* ren, vtkActor* actor)
{
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  std::string FSSource = shaders[vtkShader::Fragment]->GetSource();
  std::ostringstream toString;
  auto oglRen = vtkOpenGLRenderer::SafeDownCast(ren);

  // check for normal rendering
  auto info = actor->GetPropertyKeys();
  if (info && info->Has(vtkLightingMapPass::RENDER_NORMALS()))
  {
    vtkShaderProgram::Substitute(FSSource, "//VTK::Light::Impl",
      "vec3 n = (normalVCGSOutput + 1.0f) * 0.5;\n"
      "  gl_FragData[0] = vec4(n.x, n.y, n.z, 1.0);");
    shaders[vtkShader::Fragment]->SetSource(FSSource);
    return;
  }

  // for luminance, we don't want diffuse, specular colors to show up.
  if (info && info->Has(vtkLightingMapPass::RENDER_LUMINANCE()))
  {
    vtkShaderProgram::Substitute(FSSource, "//VTK::Light::Impl",
      "diffuseColor = vec3(1.0f, 1.0f, 1.0f);\n"
      "  specularColor = vec3(1.0f, 1.0f, 1.0f);\n"
      "  //VTK::Light::Impl\n",
      false);
  }

  int lastLightComplexity = this->PrimitiveInfo[this->LastBoundBO].LastLightComplexity;
  int lastLightCount = this->PrimitiveInfo[this->LastBoundBO].LastLightCount;

  if (actor->GetProperty()->GetInterpolation() != VTK_PBR && lastLightCount == 0)
  {
    lastLightComplexity = 0;
  }

  // For now, this mapper prototype does not do image based lighting, does not consider
  // anisotropy property or clear-coating

  // get standard lighting declarations.
  vtkShaderProgram::Substitute(FSSource, "//VTK::Light::Dec", oglRen->GetLightingUniforms());
  switch (lastLightComplexity)
  {
    case 0: // no lighting
      vtkShaderProgram::Substitute(FSSource, "//VTK::Light::Impl",
        "gl_FragData[0] = vec4(ambientColor + diffuseColor, opacity);\n"
        "  //VTK::Light::Impl\n",
        false);
      break;
    case 1: // headlight
      if (actor->GetProperty()->GetInterpolation() == VTK_PBR)
      {
        vtkErrorWithObjectMacro(
          this->OglCellGridMapper, << "Headlights are not implemented for PBR interpolation");
        break;
      }
      else
      {
        toString << "float df = max(0.0f, normalVCGSOutput.z);\n"
                    "  float sf = pow(df, specularPower);\n"
                    "  vec3 diffuse = df * diffuseColor * lightColor0;\n"
                    "  vec3 specular = sf * specularColor * lightColor0;\n"
                    "  gl_FragData[0] = vec4(ambientColor + diffuse + specular, opacity);\n"
                    "  //VTK::Light::Impl\n";
      }
      vtkShaderProgram::Substitute(FSSource, "//VTK::Light::Impl", toString.str(), false);
      break;
    case 2: // light kit
      vtkErrorWithObjectMacro(this->OglCellGridMapper, << "Light kit is not implemented!");
      break;
    case 3: // positional
      vtkErrorWithObjectMacro(this->OglCellGridMapper, << "Position lighs are not implemented!");
      break;
  }
  shaders[vtkShader::Fragment]->SetSource(FSSource);
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::ReplaceShaderTCoord(
  std::map<vtkShader::Type, vtkShader*>, vtkRenderer*, vtkActor*)
{
  // TODO: Port from vtkOpenGLPolyDataMapper
  vtkWarningWithObjectMacro(this->OglCellGridMapper,
    << "vtkOpenGLCellGridMapper::vtkInternals::" << __func__ << " is not implemented");
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::vtkInternals::ReplaceShaderPositionVC(
  std::map<vtkShader::Type, vtkShader*> shaders, vtkRenderer*, vtkActor* actor)
{
  vtkDebugWithObjectMacro(this->OglCellGridMapper, << __func__);
  std::string VSSource = shaders[vtkShader::Vertex]->GetSource();
  std::string GSSource = shaders[vtkShader::Geometry]->GetSource();
  std::string FSSource = shaders[vtkShader::Fragment]->GetSource();

  vtkShaderProgram::Substitute(
    FSSource, "//VTK::Camera::Dec", "uniform int cameraParallel;\n", false);

  // do we need the vertex in the shader in View Coordinates
  if (this->PrimitiveInfo[this->LastBoundBO].LastLightComplexity > 0 ||
    this->DrawingTubes(*this->LastBoundBO, actor))
  {
    vtkShaderProgram::Substitute(GSSource, "//VTK::PositionVC::Dec", "out vec4 vertexVCGSOutput;");
    vtkShaderProgram::Substitute(GSSource, "//VTK::PositionVC::Impl",
      "vertexVCGSOutput = MCVCMatrix * vertexMC;\n"
      "        gl_Position = MCDCMatrix * vertexMC;\n");
    vtkShaderProgram::Substitute(GSSource, "//VTK::Camera::Dec",
      "uniform mat4 MCDCMatrix;\n"
      "uniform mat4 MCVCMatrix;");
    vtkShaderProgram::Substitute(GSSource, "//VTK::PositionVC::Dec",
      "in vec4 vertexVCGSOutput[];\n"
      "out vec4 vertexVCGSOutput;");
    vtkShaderProgram::Substitute(FSSource, "//VTK::PositionVC::Dec", "in vec4 vertexVCGSOutput;");
    vtkShaderProgram::Substitute(
      FSSource, "//VTK::PositionVC::Impl", "vec4 vertexVC = vertexVCGSOutput;");
  }
  else
  {
    vtkShaderProgram::Substitute(GSSource, "//VTK::Camera::Dec", "uniform mat4 MCDCMatrix;");
    vtkShaderProgram::Substitute(
      GSSource, "//VTK::PositionVC::Impl", "gl_Position = MCDCMatrix * vertexMC;\n");
  }
  shaders[vtkShader::Vertex]->SetSource(VSSource);
  shaders[vtkShader::Geometry]->SetSource(GSSource);
  shaders[vtkShader::Fragment]->SetSource(FSSource);
}

// If MSAA is enabled, don't write to gl_FragDepth unless we absolutely have
// to. See VTK issue 16899.
void vtkOpenGLCellGridMapper::vtkInternals::ReplaceShaderDepth(
  std::map<vtkShader::Type, vtkShader*>, vtkRenderer*, vtkActor*)
{
  // noop by default
}

//------------------------------------------------------------------------------
vtkStandardNewMacro(vtkOpenGLCellGridMapper);

//------------------------------------------------------------------------------
vtkOpenGLCellGridMapper::vtkOpenGLCellGridMapper()
  : Internal(new vtkInternals(this))
{
#ifndef NDEBUG
  this->DebugOn();
#endif
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::ReleaseGraphicsResources(vtkWindow* win)
{
  if (!this->Internal->ResourceCallback->IsReleasing())
  {
    this->Internal->ResourceCallback->Release();
    return;
  }

  for (int i = vtkInternals::PrimitiveStart; i < vtkInternals::PrimitiveEnd; i++)
  {
    this->Internal->Primitives[i].ReleaseGraphicsResources(win);
    // this->Internal->SelectionPrimitives[i].ReleaseGraphicsResources(win);
  }

  this->Internal->ColorTextureGL->ReleaseGraphicsResources(win);
  this->Internal->CellConnectivityTB.Texture->ReleaseGraphicsResources(win);
  this->Internal->CellOffsetsTB.Texture->ReleaseGraphicsResources(win);
  this->Internal->SidesTB.Texture->ReleaseGraphicsResources(win);
  this->Internal->PointCoordinatesTB.Texture->ReleaseGraphicsResources(win);
  this->Internal->FieldCoefficientsTB.Texture->ReleaseGraphicsResources(win);
  this->Modified();
}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::ShallowCopy(vtkAbstractMapper*) {}

//------------------------------------------------------------------------------
void vtkOpenGLCellGridMapper::Render(vtkRenderer* ren, vtkActor* act)
{
  vtkDebugMacro(<< __func__);
  if (ren->GetRenderWindow()->CheckAbortStatus())
  {
    return;
  }
  this->Internal->PopulateInputs(act);
  this->Internal->RenderStart(ren, act);
  this->Internal->RenderDraw(ren, act);
  this->Internal->RenderFinish(ren, act);
}
