//VTK::System::Dec

/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkCellGridFS.glsl

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// Template for the cellgrid mapper geometry shader

// look up cell points, types and vertex positions from texture buffers.
uniform isamplerBuffer cellConnectivity;
uniform isamplerBuffer cellOffsets;
uniform isamplerBuffer sides;
uniform isamplerBuffer faceConnectivity;
uniform isamplerBuffer faceOffsets;
uniform isamplerBuffer edgeConnectivity;
uniform isamplerBuffer edgeOffsets;
uniform samplerBuffer cellParametrics;
uniform samplerBuffer vertexPositions;
uniform int cellType;
uniform int maxNumSides;

// vtkActor may be given custom uniforms.
//VTK::CustomUniforms::Dec

// camera and actor matrix values
//VTK::Camera::Dec

// Position of vertex in view coordinates.
//VTK::PositionVC::Dec

// Color output per vertex in the rendered primitives.
//VTK::Color::Dec

// The normal of the output primitive in view coordinates.
//VTK::Normal::Dec

layout(points) in;
layout(triangle_strip, max_vertices = 36) out; // TODO: use string substitution for this

// Input from vertex shader
in int vtkCellIdVSOutput[]; // size 1

// send cell ID to fragment shader
flat out int vtkCellIdGSOutput;

// Custom output for fragment shader
out vec3 pCoordGSOutput;

//----------------------------------------------------------------
vec3 ComputeNormal(in vec3 p1, in vec3 p2, in vec3 p3)
{
  vec3 delta32 = p3 - p2;
  vec3 delta12 = p1 - p2;
  return cross(delta32, delta12);
}

//----------------------------------------------------------------
/**
 * Draws triangle strips for the sideId'th face of a linear hexahedron.
 * sideId - index of the face which will be rendered as two triangles - [0, maxNumSides]
 * cellId - index of the vtk cell whose faces we shall render - [0, numCells]
 */
void DrawTriStrips_Hex_C1(in int sideId, in int cellId)
{
  // split each face into two triangles.
  // 0----3   0          0---3
  // |    |   | \         \  |
  // |    | ->|  \    +    \ |
  // |    |   |   \         \|
  // 1----2   1----2         2
  ivec3 triPtIds[2];
  triPtIds[0] = ivec3(0, 1, 2);
  triPtIds[1] = ivec3(0, 2, 3);
  
  int draw = texelFetch(sides, cellId * maxNumSides + sideId).r;
  int ofst = texelFetch(cellOffsets, cellId).r;
  int faceOfst = texelFetch(faceOffsets, sideId).r;

  int cellLocalPtIds[4], cellGlobalPtIds[4];
  vec3 coords[4];
  for (int i = 0; i < 4; ++i)
  {
    cellLocalPtIds[i] = texelFetch(faceConnectivity, faceOfst + i).r;
    cellGlobalPtIds[i] = texelFetch(cellConnectivity, ofst + cellLocalPtIds[i]).r;
    coords[i] = texelFetch(vertexPositions, cellGlobalPtIds[i]).xyz;
  }

  vec3 n = ComputeNormal(coords[0], coords[1], coords[2]);
  if (length(n) == 0.0)
  {
    n = ComputeNormal(coords[1], coords[2], coords[3]);
  }
  if (length(n) == 0.0)
  {
    n.z = 1.0f;
  }

  //VTK::Normal::Impl

  if (draw == 1)
  {
    for (int triId = 0; triId < 2; ++triId)
    {
      for (int i = 0; i < 3; ++i)
      {
        int triPtId = triPtIds[triId][i];
        int localPtId = cellLocalPtIds[triPtId];
        int globalPtId = cellGlobalPtIds[triPtId];

        vec4 vertexMC = vec4(texelFetch(vertexPositions, globalPtId).xyz, 1.0f);

        //VTK::PositionVC::Impl
        
        //VTK::Color::Impl

        pCoordGSOutput = texelFetch(cellParametrics, localPtId).xyz;
        EmitVertex();
      }
      EndPrimitive();
    }
  }
}

//----------------------------------------------------------------
void main()
{
  vtkCellIdGSOutput = vtkCellIdVSOutput[0];

  // render each side.
  for (int sideId = 0; sideId < maxNumSides; ++sideId)
  {
    switch(cellType)
    {
      case 12: // VTK_HEXAHEDRON
        DrawTriStrips_Hex_C1(sideId, vtkCellIdGSOutput);
        break;
      default:
        break;
    }
  }
}