/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkCellGridMapper.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkCellAttribute.h"
#include "vtkCellGridMapper.h"
#include "vtkAbstractMapper.h"
#include "vtkCellGrid.h"
#include "vtkDoubleArray.h"
#include "vtkExecutive.h"
#include "vtkFloatArray.h"
#include "vtkImageData.h"
#include "vtkInformation.h"
#include "vtkObjectFactory.h"
#include "vtkPointData.h"
#include "vtkSmartPointer.h"
#include "vtkUnsignedCharArray.h"
#include "vtkLookupTable.h"
#include "vtkUnstructuredGrid.h"
#include "vtkVariant.h"
#include <iostream>

vtkObjectFactoryNewMacro(vtkCellGridMapper);

vtkCellGridMapper::vtkCellGridMapper() = default;

void vtkCellGridMapper::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
  os << indent << "VisualizePCoords: " << this->VisualizePCoords << "\n";
  os << indent << "VisualizeBasisFunction: " << this->VisualizeBasisFunction << "\n";
}

void vtkCellGridMapper::SetInputData(vtkCellGrid* in)
{
  this->SetInputDataInternal(0, in);
}

vtkCellGrid* vtkCellGridMapper::GetInput()
{
  return vtkCellGrid::SafeDownCast(this->GetExecutive()->GetInputData(0, 0));
}

// Get the bounds for the input of this mapper as
// (xMin, xMax, yMin, yMax, zMin, zMax)
double* vtkCellGridMapper::GetBounds()
{
  if (!this->GetNumberOfInputConnections(0))
  {
    vtkMath::UninitializeBounds(this->Bounds);
    return this->Bounds;
  }
  else
  {
    if (!this->Static)
    {
      vtkInformation* inInfo = this->GetInputInformation();
      if (inInfo)
      {
        this->GetInputAlgorithm()->UpdateInformation();
        this->GetInputAlgorithm()->Update();
      }
    }
    this->ComputeBounds();

    // if the bounds indicate NAN
    if (!vtkMath::AreBoundsInitialized(this->Bounds))
    {
      return nullptr;
    }
    return this->Bounds;
  }
}

void vtkCellGridMapper::ComputeBounds()
{
  vtkCellGrid* input = this->GetInput();
  if (input && input->GetNumberOfElements(vtkDataObject::CELL))
  {
    input->GetBounds(this->Bounds);
  }
  else
  {
    vtkMath::UninitializeBounds(this->Bounds);
  }
}

int vtkCellGridMapper::FillInputPortInformation(int vtkNotUsed(port), vtkInformation* info)
{
  info->Set(vtkAlgorithm::INPUT_REQUIRED_DATA_TYPE(), "vtkCellGrid");
  return 1;
}

void vtkCellGridMapper::Update(int port)
{
  if (this->Static)
  {
    return;
  }
  this->Superclass::Update(port);
}

void vtkCellGridMapper::Update()
{
  if (this->Static)
  {
    return;
  }
  this->Superclass::Update();
}

vtkTypeBool vtkCellGridMapper::Update(int port, vtkInformationVector* requests)
{
  if (this->Static)
  {
    return 1;
  }
  return this->Superclass::Update(port, requests);
}

vtkTypeBool vtkCellGridMapper::Update(vtkInformation* requests)
{
  if (this->Static)
  {
    return 1;
  }
  return this->Superclass::Update(requests);
}

vtkUnsignedCharArray* vtkCellGridMapper::MapScalars(double alpha)
{
  auto* input = this->GetInput();
  int cellFlag; // not used
  return this->MapScalars(input, alpha, cellFlag);
}

vtkUnsignedCharArray* vtkCellGridMapper::MapScalars(double alpha, int& cellFlag)
{
  auto* input = this->GetInput();
  return this->MapScalars(input, alpha, cellFlag);
}

vtkUnsignedCharArray* vtkCellGridMapper::MapScalars(vtkCellGrid* input, double alpha)
{
  int cellFlag = 0;
  return this->MapScalars(input, alpha, cellFlag);
}

vtkUnsignedCharArray* vtkCellGridMapper::MapScalars(vtkCellGrid* input, double alpha, int& cellFlag)
{
  auto* attribute = this->GetColorAttribute(input);
  // NB: We ignore the inherited this->ScalarMode since attributes are kept in
  //     a single container regardless of where values are provided (point/cell/edge/face).

  if (!this->ScalarVisibility || !attribute || !input)
  { // No scalar colors.
    if (this->ColorCoordinates)
    {
      this->ColorCoordinates->UnRegister(this);
      this->ColorCoordinates = nullptr;
    }
    if (this->ColorTextureMap)
    {
      this->ColorTextureMap->UnRegister(this);
      this->ColorTextureMap = nullptr;
    }
    if (this->Colors)
    {
      this->Colors->UnRegister(this);
      this->Colors = nullptr;
    }
    return nullptr;
  }

  // Get the lookup table.
  auto* lkup = attribute ? attribute->GetColormap() : nullptr;
  if (lkup)
  {
    this->SetLookupTable(lkup);
  }
  else
  {
    // make sure we have a lookup table
    if (!this->LookupTable)
    {
      this->CreateDefaultLookupTable();
    }
    this->LookupTable->Build();
  }
  if (!this->UseLookupTableScalarRange)
  {
    this->LookupTable->SetRange(this->ScalarRange);
  }

  // We have a lookup table.
  // At this point, we will always turn it into a texture map and
  // let the fragment (vertex?) shader look colors up as needed.
  this->CreateColormapTexture();
#if 0
  // TODO: texture coordinates should be computed in the fragment shader.
  if (this->CanUseTextureMapForColoring(input))
  {
    this->MapScalarsToTexture(scalars, alpha);

    // Get rid of old color coordinates.
    if (this->ColorCoordinates)
    {
      this->ColorCoordinates->UnRegister(this);
      this->ColorCoordinates = nullptr;
    }
    // Get rid of old colors
    if (this->Colors)
    {
      this->Colors->UnRegister(this);
      this->Colors = nullptr;
    }
  }
  else
  {
    // TODO: Populate these like vtkOpenGLPolyDataMapper expects them to be.
    if (this->ColorCoordinates)
    {
      this->ColorCoordinates->UnRegister(this);
      this->ColorCoordinates = nullptr;
    }
    if (this->ColorTextureMap)
    {
      this->ColorTextureMap->UnRegister(this);
      this->ColorTextureMap = nullptr;
    }
    // Get rid of old colors
    if (this->Colors)
    {
      this->Colors->UnRegister(this);
      this->Colors = nullptr;
    }
  }
#endif

  return nullptr;
}

int vtkCellGridMapper::CanUseTextureMapForColoring(vtkDataObject* input)
{
  if (!this->InterpolateScalarsBeforeMapping)
  {
    return 0; // user doesn't want us to use texture maps at all.
  }

#if 0
  // index color does not use textures
  if (this->LookupTable && this->LookupTable->GetIndexedLookup())
  {
    return 0;
  }
#endif

  auto* attribute = this->GetColorAttribute(vtkCellGrid::SafeDownCast(input));
  if (!attribute)
  {
    // no scalars to render.. don't care if texture is used or ignored.
    return 1;
  }

  int cellFlag = 0;
  if (attribute->GetAttributeType() == "discontinuous Lagrange polynomial order 1")
  {
    return 1;
  }

  if (attribute->GetNumberOfComponents() <= 4 &&
    ( this->ColorMode == VTK_COLOR_MODE_DEFAULT ||
      this->ColorMode == VTK_COLOR_MODE_DIRECT_SCALARS))
  {
    // Don't use texture; use direct coloring with RGBA components.
    return 0;
  }

  return 1;
}

void vtkCellGridMapper::CreateColormapTexture()
{
  if (!this->LookupTable)
  {
    if (this->ColorTextureMap)
    {
      this->ColorTextureMap->UnRegister(this);
      this->ColorTextureMap = nullptr;
    }
    return;
  }

  // Can we use the texture we already have?
  if (this->ColorTextureMap &&
    this->GetMTime() < this->ColorTextureMap->GetMTime() &&
    this->LookupTable->GetMTime() < this->ColorTextureMap->GetMTime())
  {
    return;
  }

  // Nope, allocate one if needed.
  if (!this->ColorTextureMap)
  {
    this->ColorTextureMap = vtkImageData::New();
  }

  double* range = this->LookupTable->GetRange();
  // Get the texture map from the lookup table.
  // Create a dummy ramp of scalars.
  // In the future, we could extend vtkScalarsToColors.
  vtkIdType numberOfColors = this->LookupTable->GetNumberOfAvailableColors();
  numberOfColors += 2;
  // number of available colors can return 2^24
  // which is an absurd size for a tmap in this case. So we
  // watch for cases like that and reduce it to a
  // more reasonable size
  if (numberOfColors > 65538) // 65536+2
  {
    numberOfColors = 8192;
  }
  double k = (range[1] - range[0]) / (numberOfColors - 2);
  vtkNew<vtkDoubleArray> tmp;
  tmp->SetNumberOfTuples(numberOfColors * 2);
  double* ptr = tmp->GetPointer(0);
  bool use_log_scale = false; // FIXME
  for (int i = 0; i < numberOfColors; ++i)
  {
    *ptr = range[0] + i * k - k / 2.0; // minus k / 2 to start at below range color
    if (use_log_scale)
    {
      *ptr = pow(10.0, *ptr);
    }
    ++ptr;
  }
  // Dimension on NaN.
  double nan = vtkMath::Nan();
  for (int i = 0; i < numberOfColors; ++i)
  {
    *ptr = nan;
    ++ptr;
  }
  this->ColorTextureMap->SetExtent(0, numberOfColors - 1, 0, 1, 0, 0);
  this->ColorTextureMap->GetPointData()->SetScalars(
    this->LookupTable->MapScalars(tmp, this->ColorMode, 0));
  // this->LookupTable->SetAlpha(orig_alpha);
  this->ColorTextureMap->GetPointData()->GetScalars()->Delete();
}

vtkCellAttribute* vtkCellGridMapper::GetColorAttribute(vtkCellGrid* input) const
{
  vtkCellAttribute* attribute = nullptr;
  if (input)
  {
    if (this->ArrayAccessMode == VTK_GET_ARRAY_BY_NAME)
    {
      attribute = input->GetCellAttributeByName(this->ArrayName);
    }
    else // if (this->ArrayAccessMode = VTK_GET_ARRAY_BY_ID)
    {
      attribute = input->GetCellAttributeById(this->ArrayId);
    }
  }
  return attribute;
}
