/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkCellGridExtractSurface.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/

#include "vtkCellGridExtractSurface.h"

#include "vtkObjectFactory.h"
#include "vtkDataSetAttributes.h"
#include "vtkCellGridSidesQuery.h"
#include "vtkIdTypeArray.h"

using namespace vtk::literals; // for ""_token

vtkStandardNewMacro(vtkCellGridExtractSurface);

void vtkCellGridExtractSurface::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

vtkStringToken vtkCellGridExtractSurface::GetSideAttribute()
{
  return vtkStringToken("Sides");
}

int vtkCellGridExtractSurface::RequestData(
  vtkInformation* request, vtkInformationVector** inInfo, vtkInformationVector* ouInfo)
{
  (void)request;

  auto* input = vtkCellGrid::GetData(inInfo[0]);
  auto* output = vtkCellGrid::GetData(ouInfo);
  if (!input)
  {
    vtkWarningMacro("Empty input.");
    return 1;
  }
  if (!output)
  {
    vtkErrorMacro("Empty output.");
    return 0;
  }
  output->ShallowCopy(input);
  if (!output->Query(this->Request))
  {
    vtkErrorMacro("Input failed to respond to query.");
    return 0;
  }
  const auto& sides = this->Request->GetSides();
  vtkIdType sideCount = 0;
  for (const auto& entry : sides)
  {
    sideCount += entry.second.size();
  }
  vtkNew<vtkIdTypeArray> sideArray;
  sideArray->SetName("conn");
  sideArray->SetNumberOfComponents(2); // tuples are (cell ID, side ID)
  sideArray->SetNumberOfTuples(sideCount);
  vtkIdType sideId = 0;
  std::array<vtkIdType, 2> sideTuple;
  for (const auto& entry : sides)
  {
    sideTuple[0] = entry.first;
    for (const auto& ss : entry.second)
    {
      sideTuple[1] = ss;
      sideArray->SetTypedTuple(sideId, sideTuple.data());
      ++sideId;
    }
  }
  auto* attr = output->GetAttributes(vtkCellGridExtractSurface::GetSideAttribute());
  attr->AddArray(sideArray);
  attr->SetScalars(sideArray);
  return 1;
}
