/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkDGSidesResponder.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkDGSidesResponder.h"

#include "vtkBoundingBox.h"
#include "vtkCellGrid.h"
#include "vtkCellGridBoundsQuery.h"
#include "vtkDGHex.h"
#include "vtkDataSetAttributes.h"
#include "vtkIdTypeArray.h"
#include "vtkTypeInt64Array.h"
#include "vtkObjectFactory.h"
#include "vtkStringToken.h"

#include <unordered_set>

using namespace vtk::literals;

vtkStandardNewMacro(vtkDGSidesResponder);

namespace
{

constexpr vtkIdType faces[6][4] = {
  { 0, 4, 7, 3 }, // 0
  { 1, 2, 6, 5 }, // 1
  { 0, 1, 5, 4 }, // 2
  { 3, 7, 6, 2 }, // 3
  { 0, 3, 2, 1 }, // 4
  { 4, 5, 6, 7 }, // 5
};

} // anonymous namespace

bool vtkDGSidesResponder::Query(
  vtkCellGridSidesQuery* query,
  vtkCellMetadata* cellType,
  vtkCellGridResponders* caches)
{
  (void)query;
  (void)cellType;
  (void)caches;

  auto* grid = cellType->GetCellGrid();
  std::string cellTypeName = cellType->GetClassName();
  vtkStringToken cellAttrName(cellTypeName.substr(3));
  auto* conn = vtkTypeInt64Array::SafeDownCast(
    grid->GetAttributes(cellAttrName)->GetArray("conn"));
  if (!conn)
  {
    return false;
  }
  std::unordered_set<std::int64_t> pointIDs;
  int nc = conn->GetNumberOfComponents();
  constexpr int nf = sizeof(faces) / sizeof(faces[0]);
  constexpr int nv = sizeof(faces[0]) / sizeof(faces[0][0]);
  std::vector<vtkTypeInt64> entry;
  entry.resize(nc);
  std::array<vtkIdType, 4> face;
  for (vtkIdType ii = 0; ii < conn->GetNumberOfTuples(); ++ii)
  {
    conn->GetTypedTuple(ii, &entry[0]);
    for (int jj = 0; jj < nf; ++jj)
    {
      for (int kk = 0; kk < nv; ++kk)
      {
        face[kk] = entry[faces[jj][kk]];
      }
      query->AddSide(ii, jj, face);
    }
  }
  return true;
}
