/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkCellMetadata.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class   vtkCellMetadata
 * @brief   Metadata for a particular type of cell (finite element).
 *
 * This is a base class for metadata on cell types held by a vtkCellGrid
 * instance (not for subclasses of vtkCell).
 *
 * @sa vtkCellGrid
 */

#ifndef vtkCellMetadata_h
#define vtkCellMetadata_h

#include "vtkCellGridResponders.h"
#include "vtkObject.h"
#include "vtkNew.h" // for queries
#include "vtkSmartPointer.h" // for constructor return values
#include "vtkStringToken.h" // for vtkStringToken::Hash
#include "vtkCommonDataModelModule.h" // for export macro

#include <functional>
#include <set>
#include <unordered_map>

class vtkCellGrid;
class vtkCellGridQuery;
class vtkDataSetAttributes;
class vtkCellAttribute;

class VTKCOMMONDATAMODEL_EXPORT vtkCellMetadata : public vtkObject
{
public:
  using CellTypeId = std::size_t;
  using DOFType = vtkStringToken;
  using MetadataConstructor = std::function<vtkSmartPointer<vtkCellMetadata>(vtkCellGrid*)>;
  using ConstructorMap = std::unordered_map<vtkStringToken, MetadataConstructor>;

  vtkTypeMacro(vtkCellMetadata, vtkObject);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  /**
   * Register a subclass of vtkCellMetadata.
   */
  template<typename Subclass>
  static bool RegisterType()
  {
    auto dummy = vtkSmartPointer<Subclass>::New();
    vtkStringToken name(dummy->GetClassName());
    auto status = vtkCellMetadata::Constructors.insert(
      std::make_pair(name, [](vtkCellGrid* grid)
        {
          auto result = vtkSmartPointer<Subclass>::New();
          if (result)
          {
            result->SetCellGrid(grid);
          }
          return result;
        }
      )
    );
    return status.second; // true if insertion occurred.
  }

  static vtkSmartPointer<vtkCellMetadata> NewInstance(
    vtkStringToken className, vtkCellGrid* grid = nullptr);

  /**
   * Return a hash of the cell type.
   *
   * WARNING: If you change this method, you must also update
   * vtkCellGrid::AddCellMetadata() and vtkCellGrid::GetCellsOfType().
   */
  CellTypeId Hash() { return typeid(this).hash_code(); }

  /**
   * Set the vtkCellGrid holding DOF arrays required by this cell.
   *
   * If the given vtkCellGrid does not have the required degrees
   * of freedom arrays (as provided by GetDOFTypes()), then this
   * method will return false.
   *
   * If this method returns true, the \a parent grid will have
   * incremented the reference count of \a this class instance.
   */
  virtual bool SetCellGrid(vtkCellGrid* parent);

  /// Return the parent vtkCellGrid that owns this instance (or nullptr).
  vtkGetObjectMacro(CellGrid, vtkCellGrid);

  /// Return the number of cells of this type in the parent cell-grid object.
  /// Subclasses must override this method.
  virtual vtkIdType GetNumberOfCells() { return 0; }

  /**
   * Return the set of DOF array types this cell-type requires for the given
   * interpolant (either the cell's geometry or attributes defined over the geometry).
   *
   * In order for cell metadata to be added to a vtkCellGrid, all of the
   * array types in the returned set must be present and non-empty in
   * the cell-grid for the cell geometry at a minimum.
   *
   * For example, a traditional C-1, linear hex defined by 8 corner points
   * and which admits (1) spatially varying functions defined over points
   * and (2) spatially constant functions defined over each cell would return
   * a "mandatory" set containing "coords"_token and "conn"_token.
   * If point- or cell-data were present, "point_fields"_token and
   * "cell_fields"_token arrays would be expected.
   *
   * The "coords" attribute holds point coordinates;
   * the "conn" attribute holds integer connectivity entries (lists of corner
   * points for each cell, 8 per cell); and
   * the "point_fields"_token and "cell_fields"_token hold arrays defining
   * functions over the geometric shapes.
   *
   * A discontinuous Galerkin hex with nonlinear quadratic geometry
   * defined by corner points and spectral Legendre shape functions
   * would return a "mandatory" set holding "coords"_token, "conn"_token,
   * "shape"_token, and "order"_token.
   * The "shape" attribute would hold coefficients matching Legendre basis
   * functions for each cell while the "order" attribute would hold the
   * geometric order of each cell along each coordinate axis.
   * The "order" values determine the offset into the "shape" array of each
   * cell.
   *
   * There are other ways these cell types might organize their data, but
   * however they choose to do so, this function should return the set of
   * array-partitions needed.
   */
  // virtual std::set<DOFType> RequiredDOFs(const vtkCellAttribute& forAttribute) const;

  /// Return a reference to some subclass of vtkCellAttribute that indicates
  /// which DOF arrays the cell requires in order to provide its shape (i.e.,
  /// the map from a reference cell into world coordinates).
  /// This attribute-type is special as it is the only one required by VTK
  /// in order for cells to be drawn.
  /// Coloring, contouring, and other analyses may require others.
  // virtual const vtkCellAttribute& GetShapeAttribute() const = 0;

  /// Return all the vtkCellAttribute objects for functions defined over the cells in the parent vtkCellGrid.
  /// This method inspects the DOF arrays present in the cell-grid and determines the names and types of
  /// all the attributes present.
  // virtual std::set<std::reference_wrapper<vtkCellAttribute>> GetGridAttributes() const = 0;

  /**
   * Respond to a query on cells of this type.
   *
   * This returns true on success and false on failure.
   * If no responder has been registered for queries of this type,
   * that is considered a failure.
   */
  bool Query(vtkCellGridQuery* query);

  /**
   * Copy \a other into this instance (which must be of the same type).
   *
   * These methods exist so subclasses know when they are being copied;
   * the base class has no data to copy, so both ShallowCopy and DeepCopy
   * do nothing.
   */
  virtual void ShallowCopy(vtkCellMetadata* vtkNotUsed(other)) { }
  virtual void DeepCopy(vtkCellMetadata* vtkNotUsed(other)) { }

  /// Return the set of registered responder types.
  static vtkCellGridResponders* GetResponders() { return vtkCellMetadata::Responders; }

protected:
  vtkCellMetadata() = default;
  ~vtkCellMetadata() override;

  vtkCellGrid* CellGrid{nullptr};

  static ConstructorMap Constructors;
  static vtkNew<vtkCellGridResponders> Responders;

private:
  vtkCellMetadata(const vtkCellMetadata&) = delete;
  void operator=(const vtkCellMetadata&) = delete;
};

#endif
