/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkCellGridSidesQuery.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class   vtkCellGridSidesQuery
 * @brief   Compute external faces of a cell-grid.
 */

#ifndef vtkCellGridSidesQuery_h
#define vtkCellGridSidesQuery_h

#include "vtkCellGridQuery.h"

#include <functional>
#include <set>
#include <unordered_map>

class VTKCOMMONDATAMODEL_EXPORT vtkCellGridSidesQuery : public vtkCellGridQuery
{
public:
  static vtkCellGridSidesQuery* New();
  vtkTypeMacro(vtkCellGridSidesQuery, vtkCellGridQuery);
  virtual void PrintSelf(ostream& os, vtkIndent indent);

  void Initialize() override;
  void Finalize() override;

  std::unordered_map<vtkIdType, std::set<int>>& GetSides() { return this->Sides; }

  template<typename T, std::size_t N>
  void AddSide(vtkIdType cell, int side, const std::array<T, N>& conn)
  {
    // Find where we should start hashing and in what direction.
    // The hash always starts at the smallest entry of conn and goes
    // in the direction that has the largest next entry.
    // Examples:
    //   (3, 2, 0, 1) → starts at index 2 (0) and hashes backwards: (0, 2, 3, 1)
    //   (4, 5, 6, 7) → starts at index 0 (4) and hashes backwards: (4, 7, 6, 5)
    //   (7, 3, 6, 2) → starts at index 3 (2) and hashes forwards:  (2, 7, 3, 6)
    std::size_t ss = 0;
    T smin = conn[0];
    for (std::size_t jj = 1; jj < N; ++jj)
    {
      if (conn[jj] < smin)
      {
        smin = conn[jj];
        ss = jj;
      }
    }
    bool forward = conn[(ss + 1) % N] > conn[(ss + N - 1) % N];

    std::size_t hashedValue = std::hash<std::size_t>{}(N);
    if (forward)
    {
      for (std::size_t ii = 0; ii < N; ++ii)
      {
        std::size_t hashedToken = std::hash<T>{}(conn[(ss + ii) % N]);
        vtkCellGridSidesQuery::HashCombine(hashedValue, hashedToken);
      }
    }
    else // backward
    {
      for (std::size_t ii = 0; ii < N; ++ii)
      {
        std::size_t hashedToken = std::hash<T>{}(conn[(ss + N - ii) % N]);
        // hashedValue = hashedValue ^ (hashedToken << (ii + 1));
        vtkCellGridSidesQuery::HashCombine(hashedValue, hashedToken);
      }
    }
    this->Hashes[hashedValue].Sides.insert(Side{cell, side});
  }

protected:
  vtkCellGridSidesQuery() = default;
  ~vtkCellGridSidesQuery() override = default;

  // Hash combiner adapted from boost::hash_combine
  static inline void HashCombine(std::size_t& h, std::size_t k)
  {
    const std::size_t m = 0xc6a4a7935bd1e995ull;
    const int r = 47;

    k *= m;
    k ^= k >> r;
    k *= m;

    h ^= k;
    h *= m;

    // Completely arbitrary number, to prevent 0's
    // from hashing to 0.
    h += 0xe6546b64;
  }

  struct Side
  {
    vtkIdType DOF;
    int SideId;
    bool operator < (const Side& other) const
    {
      return
        (this->DOF < other.DOF) ||
        (this->DOF == other.DOF && this->SideId < other.SideId);
    }
  };
  struct Entry
  {
    std::set<Side> Sides;
  };
  std::unordered_map<std::size_t, Entry> Hashes;
  std::unordered_map<vtkIdType, std::set<int>> Sides;

private:
  vtkCellGridSidesQuery(const vtkCellGridSidesQuery&) = delete;
  void operator=(const vtkCellGridSidesQuery&) = delete;
};

#endif // vtkCellGridSidesQuery_h
