#include "vtkStringManager.h"

#include "vtkObjectFactory.h"

#include <algorithm>
#include <array>
#include <iostream>

constexpr vtkStringManager::Hash vtkStringManager::Invalid;

vtkStandardNewMacro(vtkStringManager);

void vtkStringManager::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
  vtkIndent i2 = indent.GetNextIndent();
  vtkIndent i3 = i2.GetNextIndent();
  os << indent << "Data: " << this->Data.size() << " entries\n";
  for (const auto& entry : this->Data)
  {
    os << i2 << entry.first << ": " << entry.second << "\n";
  }
  os << indent << "Sets: " << this->Sets.size() << " entries\n";
  for (const auto& entry : this->Sets)
  {
    os << i2 << entry.first << ": " << entry.second.size() << " entries\n";
    for (const auto& child : entry.second)
    {
      os << i3 << child << "\n";
    }
  }
}

vtkStringManager::Hash vtkStringManager::Manage(const std::string& ss)
{
  std::pair<Hash, bool> hp;
  {
    std::lock_guard<std::mutex> lock(this->WriteLock);
    hp = this->ComputeInternalAndInsert(ss);
  }
  return hp.first;
}

std::size_t vtkStringManager::Unmanage(Hash hh)
{
  std::size_t num = 0;
  this->WriteLock.lock();
  auto it = this->Data.find(hh);
  if (it == this->Data.end())
  {
    this->WriteLock.unlock();
    return num;
  }
  auto members = this->Sets.find(hh);
  if (members != this->Sets.end())
  {
    // Erase all sets contained in this set recursively.
    for (auto member : members->second)
    {
      this->WriteLock.unlock();
      num += this->Unmanage(member);
      this->WriteLock.lock();
    }
  }
  num += this->Data.erase(hh);
  this->WriteLock.unlock();
  return num;
}

const std::string& vtkStringManager::Value(Hash hh) const
{
  static const std::string empty;
  auto it = this->Data.find(hh);
  if (it == this->Data.end())
  {
    static bool once = false;
    if (!once)
    {
      once = true;
      vtkWarningMacro("Hash " << hh << " is missing from manager. Returning empty string.");
    }
    return empty;
  }
  return it->second;
}

vtkStringManager::Hash vtkStringManager::Find(const std::string& ss) const
{
  std::pair<Hash, bool> hh;
  {
    std::lock_guard<std::mutex> lock(this->WriteLock);
    hh = this->ComputeInternal(ss);
  }
  return hh.second ? hh.first : Invalid;
}

vtkStringManager::Hash vtkStringManager::Compute(const std::string& ss) const
{
  // std::lock_guard<std::mutex> lock(this->WriteLock);
  return this->ComputeInternal(ss).first;
}

vtkStringManager::Hash vtkStringManager::Insert(const std::string& ss, Hash hh)
{
  bool didInsert = false;
  // Verify \a h is managed.
  if (this->Data.find(hh) == this->Data.end())
  {
    return Invalid;
  }
  // Insert \a h into \a ss.
  std::pair<Hash, bool> setHash{ Invalid, false };
  {
    std::lock_guard<std::mutex> lock(this->WriteLock);
    setHash = this->ComputeInternalAndInsert(ss);
    didInsert = this->Sets[setHash.first].insert(hh).second;
    (void) didInsert;
  }
  return setHash.first;
}

bool vtkStringManager::Insert(Hash ss, Hash hh)
{
  bool didInsert = false;
  // Verify \a ss and \a h are managed.
  if (this->Data.find(hh) == this->Data.end() || this->Data.find(ss) == this->Data.end())
  {
    return didInsert;
  }
  {
    std::lock_guard<std::mutex> lock(this->WriteLock);
    didInsert = this->Sets[ss].insert(hh).second;
  }
  return didInsert;
}

bool vtkStringManager::Remove(const std::string& ss, Hash hh)
{
  bool didRemove = false;
  // Verify \a h is managed.
  if (this->Data.find(hh) == this->Data.end())
  {
    return Invalid;
  }
  // Remove \a hh from \a ss.
  {
    std::lock_guard<std::mutex> lock(this->WriteLock);
    auto setHash = this->ComputeInternalAndInsert(ss);
    auto it = this->Sets.find(setHash.first);
    // Verify \a setHash is managed.
    if (it == this->Sets.end())
    {
      return didRemove;
    }
    didRemove = this->Sets[setHash.first].erase(hh) > 0;
    if (didRemove)
    {
      if (this->Sets[setHash.first].empty())
      {
        this->Sets.erase(setHash.first);
      }
    }
  }
  return didRemove;
}

bool vtkStringManager::Remove(Hash ss, Hash hh)
{
  bool didRemove = false;
  std::lock_guard<std::mutex> lock(this->WriteLock);
  auto hit = this->Data.find(hh);
  auto sit = this->Sets.find(ss);
  // Verify \a h is managed and \a ss is a set.
  if (hit == this->Data.end() || sit == this->Sets.end())
  {
    return false;
  }
  // Remove \a h from \a ss.
  didRemove = this->Sets[ss].erase(hh) > 0;
  if (didRemove)
  {
    if (this->Sets[ss].empty())
    {
      this->Sets.erase(ss);
    }
  }
  return didRemove;
}

bool vtkStringManager::Contains(const std::string& ss, Hash hh) const
{
  std::lock_guard<std::mutex> lock(this->WriteLock);
  auto setHash = this->ComputeInternal(ss);
  auto sit = this->Sets.find(setHash.first);
  return (sit != this->Sets.end() && sit->second.find(hh) != sit->second.end());
}

bool vtkStringManager::Contains(Hash ss, Hash hh) const
{
  if (ss == Invalid)
  {
    auto mit = this->Data.find(hh);
    return mit != this->Data.end();
  }
  auto sit = this->Sets.find(ss);
  return (sit != this->Sets.end() && sit->second.find(hh) != sit->second.end());
}

vtkStringManager::Visit vtkStringManager::VisitMembers(Visitor visitor, Hash ss)
{
  if (!visitor)
  {
    return vtkStringManager::Visit::Halt;
  }

  this->WriteLock.lock();
  if (ss == Invalid)
  {
    // Iterate over this->Data.
    for (const auto& entry : this->Data)
    {
      this->WriteLock.unlock();
      if (visitor(entry.first) == vtkStringManager::Visit::Halt)
      {
        return vtkStringManager::Visit::Halt;
      }
      this->WriteLock.lock();
    }
    this->WriteLock.unlock();
    return vtkStringManager::Visit::Continue;
  }

  // Iterate over this->Sets[ss].
  auto sit = this->Sets.find(ss);
  if (sit == this->Sets.end())
  {
    this->WriteLock.unlock();
    return vtkStringManager::Visit::Continue;
  }
  for (const auto& entry : sit->second)
  {
    this->WriteLock.unlock();
    if (visitor(entry) == vtkStringManager::Visit::Halt)
    {
      return vtkStringManager::Visit::Halt;
    }
    this->WriteLock.lock();
  }

  this->WriteLock.unlock();
  return vtkStringManager::Visit::Continue;
}

vtkStringManager::Visit vtkStringManager::VisitSets(Visitor visitor)
{
  if (!visitor)
  {
    return vtkStringManager::Visit::Halt;
  }

  this->WriteLock.lock();
  // Iterate over this->Sets.
  for (const auto& entry : this->Sets)
  {
    this->WriteLock.unlock();
    if (visitor(entry.first) == vtkStringManager::Visit::Halt)
    {
      return vtkStringManager::Visit::Halt;
    }
    this->WriteLock.lock();
  }

  this->WriteLock.unlock();
  return vtkStringManager::Visit::Continue;
}

void vtkStringManager::Reset()
{
  this->WriteLock.lock();
  this->Data.clear();
  this->Sets.clear();
  this->WriteLock.unlock();
}

std::pair<vtkStringManager::Hash, bool> vtkStringManager::ComputeInternal(const std::string& ss) const
{
  std::pair<Hash, bool> result{ vtkStringToken::StringHash(ss.data(), ss.size()), false };
  while (true)
  {
    auto it = this->Data.find(result.first);
    if (it == this->Data.end())
    {
      return result;
    }
    else if (it->second == ss)
    {
      result.second = true;
      return result;
    }
    vtkWarningMacro(
      "String token collision " << ss << " and " << it->second << " both " << it->first << ".");
    ++result.first;
  }
  return result;
}

std::pair<vtkStringManager::Hash, bool> vtkStringManager::ComputeInternalAndInsert(const std::string& ss)
{
  std::pair<Hash, bool> result = this->ComputeInternal(ss);
  if (result.first != Invalid)
  {
    this->Data[result.first] = ss;
    result.second = true;
  }
  return result;
}
