/*=========================================================================

  Program:   Visualization Toolkit
  Module:    TestStringToken.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
// .NAME Test of vtkStringToken.
// .SECTION Description
// Tests build-time tokenizing strings using the vtkStringToken class.

#include "vtkDebugLeaks.h"
#include "vtkStringToken.h"
#include "vtkStringManager.h"

#include <string>
#include <vector>

int TestStringToken(int, char*[])
{
  using namespace vtk::literals; // for ""_token().
  int result = 0;

  vtkStringToken defaultToken; // default constructor should be Invalid token.
  vtkStringToken eid("");
  auto missing = "not present in manager because token constructed from hash, not string."_token;
  vtkStringToken fooTheBar1("foo the bar");
  auto fooTheBar2 = "foo the bar"_token;

  std::cout << "default token is " << defaultToken.GetId() << "\n";
  std::cout << "empty string is " << eid.GetId() << "\n";
  std::cout << "missing is " << missing.GetId() << "\n";
  std::cout << "foo the bar is " << fooTheBar1.GetId() << " == " << fooTheBar2.GetId() << "\n";

  if (defaultToken.GetId() != vtkStringManager::Invalid)
  {
    std::cerr << "ERROR: Default token constructor should be initialized to Invalid token\n";
  }

  std::vector<std::pair<std::string, vtkStringToken::Hash>> tests{{
    { "", 2166136261u },
    { "a", 3826002220u },
    { "b", 3876335077u },
    { "cell", 1759288501u },
    { "curve", 2570585620u },
    { "edge", 1459017788u },
    { "face", 292255708u },
    { "point", 414084241u },
    { "surface", 425316092u },
    { "vertex", 2488493991u },
    { "volume", 786459023u }
  }};
  for (const auto& test : tests)
  {
    vtkStringToken xx(test.first);
    std::cout << "  " << xx.GetId() << " \"" << test.first << "\"\n";
    if (test.second != xx.GetId())
    {
      std::cerr <<"    ERROR: Expected " << test.second << " got " << xx.GetId() << "\n";
    }
  }

  std::cout << "Now look up strings from hash values via vtkStringManager\n";
  auto* sm = vtkStringToken::GetManager();
  sm->VisitMembers([&sm](vtkStringManager::Hash hh)
    {
      std::cout << "  " << hh << " \"" << sm->Value(hh) << "\"\n";
      return vtkStringManager::Visit::Continue;
    }
  );
  auto missingString = sm->Value(missing.GetId());
  std::cout << "missing string from hash: \"" << missingString << "\" (len " << missingString.size() << ")\n";

  return result;
}
