// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
/**
 * @class   vtkJitterPoints
 * @brief   randomly perturb point positions
 *
 * vtkJitterPoints randomly perturbs point positions. It operates on any input
 * vtkPointSet, modifying only the point positions. The amount of perturbation,
 * and the direction of perturbation can be specified. The input topology and
 * point and cell attribute data are preserved on output - the output dataset
 * type is the same as the input.
 *
 * This utility class may be used to randomize probes and data. It is very
 * useful for improving the behavior of Voronoi and Delaunay tessellations
 * as it removes degeneracies.
 *
 * @warning
 * Large amount of jitter may turn topology inside out, and/or cause
 * surface self-intersection.
 *
 * @warning
 * This class has been threaded with vtkSMPTools. Using TBB or other
 * non-sequential type (set in the CMake variable
 * VTK_SMP_IMPLEMENTATION_TYPE) may improve performance significantly.
 *
 * @sa
 * vtkVoronoi2D vtkVoronoi3D vtkDelunay2D vtkDelaunay3D
 */

#ifndef vtkJitterPoints_h
#define vtkJitterPoints_h

#include "vtkFiltersModelingModule.h" // For export macro
#include "vtkPointSetAlgorithm.h"

VTK_ABI_NAMESPACE_BEGIN

class vtkMinimalStandardRandomSequence;

class VTKFILTERSMODELING_EXPORT vtkJitterPoints : public vtkPointSetAlgorithm
{
public:
  ///@{
  /**
   * Standard methods for instantiating, obtaining type information, and
   * printing information.
   */
  static vtkJitterPoints* New();
  vtkTypeMacro(vtkJitterPoints, vtkPointSetAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;
  ///@}

  ///@{
  /**
   * Specify the maximum jitter distance. The jitter radius can be
   * specified as an absolute number, or a fraction of the input
   * dataset bounding box (diagonal) length. By default, it is a relative
   * number.
   */
  vtkSetClampMacro(Radius,double,0,VTK_DOUBLE_MAX);
  vtkGetMacro(Radius,double);
  vtkSetMacro(RadiusIsAbsolute, vtkTypeBool);
  vtkBooleanMacro(RadiusIsAbsolute, vtkTypeBool);
  vtkGetMacro(RadiusIsAbsolute, vtkTypeBool);
  ///@}


  ///@{
  enum JitterConstraints
  {
    UNCONSTRAINED=0,
    XY_PLANE=1,
    XZ_PLANE=2,
    YZ_PLANE=3
  };

  /**
   * Specify constraints on the point perturbations (i.e., jitter motion). By
   * default, points are unconstrained and randomly jittered in the x-y-z
   * directions (UNCONSTRAINED). Other options include motion constraints in
   * the x-y plane (XY_PLANE), the x-z plane (XZ_PLANE), and the y-z plane
   * (YZ_PLANE).
   */
  vtkSetClampMacro(Jitter, int, UNCONSTRAINED, YZ_PLANE);
  vtkGetMacro(Jitter, int);
  void SetJitterToUnconstrained() { this->SetJitter(UNCONSTRAINED); }
  void SetJitterToXYPlane() { this->SetJitter(XY_PLANE); }
  void SetJitterToXZPlane() { this->SetJitter(XZ_PLANE); }
  void SetJitterToYZPlane() { this->SetJitter(YZ_PLANE); }
  ///@}

  // Jitter a single point at input position xIn to produce the output position
  // xOut (xIn and xOut may be computed in-place). The radius is the allowable
  // range of jitter in the sphere. A sequence is provided, assumed properly
  // initialized, to produce random (0,1) values. Note that if this method is
  // invoked in a thread, separate sequence instanitations (one per thread)
  // should be provided.
  static void JitterXYZ(double inX[3], double outX[3], double radius,
                        vtkMinimalStandardRandomSequence* sequence);

  // Jitter a single point at input position xIn to produce the output
  // position xOut (xIn and xOut may be computed in-place). The radius is the
  // allowable range of jitter in the circle in the x-y plane. A sequence is
  // provided, assumed properly initialized, to produce random (0,1) values.
  // Note that if this method is invoked in a thread, separate sequence
  // instanitations (one per thread) should be provided.
  static void JitterXY(double inX[3], double outX[3], double radius,
                       vtkMinimalStandardRandomSequence* sequence);

  // Jitter a single point at input position xIn to produce the output
  // position xOut (xIn and xOut may be computed in-place). The radius is the
  // allowable range of jitter in the circle in the x-z plane. A sequence is
  // provided, assumed properly initialized, to produce random (0,1) values.
  // Note that if this method is invoked in a thread, separate sequence
  // instanitations (one per thread) should be provided.
  static void JitterXZ(double inX[3], double outX[3], double radius,
                       vtkMinimalStandardRandomSequence* sequence);

  // Jitter a single point at input position xIn to produce the output
  // position xOut (xIn and xOut may be computed in-place). The radius is the
  // allowable range of jitter in the circle in the y-z plane. A sequence is
  // provided, assumed properly initialized, to produce random (0,1) values.
  // Note that if this method is invoked in a thread, separate sequence
  // instanitations (one per thread) should be provided.
  static void JitterYZ(double inX[3], double outX[3], double radius,
                       vtkMinimalStandardRandomSequence* sequence);

protected:
  vtkJitterPoints();
  ~vtkJitterPoints() override;

  double Radius;
  vtkTypeBool RadiusIsAbsolute;
  int Jitter;

  // Support second input
  int RequestData(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;

private:
  vtkJitterPoints(const vtkJitterPoints&) = delete;
  void operator=(const vtkJitterPoints&) = delete;
};

VTK_ABI_NAMESPACE_END
#endif
