// SPDX-FileCopyrightText: Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
// SPDX-License-Identifier: BSD-3-Clause
#include "vtkJitterPoints.h"

#include "vtkArrayDispatch.h"
#include "vtkCellData.h"
#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkMinimalStandardRandomSequence.h"
#include "vtkObjectFactory.h"
#include "vtkPointData.h"
#include "vtkPointSet.h"
#include "vtkPoints.h"
#include "vtkSMPThreadLocal.h"
#include "vtkSMPThreadLocalObject.h"
#include "vtkSMPTools.h"
#include "vtkStreamingDemandDrivenPipeline.h"

VTK_ABI_NAMESPACE_BEGIN
vtkStandardNewMacro(vtkJitterPoints);

namespace //anonymous
{

// Perform the actual point jittering depending on mode.
struct JitterWorker
{
  template <typename InPT, typename OutPT>
  void operator()(InPT* inPts, OutPT* outPts, vtkJitterPoints* self, double radius)
  {
    vtkIdType numPts = inPts->GetNumberOfTuples();
    const auto ipts = vtk::DataArrayTupleRange<3>(inPts);
    auto opts = vtk::DataArrayTupleRange<3>(outPts);
    int jitterMode = self->GetJitter();

    // We use a threshold to test if the data size is small enough
    // to execute the functor serially.
    vtkSMPThreadLocalObject<vtkMinimalStandardRandomSequence> LocalGenerator;
    vtkSMPTools::For(0, numPts, 10000,
      [self, jitterMode, radius, &ipts, &opts, &LocalGenerator]
      (vtkIdType ptId, vtkIdType endPtId)
      {
        auto& localGen = LocalGenerator.Local();
        bool isFirst = vtkSMPTools::GetSingleThread();
        for (; ptId < endPtId; ++ptId)
        {
          if (isFirst)
          {
            self->CheckAbort();
          }
          if (self->GetAbortOutput())
          {
            break;
          }
          const auto xi = ipts[ptId];
          auto xo = opts[ptId];

          double x[3];
          x[0] = static_cast<double>(xi[0]);
          x[1] = static_cast<double>(xi[1]);
          x[2] = static_cast<double>(xi[2]);

          localGen->Initialize(ptId); //produce invariant output
          switch (jitterMode)
          {
            case vtkJitterPoints::UNCONSTRAINED:
              vtkJitterPoints::JitterXYZ(x,x,radius,localGen);
              break;
            case vtkJitterPoints::XY_PLANE:
              vtkJitterPoints::JitterXY(x,x,radius,localGen);
              break;
            case vtkJitterPoints::XZ_PLANE:
              vtkJitterPoints::JitterXZ(x,x,radius,localGen);
              break;
            case vtkJitterPoints::YZ_PLANE:
              vtkJitterPoints::JitterYZ(x,x,radius,localGen);
              break;
          }

          xo[0] = x[0];
          xo[1] = x[1];
          xo[2] = x[2];
        }
      }); // lambda
  }
}; // JitterWorker

} // anonymous namespace

//------------------------------------------------------------------------------
vtkJitterPoints::vtkJitterPoints()
{
  this->Radius = 0.0001;
  this->RadiusIsAbsolute = false;
  this->Jitter = vtkJitterPoints::UNCONSTRAINED;
}

//------------------------------------------------------------------------------
vtkJitterPoints::~vtkJitterPoints() = default;

//------------------------------------------------------------------------------
int vtkJitterPoints::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector, vtkInformationVector* outputVector)
{
  // get the info objects
  vtkInformation* inInfo = inputVector[0]->GetInformationObject(0);
  vtkInformation* outInfo = outputVector->GetInformationObject(0);

  // get the input and output
  vtkPointSet* input = vtkPointSet::SafeDownCast(inInfo->Get(vtkDataObject::DATA_OBJECT()));
  vtkPointSet* output = vtkPointSet::SafeDownCast(outInfo->Get(vtkDataObject::DATA_OBJECT()));

  // Set the jitter radius.
  double radius = this->Radius;
  if ( ! this->RadiusIsAbsolute )
  {
    radius = input->GetLength() * this->Radius;
  }

  // Shallow copy the input to the output. We'll replace the jittered points
  // as well.
  output->CopyStructure(input);
  output->GetPointData()->PassData(input->GetPointData());
  output->GetCellData()->PassData(input->GetCellData());

  vtkPoints *inPts = input->GetPoints();
  vtkNew<vtkPoints> newPts;
  newPts->SetDataType(inPts->GetDataType());
  newPts->SetNumberOfPoints(inPts->GetNumberOfPoints());
  output->SetPoints(newPts);

  // Dispatch the jitter process. Fastpath for real types, fallback to slower
  // path for non-real types.
  using vtkArrayDispatch::Reals;
  using JitterDispatch = vtkArrayDispatch::Dispatch2ByValueType<Reals,Reals>;
  JitterWorker jitterWorker;

  // Finally execute the jittering operation.
  if (!JitterDispatch::Execute(
    inPts->GetData(), newPts->GetData(), jitterWorker, this, radius))
  { // fallback to slowpath
    jitterWorker(inPts->GetData(), newPts->GetData(), this, radius);
  }

  return 1;
}

//------------------------------------------------------------------------------
// Jitter a single point at input position xIn to produce the output position
// xOut (xIn and xOut may be computed in-place). The radius is the allowable
// range of jitter in the sphere. A sequence is provided, assumed properly
// initialized, to produce random (0,1) values.
void vtkJitterPoints::JitterXYZ(double xIn[3], double xOut[3], double radius,
                                vtkMinimalStandardRandomSequence* sequence)
{
  double cosphi = 1 - 2 * sequence->GetNextRangeValue(0,1);
  double sinphi = sqrt(1 - cosphi * cosphi);
  double rho = radius * pow(sequence->GetNextRangeValue(0,1), 0.33333333);
  double R = rho * sinphi;
  double theta = 2.0 * vtkMath::Pi() * sequence->GetNextRangeValue(0,1);
  xOut[0] = xIn[0] + R * cos(theta);
  xOut[1] = xIn[1] + R * sin(theta);
  xOut[2] = xIn[2] + rho * cosphi;
}

//------------------------------------------------------------------------------
// Jittering constrained to x-y plane.
void vtkJitterPoints::JitterXY(double xIn[3], double xOut[3], double radius,
                               vtkMinimalStandardRandomSequence* sequence)
{
  double R = radius * sequence->GetNextRangeValue(0,1);
  double theta = 2.0 * vtkMath::Pi() * sequence->GetNextRangeValue(0,1);
  xOut[0] = xIn[0] + R * cos(theta);
  xOut[1] = xIn[1] + R * sin(theta);
  xOut[2] = xIn[2];
}

//------------------------------------------------------------------------------
// Jittering constrained to x-z plane.
void vtkJitterPoints::JitterXZ(double xIn[3], double xOut[3], double radius,
                               vtkMinimalStandardRandomSequence* sequence)
{
  double R = radius * sequence->GetNextRangeValue(0,1);
  double theta = 2.0 * vtkMath::Pi() * sequence->GetNextRangeValue(0,1);
  xOut[0] = xIn[0] + R * cos(theta);
  xOut[1] = xIn[1];
  xOut[2] = xIn[2] + R * sin(theta);
}

//------------------------------------------------------------------------------
// Jittering constrained to y-z plane.
void vtkJitterPoints::JitterYZ(double xIn[3], double xOut[3], double radius,
                               vtkMinimalStandardRandomSequence* sequence)
{
  double R = radius * sequence->GetNextRangeValue(0,1);
  double theta = 2.0 * vtkMath::Pi() * sequence->GetNextRangeValue(0,1);
  xOut[0] = xIn[0];
  xOut[1] = xIn[1] + R * cos(theta);
  xOut[2] = xIn[2] + R * sin(theta);
}

//------------------------------------------------------------------------------
void vtkJitterPoints::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);

  os << indent << "Jitter: " << this->Jitter << "\n";
  os << indent << "Radius: " << this->Radius << "\n";
  os << indent << "Radius Is Absolute: " << (this->RadiusIsAbsolute ? "True\n" : "False\n");
}
VTK_ABI_NAMESPACE_END
