/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkVoronoi3D.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkVoronoi3D.h"

#include "vtkCellArray.h"
#include "vtkCellData.h"
#include "vtkDoubleArray.h"
#include "vtkIdTypeArray.h"
#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkMath.h"
#include "vtkMinimalStandardRandomSequence.h"
#include "vtkObjectFactory.h"
#include "vtkPlane.h"
#include "vtkPointData.h"
#include "vtkPointSet.h"
#include "vtkPolyData.h"
#include "vtkSMPThreadLocalObject.h"
#include "vtkSMPTools.h"
#include "vtkStaticPointLocator.h"
#include "vtkStreamingDemandDrivenPipeline.h"
#include "vtkTriangle.h"
#include "vtkUnsignedCharArray.h"
#include "vtkUnstructuredGrid.h"

#include <vector>

VTK_ABI_NAMESPACE_BEGIN
vtkStandardNewMacro(vtkVoronoi3D);

namespace //anonymous
{
//======= These structs and methods are for topologically merging points.

// The struct TopoCoord is the coordinate of a Voronoi hull vertex/point
// represented in terms of a 4-tuple: the four neighboring Voronoi generators
// which produce the hull vertex. The components of the 4-tuple are sorted in
// ascending order. It just so happens that due to the dual relationship of
// Voronoi and Delaunay, the TopoCoord is a tetrahedron, and the x-y-z point
// corresponding to the vertex/point position is the circumcenter of the
// tetrahedron.
struct TopoCoord
{
  // Four points defining a topological coord tuple / Delaunay tet. std::array
  // has some nice properties including being easily sortable.
  std::array<vtkIdType,4> Ids; // four points defining a topological tuple / Delaunay tet

  // Various flavors of constructors
  TopoCoord() : Ids{0} {}
  TopoCoord(vtkIdType ids[3], vtkIdType ptId)
    : Ids{ids[0],ids[1],ids[2],ptId}
  {
    std::sort(this->Ids.data(),this->Ids.data()+this->Ids.size());
  }

  // Copy constructor assumes that tuple ids are already sorted.
  TopoCoord(const TopoCoord& tt)
  {
    this->Ids = tt.Ids;
  }

  // Operator< used to support a subsequent sort operation of the 4-tuples
  // (used for uniquely identifying and producing a topologcally coincident
  // point).
  bool operator<(const TopoCoord& tuple) const
  {
    return this->Ids < tuple.Ids;
  }
}; // TopoCoord

// The MergeTuple is used to keep track of the ids of the hull vertices that
// produce a corresponding TopoCoord. All hull vertices with the same
// TopoCoord are coincident.
struct MergeTuple : public TopoCoord
{
  vtkIdType HullVId; // the id of the hull vertex that produced this tuple

  MergeTuple() : HullVId(-1) {}
  bool operator!=(const MergeTuple& mt) const
  {
    return (this->Ids != mt.Ids);
  }
}; // MergeTuple

//======= These structs and methods are for creating and clipping polyhedron
//------------------------------------------------------------------------------
// Internal data structures and methods to generate 3D Voronoi
// tessellations. The filter operates by performing polyhedral clipping by
// processing only neighborhood points falling within a distance measure (the
// Voronoi flower).  The clipping operation is done independently in each
// thread, and then composited together to produce the filter output.
//
//------------------------------------------------------------------------------
// Implementation comment:
// The size of the polyhedron is assumed to be typically dozens (or fewer)
// faces.  Consequently the implementation is serial, and relatively
// simple. For very large polyhedra, many of the operations could be
// threaded.

// Represent a point of the Voronoi polyhedron. This includes the position,
// the evaluated value against the current cutting plane, and radius**2 of
// the distance of the point to the generating point (i.e., the Voronoi petal
// radius**2).  Note: the id of the point is implicitly known from its
// position in the PointsArray. Points can be marked (and mapped to
// a new point id) as being used so as to gather information about the
// points. The Faces[3] are the three faces (defined by the three nearby point
// generators) whose separation planes intersect to produce the point. If
// more than three faces meet at the point then the point is degenerate.
struct VPoint
{
  double X[3]; //position
  double Val;  //value against half-space clipping plane
  double R2;   //Voronoi petal radius
  vtkIdType PtMap; //keep track of points that are actually used by selected faces
  vtkIdType Faces[3]; //the three neighboring generating points defining this point
  bool Degenerate; //indicate whether the point is degenerate

  // methods to define new point
  VPoint(double x, double y, double z)
    : X{x,y,z}
    , Val(0)
    , R2(0.0)
    , PtMap(-1)
    , Faces{-1,-1,-1}
    , Degenerate(false)
  {
  }
  VPoint(double x[3])
    : X{x[0],x[1],x[2]}
    , Val(0)
    , R2(0.0)
    , PtMap(-1)
    , Faces{-1,-1,-1}
    , Degenerate(false)
  {
  }
  // Note that besides copying the x-y-z coordinate, the copy constructor
  // copies the R2 as we do not want to recalculate it. The using faces are
  // also copied. (These using faces are in effect a topological coordinate
  // which can be used to merge coincident points and/or produce a Delaunay
  // tetrahedralization.)
  VPoint(const VPoint& p)
    : X{p.X[0],p.X[1],p.X[2]}
    , Val(0)
    , R2(p.R2)
    , PtMap(-1)
    , Faces{p.Faces[0],p.Faces[1],p.Faces[2]}
    , Degenerate(p.Degenerate)
  {
  }
};   // VPoint

// Represent a polyhedral edge intersected by the current cutting plane.  The
// edge tuple (the edge's two end points) plus the id of the point of
// intersection, and the two faces intersecting the edge, are retained. The
// edge tuple assumes that the two vertex end points are ordered v0 < v1.
//
// Note that after intersection with the cutting plane, the list of EdgeTuple
// intersection points form a new face. However, the intersection points must
// be sorted around the perimeter of the face, hence determining the loop
// index which represents an ordering of points around the face. The loop
// index is computed by defining a coordinate system from the trim plane
// normal, and the fan of diagonals connecting the points of the face loop.
// This can be used to create new planes that cut the loop into pieces. By
// counting the number of points on either side of these each plane, a loop
// index can be determined.
struct EdgeTuple
{
  vtkIdType V0; //min edge vertex id
  vtkIdType V1; //max edge vertex id
  vtkIdType Id; //point id of inserted point
  vtkIdType LoopIdx; //order of the point around the new face.
  vtkIdType Faces[2]; //the two faces using the edge.

  EdgeTuple(vtkIdType v0, vtkIdType v1)
    : V0(v0)
    , V1(v1)
    , Id(-1)
    , LoopIdx(-1)
    , Faces{-1,-1}
  {
    if (this->V0 > this->V1)
    {
      std::swap(this->V0, this->V1);
    }
  }
  bool operator==(const EdgeTuple& et) const { return this->V0 == et.V0 && this->V1 == et.V1; }
  bool operator!=(const EdgeTuple& et) const { return this->V0 != et.V0 || this->V1 != et.V1; }
  bool IsEdge(vtkIdType v0, vtkIdType v1) const
  {
    if (v0 < v1) // ordered properly
    {
      return (this->V0 == v0 && this->V1 == v1);
    }
    else // swap comparison required
    {
      return (this->V0 == v1 && this->V1 == v0);
    }
  }
};

using PointsArray = std::vector<VPoint>; //the polyhedral points (and associated attributes)
using InsertedPointsMap = std::vector<vtkIdType>; //maps input point id to output point id
using InsertedEdgePointsArray = std::vector<EdgeTuple>; //collect edge intersection points


// The face array is just a list of faces for the polyhedron, encoded in a
// contiguous array. For each face there are: 1) the number of points in the
// face; 2) the id of the neighboring point generating the face; and 3) the
// list of point ids defining the face.
using FaceArray = std::vector<vtkIdType>; //a list of polyhedral faces

// Information objects collect information about the parallel processing
// process. This information is gathered and typically rolled up (e.g.,
// prefix sum) and used to configure the output.
struct DataInformation
{
  // Initially these are "number of.." that are transformed to offsets
  // via a subsequent prefix sum operation.
  vtkIdType NumPts; // number of points produced
  vtkIdType NumCells;// number of cells/prims produced
  vtkIdType NumFaces;// number of cell faces produced
  vtkIdType CellConnSize;// size of cell connectivity array
  vtkIdType FaceConnSize;// size of cell faces connectivity array

  DataInformation()
    : NumPts(0)
    , NumCells(0)
    , NumFaces(0)
    , CellConnSize(0)
    , FaceConnSize(0) {}

  // Operator += provides support for prefix sum. Converts counts
  // to offsets.
  DataInformation& operator+=(const DataInformation& info)
  {
    this->NumPts += info.NumPts;
    this->NumCells += info.NumCells;
    this->NumFaces += info.NumFaces;
    this->CellConnSize += info.CellConnSize;
    this->FaceConnSize += info.FaceConnSize;

    return *this;
  }
};
using DataInformationType = std::vector<DataInformation>;

//======= These structs and methods are for creating the processing the
//======= adjacency graph / wheel and spokes data structure.

// The adjacency graph is composed of spokes and wheels.  Spokes are used to
// track connecting edges between generating points (i.e., a connected edge,
// or spoke, connects two generating Voronoi points that produce a clipping
// half-space). A list of spokes forms a wheel. (Note in 2D, the spokes are
// sorted around each generating point in CCW order.) Each spoke corresponds
// to a Voronoi hull face (3D) or tile edge (2D).  The spoke and wheel edge
// data structure can be used to optionally validate the topology, and
// ultimately generate output (such as a surface net, or a Delaunay
// tetrahedralization, as requested). Note that the Wheels array is
// essentially an offset into the array of spokes, indicating the beginning
// of a group of n spokes that forms the wheel.
//
// Each spoke may be classified differently. The classification takes into
// account the point ids on each end of the spoke and the order/direction in
// which the two end points occur, and if provided, the scalar region ids
// associated with the spoke's two end points. In this classification
// process, a spoke is described by its two end points (ptId,neiId), and the
// optional scalar region ids (p,n). The classification for VALID edges is as
// follows. (Note that ptId is always valid with ptId >= 0.):
// + neiId < 0 || n < 0 : DOMAIN_BOUNDARY (e.g., Voronoi domain boundary)
// + ptId < neiId : FORWARD_SPOKE (may produce output face)
// + FORWARD_SPOKE && p != n : REGION_BOUNDARY (e.g., Surface Net)
// Note that VALID spokes may be in the backward direction; since we use
// ordered selection (ptId < neiId), backward spokes do not produce output.
//
// The classification for INVALID edges is:
// + Both spokes (ptId,neiId) and (neiId,ptId) must exist, if not: DEGENERATE
// + (ptId,neiId) and (neiId,ptId) both exist as FORWARD_EDGES,
//   but form part of a topological bubble: BUBBLE
// + When just one spoke is associated with a wheel, the spoke is a SINGLETON
struct Spoke
{
  vtkIdType NeiId; // Id of the wheel that the spoke is connected to (wheelId,Id)
  unsigned char Classification; // Indicate the classification of this spoke
  enum SpokeClass
  {
    INVALID=0,         // Bit 0: Invalid spoke, does not contribute to the output
    VALID=1,           // Bit 0: A valid spoke, may contribute to the output
    FORWARD_SPOKE=3,   // A valid spoke, in the forward direction
    DOMAIN_BOUNDARY=5, // A valid spoke, on the domain boundary
    REGION_BOUNDARY=11,// A forward spoke, connecting two different valid regions
    DEGENERATE=2,      // An invalid spoke, used only one time by a spoke
    BUBBLE=4,          // An invalid spoke, forms a topological bubble
    SINGLETON=8,       // Wheel with only one spoke, should never happen
  };
  Spoke() : NeiId(-1), Classification(VALID) {}
  Spoke(vtkIdType neiId, unsigned char classification) :
    NeiId(neiId), Classification(classification) {}
};

// Typedefs related to edge spoke structure. Note that the WheelsType vector
// is used to keep track of the number of spokes in a Voronoi hull, as well
// as the number of faces in the Voronoi hull (due to dual property, there is
// a one-to-one correspondance between spoke and a hull face).
using WheelsType = std::vector<vtkIdType>;
using SpokeType = std::vector<Spoke>;
using SpokerIterator = SpokeType::iterator;

// Accumulate local points and cells produced by each thread. Later this
// information may be combined to produce filter output.
struct HullVertex
{
  double X[3];
  HullVertex(double x, double y, double z)
    : X{x,y,z} {}
  HullVertex(const double x[3])
    : X{x[0],x[1],x[2]} {}
};

// Used to collect thread local output. Note that the CellConnType is a list
// of vtkIdType, with format (numPrimPts, p0, p1, ... numPrimPts, p0, p1, ...).
using HullVertexType = std::vector<HullVertex>;
using TopoCoordType = std::vector<TopoCoord>;
using CellConnType = std::vector<vtkIdType>;
using MergeTupleType = std::vector<MergeTuple>; //support point merging/smoothing
using MergeMapType = std::vector<vtkIdType>; //maps point ids
using PtsWrittenFlags = std::vector<unsigned char>; //track writing merged points

// ========== These helper functions are created to break cyclic dependencies
// ========== between classes.

// Helper function: Given an input point id, return the offset into the spokes
// array.
vtkIdType GetWheelOffset(const WheelsType& offsets, vtkIdType id)
{ return offsets[id]; }

// Helper function: Given an input point id (or hull id, since there is a
// hull produced for each point), return the number of spokes defining the hull
// polyhedron (i.e., equivalent to the number of polyhedron faces).
vtkIdType GetNumberOfSpokes(const WheelsType& offsets, vtkIdType id)
{ return (offsets[id+1] - offsets[id]); }

// Helper function: Given an input point id (or hull id, since there is a
// hull produced from each input point), return a list of spokes for this
// hull.
const Spoke* GetWheelSpokes(const WheelsType& wheels, const SpokeType& spokes,
                            vtkIdType ptId, vtkIdType& numSpokes)
{
  numSpokes = (wheels[ptId+1] - wheels[ptId]);
  return (spokes.data() + GetWheelOffset(wheels,ptId));
}

// Helper function: given a list of spokes, determine the number of domain boundary
// faces (classified DOMAIN_BOUNDARY spokes); the number of region boundary faces
// (REGION_BOUNDARY); and the number of forward spokes (FORWARD_SPOKE).
void CountFaces(const Spoke* spokes, int numSpokes, int& numDomainBoundaryFaces,
                int& numRegionBoundaryFaces, int& numForwardFaces)
{
  numDomainBoundaryFaces = 0;
  numRegionBoundaryFaces = 0;
  numForwardFaces = 0;
  for (int i=0; i<numSpokes; ++i)
  {
    if ( spokes[i].Classification == Spoke::DOMAIN_BOUNDARY )
    {
      numDomainBoundaryFaces++;
    }
    else if ( spokes[i].Classification == Spoke::REGION_BOUNDARY )
    {
      numRegionBoundaryFaces++;
    }
    else if ( spokes[i].Classification == Spoke::FORWARD_SPOKE )
    {
      numForwardFaces++;
    }
  } // for all spokes. Note backward spokes and invalid spokes ommitted
}

// This class does the actual work of building and operating on the 3D
// Voronoi polyhedron cell. It presumes that it is convex, and intersection
// operations are performed against infinite clipping planes. The class is
// used in tandem with another instance of VPolyhedron to repeatedly clip an
// input polyhedron and then output a new polyhedron (i.e., a double buffering
// approach).
struct VPolyhedron
{
  vtkIdType PtId; // Generating point id
  double X[3]; // Generating point position
  double R2; // Radius**2 of the circumflower
  vtkIntArray *RegionIds; // Optional region information for classification
  bool Prune; // Indicate whether to produce degenerate faces

  // These represent the constructed polyhedron
  PointsArray Points; // Array of points defining this polyhedron
  InsertedPointsMap InsertedPoints; // Lookup map between input / output points
  InsertedEdgePointsArray InsertedEdgePoints; // New points generated on intersected edges
  vtkIdType NumFaces; // The number of faces in the face array
  FaceArray Faces; // A list of faces forming this polyhedron.

  // These are for managing degeneracies
  double Tol;
  double Tol2;

  // The polyhedron is initialized with Initialize().
  VPolyhedron()
    : PtId(-1)
    , X{0,0,0}
    , R2(0)
  {
  }

  // Empty out the polyhedron.
  void Clear()
  {
    // Empty out the points and faces
    this->Points.clear();
    this->Faces.clear();
    this->NumFaces = 0;
  }

  // Update the generating point position and id.
  void Update(vtkIdType genPtId, const double genPt[3])
  {
    this->PtId = genPtId;
    this->X[0] = genPt[0];
    this->X[1] = genPt[1];
    this->X[2] = genPt[2];
  }

  // Create the initial Voronoi polyhedron based on the given
  // bounding box (xMin,xMax,yMin,yMax,zMin,zMax). This creates
  // eight points and six faces.
  void Initialize(double bds[6])
  {
    // Empty out the points and faces
    this->Clear();

    // Add the eight points of the initial bounding polyhedron (box). Also
    // add the three faces that intersect at each point. These three faces
    // are the bounding polyhedron faces (hence their negated value).
    this->AddNewPoint(bds[0],bds[2],bds[4]); //point 0
    this->SetPointFaces(0, -1, -3, -5);
    this->AddNewPoint(bds[1],bds[2],bds[4]); //point 1
    this->SetPointFaces(1, -2, -3, -5);
    this->AddNewPoint(bds[0],bds[3],bds[4]); //point 2
    this->SetPointFaces(2, -1, -4, -5);
    this->AddNewPoint(bds[1],bds[3],bds[4]); //point 3
    this->SetPointFaces(3, -2, -4, -5);
    this->AddNewPoint(bds[0],bds[2],bds[5]); //point 4
    this->SetPointFaces(4, -1, -3, -6);
    this->AddNewPoint(bds[1],bds[2],bds[5]); //point 5
    this->SetPointFaces(5, -2, -3, -6);
    this->AddNewPoint(bds[0],bds[3],bds[5]); //point 6
    this->SetPointFaces(6, -1, -4, -6);
    this->AddNewPoint(bds[1],bds[3],bds[5]); //point 7
    this->SetPointFaces(7, -2, -4, -6);

    // Add the six outside faces of the initial boundng polyhedron
    // (box). Note that the "neighboring" points are outside of the bounding
    // box (indicated by <0 values), representing the infinite space bounding
    // the Voroni cell.
    this->AddNewFace(4,-1); //face 0 of the polyhedron, marked as -1
    this->AddFacePoint(0);
    this->AddFacePoint(4);
    this->AddFacePoint(6);
    this->AddFacePoint(2);

    this->AddNewFace(4,-2); //face 1
    this->AddFacePoint(1);
    this->AddFacePoint(3);
    this->AddFacePoint(7);
    this->AddFacePoint(5);

    this->AddNewFace(4,-3); //face 2
    this->AddFacePoint(0);
    this->AddFacePoint(1);
    this->AddFacePoint(5);
    this->AddFacePoint(4);

    this->AddNewFace(4,-4); //face 3
    this->AddFacePoint(2);
    this->AddFacePoint(6);
    this->AddFacePoint(7);
    this->AddFacePoint(3);

    this->AddNewFace(4,-5); //face 4
    this->AddFacePoint(0);
    this->AddFacePoint(2);
    this->AddFacePoint(3);
    this->AddFacePoint(1);

    this->AddNewFace(4,-6); //face 5
    this->AddFacePoint(4);
    this->AddFacePoint(5);
    this->AddFacePoint(7);
    this->AddFacePoint(6);

    // The initial tolerance is determined during the first half-space clip.
    // Here we indicate that the tolerance is uninitialized.
    this->Tol = this->Tol2 = (-1); //mark unset
  }

  // Copy tolerancing information from the sister VPolyhedron. The tolerance
  // is initially determined from the first half space clip of the bounding
  // box, and then is used until the hull is completed.
  void CopyTolerance(VPolyhedron* vpoly)
  {
    this->Tol = vpoly->Tol;
    this->Tol2 = vpoly->Tol2;
  }

  // Compute the bounding Voronoi flower circumcircle (i.e., contains all
  // petals of the Voronoi flower).
  double ComputeCircumFlower()
  {
    double r2;
    this->R2=VTK_FLOAT_MIN; // Initialize

    // Check against the flower petals
    for (auto pItr=this->Points.begin(); pItr != this->Points.end(); ++pItr)
    {
      this->R2 = ( pItr->R2 > this->R2 ? pItr->R2 : this->R2 );
    }

    this->R2 = (4.0 * this->R2); // 2 * (max petal radius)

    return this->R2;
  }

  // Add a new point to this polyhedron. It also computes the Voronoi
  // flower petal radius, and returns its implicit point id (i..e., location
  // in the points array).
  vtkIdType AddNewPoint(double x, double y, double z)
  {
    vtkIdType id = this->Points.size();
    this->Points.emplace_back(VPoint(x,y,z));
    this->Points.back().R2 =
      vtkMath::Distance2BetweenPoints(this->Points.back().X,this->X);
    return id;
  }
  vtkIdType AddNewPoint(double x[3])
  {
    return this->AddNewPoint(x[0], x[1], x[2]);
  }

  // Copy a point from an input polyhedron to this output polyhedron.  Return
  // the new id of the point in the output polyhedron points list. Do not
  // recompute the Voronoi flower petal radius.
  vtkIdType CopyPoint(PointsArray& inPts, vtkIdType inId )
  {
    vtkIdType id = this->Points.size();
    this->Points.emplace_back(inPts[inId]);
    return id;
  }

  // Each point of the polyhedron is created by the intersection of three
  // or more half-space clipping planes (planes defined by neighboring point
  // generators). Note that in degenerate situations, more than three faces
  // may join at a point, in which case a special value indicates a
  // degenerate point.
  // Specify the three faces which produced the given point.
  void SetPointFaces(vtkIdType pId, vtkIdType f0, vtkIdType f1, vtkIdType f2)
  {
    this->Points[pId].Faces[0] = f0;
    this->Points[pId].Faces[1] = f1;
    this->Points[pId].Faces[2] = f2;
  }

  // Mark this point degenerate.
  void MarkPointDegenerate(vtkIdType pId)
  {
    this->Points[pId].Degenerate = true;
  }

  // Add a new face to the list of polyhedron faces. Provide the number of
  // faces defining the face, plus the associated neighbor point id which
  // generates the Voronoi face. Return an offset into the faces array so
  // that the number of points in the face can be updated if necessary.
  vtkIdType AddNewFace(vtkIdType npts, vtkIdType neiPtId)
  {
    vtkIdType offset = this->Faces.size();
    this->Faces.emplace_back(npts);
    this->Faces.emplace_back(neiPtId);
    this->NumFaces++;
    return offset;
  }

  // Add a point id defining the current face. This method is called
  // after AddNewFace().
  void AddFacePoint(vtkIdType ptId)
  {
    this->Faces.emplace_back(ptId);
  }

  // Update the number of points in the current face. Provide the offset
  // into the current face (from AddNewFace()).
  void UpdateFaceCount(vtkIdType offset, vtkIdType npts)
  {
    this->Faces[offset] = npts;
  }

  // Prepare to add a batch of N points.
  vtkIdType AddNFacePoints(vtkIdType batchSize)
  {
    vtkIdType offset = this->Faces.size();
    this->Faces.resize(offset+batchSize);
    return offset;
  }

  // Add a point to previously allocated batch of N points. The loop index
  // loopIdx must be < batchSize.
  void AddNthFacePoint(vtkIdType offset, vtkIdType loopIdx, vtkIdType ptId)
  {
    this->Faces[offset+loopIdx] = ptId;
  }

  // Determine whether the supplied face should be pruned. The method
  // compares the progressively determined face area against the area
  // threshold.
  bool ShouldPruneFace(double area, vtkIdType numFacePts, vtkIdType* faceIds)
  {
    // Compute area by accumulating triangle areas in a fan pattern. If the
    // accumulated area exceeds the input area, then break out as pruning is
    // not needed.
    double totalArea = 0.0;
    double *p0 = this->Points[faceIds[0]].X;
    double *p1, *p2;

    for (int ptId=1; ptId < numFacePts-1; ++ptId)
    {
      p1 = this->Points[faceIds[ptId]].X;
      p2 = this->Points[faceIds[ptId+1]].X;

      totalArea += vtkTriangle::TriangleArea(p0, p1, p2);
      if ( totalArea >= area )
      {
        return false;
      }
    }
    return true;
  }

  // If pruning faces (and the associated spokes) is enabled, this method
  // will collapse faces of near zero area (relative to the connecting
  // spoke).  Return the number of faces pruned.
  int PruneFaces(double pruneTol2, const double *inPts)
  {
    vtkIdType *faces = this->Faces.data();
    vtkIdType neiPtId, *faceIds, numFacePts, ptId, offset = 0;
    int numPrunes = 0;

    for (int faceNum=0; faceNum < this->NumFaces; ++faceNum)
    {
      vtkIdType *face = faces + offset;
      numFacePts = face[0];
      if ( (neiPtId = face[1]) >= 0 )
      {
        faceIds = face + 2;

        // Compute the spoke length. Determine the area (ignore Pi)
        double spokeLen2 = vtkMath::Distance2BetweenPoints(this->X, inPts+3*neiPtId);
        double area = pruneTol2 * spokeLen2;

        if ( this->ShouldPruneFace(area, numFacePts, faceIds) )
        {
          numPrunes++;
        }
      }

      offset += (numFacePts + 2);
    }
    return numPrunes;
  }

  // Evaluate the 3D plane equation for a given point x. Normal n is expected
  // to be a unit normal to the plane; o is a plane origin (i.e., point on
  // the plane).
  double EvaluatePlane(double x[3], double o[3], double n[3])
  {
    return (((x[0]-o[0])*n[0]) + ((x[1]-o[1])*n[1]) + ((x[2]-o[2])*n[2]));
  }

  // Intersect this Voronoi polyhedron with the plane defined by an origin
  // and unit normal. The Voronoi neighbor point id associated with this
  // plane must also be provided. The result of the operation is placed
  // into a new Voronoi polyhedron.
  bool IntersectWithPlane(double origin[3], double normal[3], vtkIdType neiPtId,
                          VPolyhedron *outPoly)
  {
    // Check to see if a tolerance has been determined. The first clip plane
    // (i.e., the first time this method is invoked for the current
    // polyhedron) is defined by the closest point to the initial polyhedron,
    // so it is a good time to define the tolerance for this
    // polyhedron. Note, however, this means the tolerance is adaptive to
    // point spacing.
    if ( this->Tol < 0 )
    {
      double e = std::numeric_limits<double>::epsilon();
      this->Tol = 10 * e * sqrt(vtkMath::Distance2BetweenPoints(origin,this->X));
      this->Tol2 = this->Tol * this->Tol;
    }

    // Begin by evaluating all the polyhedron points against the plane.  If
    // all points are "inside" (<=0 based on the tolerance) then there is no
    // intersection.
    double val;
    bool intersects=false;
    for (auto pItr=this->Points.begin(); pItr != this->Points.end(); ++pItr)
    {
      val = this->EvaluatePlane(pItr->X, origin, normal);
      val = ( (-this->Tol <= val && val <= this->Tol) ? 0.0 : val);
      if ( val > 0 )
      {
        intersects = true;
      }
      pItr->Val = val;
    }
    if ( !intersects )
    {
      return false;
    }

    // An intersection has occurred. Prepare this polyhedron for further
    // processing, as well as the output polyhedron.
    this->InsertedPoints.resize(this->Points.size());
    std::fill_n(this->InsertedPoints.begin(), this->Points.size(), (-1));
    this->InsertedEdgePoints.clear();
    outPoly->Clear();

    // Now process the faces that compose the current polyhedron. Output
    // points and faces to the output polyhedron.
    vtkIdType *faces = this->Faces.data();
    vtkIdType npts, *pids, neiId;
    vtkIdType offset=0, fOffset;
    while ( offset < this->Faces.size() )
    {
      vtkIdType *face = faces + offset;
      npts = face[0];
      neiId = face[1];
      pids = face + 2;

      // Process the current face. Points in face with Val<=0 are retained.
      // Edges that intersect the plane produce new points. These new points
      // are gathered into a loop to produce one new face.
      vtkIdType i, v0, v1, pid, v, numNewPts=0;
      double val0, val1, t, x[3];
      for ( i=0; i < npts; ++i)
      {
        v0 = pids[i];
        v1 = pids[(((i+1)==npts) ? 0 : (i+1))];

        val0 = this->Points[v0].Val;
        val1 = this->Points[v1].Val;

        // If the plane evaluated value is <=0, add the exisiting point to a
        // new face.
        if ( val0 <= 0.0 )
        {
          // See if this point has been copied previously. If not, insert it.
          if ( (pid = this->InsertedPoints[v0]) < 0 )
          {
            pid = outPoly->CopyPoint(this->Points,v0);
            this->InsertedPoints[v0] = pid;
          }
          // Create a new face if necessary
          if (!numNewPts)
          {
            fOffset = outPoly->AddNewFace(0,neiId);
          }
          // Add the point to the current face.
          outPoly->AddFacePoint(pid);
          numNewPts++;
        } //copying point

        // Now check to see if there is a face edge intersection. If so, create
        // a new point if not already created. Watch out for intersections within
        // tolerance of the edge end points.
        if ( (val0 <= 0.0 && val1 > 0.0) || (val0 > 0.0 && val1 <= 0.0) )
        {
          InsertedEdgePointsArray::iterator insItr =
            std::find(this->InsertedEdgePoints.begin(), this->InsertedEdgePoints.end(),EdgeTuple(v0,v1));
          if ( insItr != this->InsertedEdgePoints.end() )
          {
            pid = insItr->Id;
            insItr->Faces[1] = neiId; //second face
          }
          else // Intersect the edge
          {
            t = -val0 / (val1-val0);
            x[0] = this->Points[v0].X[0] + t*(this->Points[v1].X[0]-this->Points[v0].X[0]);
            x[1] = this->Points[v0].X[1] + t*(this->Points[v1].X[1]-this->Points[v0].X[1]);
            x[2] = this->Points[v0].X[2] + t*(this->Points[v1].X[2]-this->Points[v0].X[2]);
            pid = outPoly->AddNewPoint(x);
            this->InsertedEdgePoints.emplace_back(EdgeTuple(v0,v1));
            this->InsertedEdgePoints.back().Id = pid;
            this->InsertedEdgePoints.back().Faces[0] = neiId; //first face
          }

          // Create a new face if necessary
          if (!numNewPts)
          {
            fOffset = outPoly->AddNewFace(0,neiId);
          }
          outPoly->AddFacePoint(pid);
          numNewPts++;
        } // Create a new face intersection point
      } //for all points and edges in this face

      // See whether the face was copied, or clipped. If so, a new face was created.
      // If not, this face was clipped away.
      if ( numNewPts > 0 )
      {
        outPoly->UpdateFaceCount(fOffset,numNewPts);
      }
      else
      {
        this->NumFaces--; //this face entirely removed / clipped away
      }

      // Advance to the next face
      offset += (npts+2);
    } // while processing faces

    // All the faces have been clipped by the plane. We've gathered all of
    // the intersection points. These now must be circumferentially ordered
    // to create a final capping polygon. Fortunately, the polygon is convex,
    // so the ordering (i.e., assigning of loop index/order) can be performed
    // by counting the number of points on one side of the polygon fan
    // diagonals. (This cute method works well for small number of vertices
    // in the capping polygon--no sqrt nor atan functions, nor angle
    // sort. However, for large numbers of points, which is rare, the method
    // will not scale well and may require an alternative method.

    // The number of vertices forming the capping polygon.
    npts = this->InsertedEdgePoints.size();
    if ( npts < 3 )
    {
      cout << "Degenerate loop with: (" << this->PtId << "," << npts << ") verts\n";
      return true;
    }

    // The fan pivot point, assigned loop index 0. All diagonals across the
    // capping polygon start from the pivot point.
    this->InsertedEdgePoints[0].LoopIdx = 0;
    double *xO = outPoly->Points[this->InsertedEdgePoints[0].Id].X;

    // Now create lines between the pivot point and other points in the
    // capping polygon. These lines define separation planes orthogonal to
    // the clipped polygon. By counting the number of points on one side of
    // the separation plane, we can assign a loop index value.
    double xxO[3], sepN[3];
    for (vtkIdType i=1; i < npts; ++i)
    {
      // Compute the separation plane.
      double *x = outPoly->Points[this->InsertedEdgePoints[i].Id].X;
      xxO[0] = x[0] - xO[0];
      xxO[1] = x[1] - xO[1];
      xxO[2] = x[2] - xO[2];
      vtkMath::Cross(xxO,normal,sepN);

      // Now evaluate all other points against the separation plane. We
      // don't worry about normalizing the separation plane normal because
      // we just counting values >0.
      vtkIdType idx=1; //the index is one to begin with.
      for (vtkIdType j=1; j < npts; ++j)
      {
        if ( i == j )
        {
          continue;
        }
        double *x = outPoly->Points[this->InsertedEdgePoints[j].Id].X;
        if ( this->EvaluatePlane(x, xO, sepN) > 0 )
        {
          idx++;
        }
      } //point evaluation

      // Assign loop index
      this->InsertedEdgePoints[i].LoopIdx = idx;
    } //for all non-origin points forming the capping polygon.

    // Finally, create a new face (the capping polygon) with the points
    // inserted in the correct order (using the loop index).
    outPoly->AddNewFace(npts,neiPtId);
    offset = outPoly->AddNFacePoints(npts);
    for (vtkIdType i=0; i < npts; ++i)
    {
      outPoly->AddNthFacePoint(offset,this->InsertedEdgePoints[i].LoopIdx,
                               this->InsertedEdgePoints[i].Id);
      outPoly->SetPointFaces(this->InsertedEdgePoints[i].Id,
                             this->InsertedEdgePoints[i].Faces[0],
                             this->InsertedEdgePoints[i].Faces[1],
                             neiPtId);
    }

    return true;
  }

  // See VHull documentation.
  const Spoke*
  AddAdjacencyInformation(vtkIdType ptId, WheelsType& wheels,
                          SpokeType& spokes, int& numSpokes,
                          int& maxPoints, int& maxFaces)
  {
    // Keep track of the starting position at which
    // spokes may be added.
    vtkIdType startPos = spokes.size();

    vtkIdType numPts = this->Points.size();
    vtkIdType *faces = this->Faces.data();
    vtkIdType offset = 0;
    numSpokes = 0;
    while ( offset < this->Faces.size() )
    {
      vtkIdType *face = faces + offset;
      vtkIdType npts = face[0];
      vtkIdType neiId = face[1];

      // Classify the spoke - this information is very useful for later
      // processing.
      unsigned char spokeClass = Spoke::VALID;
      if ( neiId < 0 || (this->RegionIds && this->RegionIds->GetValue(neiId)<0))
      {
        spokeClass = Spoke::DOMAIN_BOUNDARY;
      }
      else if ( ptId < neiId )
      {
        spokeClass = Spoke::FORWARD_SPOKE;
        if ( this->RegionIds && (this->RegionIds->GetValue(ptId) !=
                                 this->RegionIds->GetValue(neiId)) )
        {
          spokeClass = Spoke::REGION_BOUNDARY;
        }
      }

      // Create the spoke and add it the spokes vector
      spokes.emplace_back(Spoke(neiId,spokeClass));
      offset += (npts+2);
      ++numSpokes;
    } // while processing polyhedral faces

    wheels[ptId] = numSpokes; //numFaces == numSpokes
    maxPoints = (numPts > maxPoints ? numPts : maxPoints);
    maxFaces = (numSpokes > maxFaces ? numSpokes : maxFaces);

    // Spokes are added with emplace_back(), so may cause
    // reallocations. So we wait for all spokes to be added
    // before returning the pointer to the list of spokes.
    return spokes.data() + startPos;
  }

  // See VHull documentation.
  void AddSurfaceInformation(int outputType, vtkIdType ptId, const Spoke* spokes,
                             int numSpokes, DataInformationType& surfInfo,
                             HullVertexType& localPts, TopoCoordType& localTups,
                             CellConnType& localConn)
  {
    // Generate output only if hull faces exist
    if ( this->NumFaces <= 0 )
    {
      return;
    }

    // Only process the appropriate boundary faces to
    // avoid creating duplicates output etc.
    if ( outputType == vtkVoronoi3D::ADJACENCY_GRAPH ) // add line segments
    {
      int numDomainBoundaryFaces, numRegionBoundaryFaces, numForwardFaces;
      CountFaces(spokes, numSpokes, numDomainBoundaryFaces,
                 numRegionBoundaryFaces, numForwardFaces);

      // Note that the adjacency graph reuses the input points.
      // Only edges within the Voronoi domain are output.
      surfInfo[ptId].NumPts = 0; //no new points are created
      surfInfo[ptId].NumCells = numRegionBoundaryFaces + numForwardFaces;
      surfInfo[ptId].CellConnSize = 2 * surfInfo[ptId].NumCells;
    }

    else //add selected faces of polyhedron
    {
      int faceTypes;
      if ( outputType == vtkVoronoi3D::BOUNDARY )
      {
        faceTypes = Spoke::DOMAIN_BOUNDARY;
      }
      else if ( outputType == vtkVoronoi3D::SURFACE_NET )
      {
        faceTypes = Spoke::REGION_BOUNDARY;
      }
      else // if (outputType == vtkVoronoi3D::POLYGONAL_COMPLEX)
      {
        faceTypes = Spoke::DOMAIN_BOUNDARY;
      }

      // Output domain boundary faces
      int numPts, numFaces, connSize;
      this->AddSurfaceFaces(faceTypes, spokes, numSpokes,
                            numPts, numFaces, connSize, localPts, localTups,
                            localConn);
      surfInfo[ptId].NumPts = numPts;
      surfInfo[ptId].NumCells = numFaces;
      surfInfo[ptId].CellConnSize = connSize;
    }

  } // AddSurfaceInformation()

  // Output local points and faces from current polyhedron, as specified by
  // faceTypes.  The face types are an "ORed |" combination of output types
  // as contained in the provided spokes. Faces matching the faceTypes are
  // output: their points and connectivity. Also, information is gathered
  // about the number of output points and faces, and size of connectivity.
  void AddSurfaceFaces(int faceTypes, const Spoke* spokes, int numSpokes,
                       int &numOutputPts, int& numOutputFaces, int& connSize,
                       HullVertexType& hullPts, TopoCoordType& tups,
                       CellConnType& faceConn)
  {
    // Loop over all the polyhedron faces, checking against the specified type(s).
    vtkIdType *faces = this->Faces.data();
    vtkIdType *faceIds, numFacePts, ptId, offset = 0;
    numOutputFaces = 0;
    connSize = 0;
    numOutputPts = 0;

    for (int faceNum=0; faceNum < this->NumFaces; ++faceNum)
    {
      vtkIdType *face = faces + offset;
      numFacePts = face[0];

      if (spokes[faceNum].Classification == faceTypes)
      {
        numOutputFaces++;
        faceConn.emplace_back(numFacePts);
        faceIds = face + 2;
        connSize += numFacePts;
        for (vtkIdType i=0; i < numFacePts; ++i)
        {
          if (this->Points[faceIds[i]].PtMap < 0)
          {
            this->Points[faceIds[i]].PtMap = numOutputPts++;
            hullPts.emplace_back(HullVertex(this->Points[faceIds[i]].X));
            tups.emplace_back(TopoCoord(this->Points[faceIds[i]].Faces,this->PtId));
          }
          faceConn.emplace_back(this->Points[faceIds[i]].PtMap);
        }
      } // specified face type matches
      offset += (numFacePts + 2);
    } // while processing polyhedral faces
  }

  // See VHull documentation.
  void AddVolumetricInformation(int outputType, vtkIdType ptId, const Spoke* spokes,
                                int numSpokes, DataInformationType& volInfo,
                                HullVertexType& hullPts, TopoCoordType& tups,
                                CellConnType& faceConn)
  {
    // Generate output only if hull points exist
    if ( this->Points.size() <= 0 )
    {
      return;
    }

    // Output possibly multiple tets if Delaunay
    if ( outputType == vtkVoronoi3D::DELAUNAY )
    {
      vtkIntArray* regionIds = this->RegionIds;
      vtkIdType numTets = 0;
      for ( auto pItr=this->Points.begin(); pItr < this->Points.end(); ++pItr )
      {
        vtkIdType *faces = pItr->Faces;
        if ( (ptId<faces[0] && ptId<faces[1] && ptId<faces[2]) && //minimal id tet
             (faces[0] >= 0 && faces[1] >= 0 && faces[2] >= 0) && //all neighbors non-boundary
             (!regionIds || (regionIds->GetValue(faces[0])>=0 &&  //all interior regions
                             regionIds->GetValue(faces[1])>=0 && regionIds->GetValue(faces[2])>=0)) )
        {
          tups.emplace_back(TopoCoord(pItr->Faces,ptId));
          numTets++;
        }
      }

      volInfo[ptId].NumPts = 0; // reuse input points, none created
      volInfo[ptId].NumCells = numTets; // only tets produced
      volInfo[ptId].CellConnSize = numTets * 4;
    }

    // Else, output a single polyhedral hull: all the hull points and faces.
    else // if ( outputType == vtkVoronoi3D::VORONOI )
    {
      // Start by outputting all Voronoi hull points and associated
      // topological coordinates. This has the side effect of (locally)
      // numbering the points.
      vtkIdType numHullPts = this->Points.size();
      for (int i=0; i < numHullPts; ++i)
      {
        this->Points[i].PtMap = i; //number the point
        hullPts.emplace_back(HullVertex(this->Points[i].X));
        tups.emplace_back(TopoCoord(this->Points[i].Faces,this->PtId));
      }

      // There is no need to output cell connectivity: it is implicit in the
      // number of hull points. This is handled in the VolumetricOutput.

      // Now output all of the Voronoi hull faces.
      vtkIdType *faces = this->Faces.data();
      vtkIdType *faceIds, numFacePts, offset = 0;
      vtkIdType faceConnSize = 0;

      for (int faceNum=0; faceNum < this->NumFaces; ++faceNum)
      {
        vtkIdType *face = faces + offset;
        numFacePts = face[0];
        faceConn.emplace_back(numFacePts);
        faceIds = face + 2;
        faceConnSize += numFacePts;
        for (vtkIdType i=0; i < numFacePts; ++i)
        {
          faceConn.emplace_back(this->Points[faceIds[i]].PtMap);
        }
        offset += (numFacePts + 2);
      } // for all hull faces

      volInfo[ptId].NumPts = numHullPts;
      volInfo[ptId].NumCells = 1;
      volInfo[ptId].NumFaces = this->NumFaces;
      volInfo[ptId].CellConnSize = numHullPts;
      volInfo[ptId].FaceConnSize = faceConnSize;
    } // else process the hull
  } // AddVolumetricInformation

}; // VPolyhedron

// The incremental Voronoi class VHull is used to create a 3D Voronoi cell,
// v_i, from incremental insertion of neighboring points around a Voronoi
// generating point p_i. (Note that the Voronoi cell v_i is defined by a
// convex polyhedron whose interior is closer to p_i than any od the the
// neighboring points p_j.) To use this class, define the Voronoi generator
// point p_i, and a 3D bounding box B in which the Voronoi cell exists.
// Then, incrementally inject neighboring points p_j to produce the Voronoi
// cell. Note, that inserted points p_j are not required to intersect the
// v_i; but the bounding box B must strictly contain the generator point
// p_i. Because Voronoi cells may be semi-infinite in extent, B may bound the
// v_i.
//
// The algorithm proceeds as follows. The p_i is initially placed interior to
// B, so that v_i == B. Then as each point p_j is inserted, a Voronoi
// half-space is created by defining a plane which equally separates p_i from
// p_j. This plane is then used to intersect each face of the polyhedron v_i,
// producing a new v_i. The process continues until all neighboring points
// have been inserted.
//
// Note that the class supports related methods such as computing the Voronoi
// flower, and the Voronoi circumflower. It can also output 1) the adjacency
// graph (wheels and spokes connectivity graph); 2) a VTK polydata (faces of
// the v_i, both interior and exterior); 3) a generalized surface net; 4) a
// VTK polyhedral cell; and 5) a local Delaunay tetrahedralization. Internally,
// it uses two instances of VPolyhedron (via a double buffering approach) to
// iteratively build the v_j.
struct VHull
{
  // Define the Voronoi generating point including its associated id.
  vtkIdType NPts;                   // total number of points in dataset
  const double *Points;             // input dataset points
  vtkIntArray *RegionIds;           // Optional region information for classification
  vtkIdType PointId;                // generating hull point id (for this hull)
  double TileX[3];                  // generating hull point - x-y-z coordinates
  vtkStaticPointLocator* Locator;   // locator
  double PaddedBounds[6];           // the domain over which Voronoi is calculated
  double Bounds[6];                 // locator bounds
  int Divisions[3];                 // locator binning dimensions
  double H[3];                      // locator spacing
  double Padding2;                  // Bounding box padding distance
  vtkIdType NumClips;               // The number of clips so far

  // Use a double buffering approach to incrementally construct the Voronoi
  // cells as the cell is repeatedly clipped by the addition of neighboring
  // points.  VPO is the current polyhedron; when a neighboring point is used
  // to trim VPO, the result is written into VP1. Then VP0 and VP1 are
  // swapped. We use double buffering to avoid repeated memory allocation and
  // deletion.
  VPolyhedron *VP0;
  VPolyhedron *VP1;

  // This is used to identify and map the points of the polyhedron to be output.
  vtkIdType NumMarkedPts;

  VHull()
  {
    this->VP0 = new VPolyhedron();
    this->VP1 = new VPolyhedron();
  }
  // Copy constructor needed for SMP
  VHull(const VHull& v)
  {
    this->VP0 = new VPolyhedron();
    this->VP1 = new VPolyhedron();
  }
  ~VHull()
  {
    delete this->VP0;
    delete this->VP1;
  }

  // Convenience methods for setting region segmenation labals
  void SetRegionIds(vtkIntArray* regionIds)
  {
    this->VP0->RegionIds = regionIds;
    this->VP1->RegionIds = regionIds;
  }

  /**
   * Initialize the Voronoi cell v_i by setting it to the provided bounding
   * box B, with generating point p_i. Note that p_i must be strictly
   * contained within B.
   */
  void Initialize(vtkIdType genPtId, const double genPt[3])
  {
    // Update the hull
    this->PointId = genPtId;
    this->TileX[0] = genPt[0];
    this->TileX[1] = genPt[1];
    this->TileX[2] = genPt[2];

    // Initialize the number of clips
    this->NumClips = 0;

    // Start the incremental polyhedral generation process.
    this->VP0->Update(genPtId,genPt);
    this->VP1->Update(genPtId,genPt);
    this->VP0->Initialize(this->PaddedBounds);
  }

  /**
   * Insert the next point neighboring point p_j. The method will return
   * true if the v_i is modified as a result of inserting the point. Otherwise
   * the v_i is not modified. Make sure that Initialize() has been invoked
   * prior to calling this method.
   */
  bool InsertPoint(vtkIdType neiPtId, const double neiPt[3])
  {
    // Order the calculations to obtain the same result.
    double origin[3], normal[3];
    bool negate = false;
    if ( neiPtId < this->PointId )
    {
      origin[0] = (neiPt[0] + this->TileX[0]) / 2.0;
      origin[1] = (neiPt[1] + this->TileX[1]) / 2.0;
      origin[2] = (neiPt[2] + this->TileX[2]) / 2.0;
      normal[0] = neiPt[0] - this->TileX[0];
      normal[1] = neiPt[1] - this->TileX[1];
      normal[2] = neiPt[2] - this->TileX[2];
    }
    else
    {
      origin[0] = (this->TileX[0] + neiPt[0]) / 2.0;
      origin[1] = (this->TileX[1] + neiPt[1]) / 2.0;
      origin[2] = (this->TileX[2] + neiPt[2]) / 2.0;
      normal[0] = this->TileX[0] - neiPt[0];
      normal[1] = this->TileX[1] - neiPt[1];
      normal[2] = this->TileX[2] - neiPt[2];
      negate = true;
    }
    vtkMath::Normalize(normal);
    if (negate)
    {
      normal[0] = (-normal[0]);
      normal[1] = (-normal[1]);
      normal[2] = (-normal[2]);
    }

    // If an intersection occurs, swap the Voronoi polyhedra. Otherwise,
    // retain the current Voronoi polygon. Also update the circumflower
    // radius.
    bool intersects =
      this->VP0->IntersectWithPlane(origin,normal,neiPtId, this->VP1);
    if ( intersects )
    {
      // Forward pass the computed tolerance.
      this->VP1->CopyTolerance(this->VP0);
      std::swap(this->VP0, this->VP1);
    }

    return intersects;
  }

  // Get the circumflower**2 for thr the current polyhedron.
  double GetCircumFlower()
  {
    return this->VP0->R2;
  }

  // Determine whether the provided point is within the Voronoi circumflower.
  // Return true if it is; false otherwise.
  bool InCircumFlower(const double p[3])
  {
    double r2 = vtkMath::Distance2BetweenPoints(p,this->VP0->X);
    if ( r2 <= this->VP0->R2 )
    {
      return true;
    }

    // Point not in the circumflower
    return false;
  }

  // Determine whether the provided point is within the Voronoi flower
  // neighborhood metric (i.e., inside one of the petals). Return true if it
  // is; false otherwise.
  bool InFlower(const double p[3])
  {
    // Check against the flower petals
    for (auto pItr=this->VP0->Points.begin(); pItr != this->VP0->Points.end(); ++pItr)
    {
      double fr2 = pItr->R2;
      double r2 = vtkMath::Distance2BetweenPoints(p,pItr->X);
      if ( r2 <= fr2 )
      {
        return true;
      }
    }

    // Point not in the flower
    return false;
  }

  // Compute the Voronoi circumflower
  double ComputeCircumFlower()
  {
    return this->VP0->ComputeCircumFlower();
  }

  // Generate a 3D Voronoi hull by iterative clipping of the hull with nearby
  // points.  Termination of the clipping process occurs when the neighboring
  // points become "far enough" away from the generating point (i.e., the
  // Voronoi Flower error metric is satisfied).
  bool BuildHull(vtkIdList* pIds, vtkDoubleArray *radii2, const double* pts,
                 vtkIdType maxClips)
  {
    // Ensure there are clips to be performed.
    if ( maxClips <= 0 )
    {
      return true;
    }

    const double* v;
    vtkIdType& numClips=this->NumClips;
    vtkIdType ptId,  numPts=this->NPts;

    // Request neighboring points around the generating point in annular
    // shells. The shells are defined by an inner and outer radius
    // (min,max]. The requested points fall within the shell, with shell
    // radii r:(min<r<=max). The neighboring points are used to perform
    // half-space clipping of the Voronoi hull. (The original polyhedral hull
    // around the generating point is defined from the bounding box of the
    // domain.) The Voronoi Flower and Circumflower neighborhood metrics are
    // used to terminate the clipping process. The Voronoi Flower is the set
    // of all flower (i.e., Delaunay circumspheres) located at the Voronoi
    // hull vertices. The Circumflower is the sphere that bounds all petals
    // of the Voronoi Flower.
    constexpr int INIT_QUERY_SIZE = 16;
    constexpr int QUERY_SIZE = 8;
    double R2 = VTK_FLOAT_MAX;
    double annulusMin2 = 0.0;
    double annulusMax2 =
      this->Locator->FindNPointsInShell(INIT_QUERY_SIZE, this->TileX, pIds, radii2, annulusMin2);
    vtkIdType numPtIds, *pIdsPtr;
    double *radii2Ptr;
    bool initQuery=true;

    // Now add additional points until they are outside of the Voronoi
    // flower. For speed, we use the bounding Voronoi circumcircle to
    // determine whether points are outside of the flower. Note that in the
    // while() loop below, if the number of points pIds<=0, then all points
    // have been exhausted and the loop is exited.
    while ( (numPtIds = pIds->GetNumberOfIds()) > 0 && annulusMin2 <= R2 && numClips < maxClips )
    {
      pIdsPtr = pIds->GetPointer(0);
      radii2Ptr = radii2->GetPointer(0);
      for ( vtkIdType i=0; i < numPtIds && numClips < maxClips; ++i )
      {
        ptId = pIdsPtr[i];
        v = pts + 3 * ptId;
        // Only check CircumFlower and InFlower() after the first pass
        if ( initQuery || (radii2Ptr[i] <= R2 && this->InFlower(v)) )
        {
          if ( this->InsertPoint(ptId,v) )
          {
            numClips++;
          }
        }
      } // process all points in requested annulus

      // Okay start checking against the CircumFlower and Voronoi Flower
      initQuery = false;
      R2 = this->ComputeCircumFlower();

      // See if circumflower radius is less then radius of annulus request; if so, the
      // Voronoi hull has been formed.
      if ( R2 < annulusMax2 )
      {
        break;
      }

      // Grab the next ring / annulus of points
      annulusMin2 = annulusMax2;
      annulusMax2 = this->Locator->FindNPointsInShell(QUERY_SIZE, this->TileX, pIds, radii2, annulusMin2);
    } // while points still in Voronoi circumflower

    return true;
  } // BuildHull

  // Regenerate a 3D Voronoi hull by clipping the initial bounding box with
  // the points provided (from the adjacency/wheels & spokes structure). This
  // method is used to reconstruct a hull already computed, so the appropriate
  // data can be output from the VTK filter.
  bool RebuildHull(vtkIdType ptId, WheelsType& wheels, SpokeType& spokes,
                   const double* pts)
  {
    vtkIdType numSpokes;
    const Spoke *s = GetWheelSpokes(wheels,spokes,ptId,numSpokes);
    vtkIdType pId;
    const double* v;
    for ( vtkIdType i=0; i < numSpokes; ++i, ++s )
    {
      pId = s->NeiId;
      v = pts + 3 * pId;
      this->InsertPoint(pId, v);
    }

    return true;
  }

  // If pruning faces (and the associated spokes) is enabled, this method
  // will collapse faces of near zero area (relative to the connecting
  // spoke).  Return the number of faces pruned.
  int PruneFaces(double pruneTol2, const double *inPts)
  {
    return this->VP0->PruneFaces(pruneTol2,inPts);
  }

  // Add the adjacency information for the specified generating point ptId,
  // i.e., the neighboring points (or spokes) that created this hull. Place
  // the information into the local hull's wheels and spokes data structure,
  // and return a list of this hull's spokes and the number of spokes found.
  // Also keep track of the maximum number of points and faces created across
  // all hulls processed by this thread. Note: the spokes is a thread local
  // SpokeType vector, the returned list const Spoke* is ephemeral and only
  // valid until AddAdjacency() is called again (within the current thread).
  const Spoke*
  AddAdjacencyInformation(vtkIdType ptId, WheelsType& globalWheels,
                          SpokeType& localSpokes, int& numSpokes,
                          int& maxPoints, int& maxFaces)
  {
    return this->VP0->
      AddAdjacencyInformation(ptId,globalWheels,localSpokes,numSpokes,
                              maxPoints,maxFaces);
  }

  // Use this method to provide information about the output Voronoi or
  // Delaunay triangulation. This method is called at the end of processing a
  // hull; i.e., once the Voronoi polyhedron is computed. This method records
  // the number of points, number of cells and faces, and the size of the
  // connectivity lists representing the cells and faces. It adds this
  // information to the provided thread local vectors (for later
  // compositing).
  void AddVolumetricInformation(int outputType, vtkIdType ptId, const Spoke* spokes,
                                int numSpokes, DataInformationType& volInfo,
                                HullVertexType& localPts, TopoCoordType& localTups,
                                CellConnType& localConn)
  {
    this->VP0->AddVolumetricInformation(outputType, ptId, spokes, numSpokes,
                                        volInfo, localPts, localTups, localConn);
  }

  // Use this method to provide information about the output polygonal mesh
  // or surface net. This method is called at the end of processing the hull;
  // i.e., once the Voronoi polyhedron is computed. This method records the
  // number of points, number of faces/lines, and the size of the
  // connectivity list representing the faces/lines. It also adds cell
  // connectivity and generated points to the provided thread local data
  // vectors (for later compositing).
  void AddSurfaceInformation(int outputType, vtkIdType ptId, const Spoke* spokes,
                             int numSpokes, DataInformationType& surfInfo,
                             HullVertexType& localPts, TopoCoordType& localTups,
                             CellConnType& localConn)
  {
    this->VP0->AddSurfaceInformation(outputType,ptId,spokes,numSpokes,surfInfo,
                                     localPts, localTups, localConn);
  }

}; //VHull

//======= These structs and methods are for managing the threading of
//======= batches of data.

// Class to manage batches of points. This is used to improve threaded
// performance and reduce memory.
struct BatchManager
{
  vtkIdType Num; //Number of total items to process
  vtkIdType BatchSize; //The desired batch size (clamped by Num)
  vtkIdType NumBatches; //The total number of batches to process
  BatchManager(vtkIdType num, vtkIdType batchSize) : Num(num), BatchSize(batchSize)
  {
    this->NumBatches = static_cast<vtkIdType>(ceil(static_cast<double>(num) / batchSize));
  }
  BatchManager(const BatchManager& batchManager)
  {
    this->Num = batchManager.Num;
    this->BatchSize = batchManager.BatchSize;
    this->NumBatches = batchManager.NumBatches;
  }
  vtkIdType GetNumberOfBatches() const
  {
    return this->NumBatches;
  }
  vtkIdType GetBatchItemRange(vtkIdType batchNum, vtkIdType &startId, vtkIdType &endId) const
  {
    startId = batchNum * this->BatchSize;
    endId = startId + this->BatchSize;
    endId = (endId > this->Num ? this->Num : endId);
    return (endId - startId);
  }
}; //BatchManager

// This is used to track information about each batch that is processed.
struct BatchInfo
{
  vtkIdType Id; //the batch id
  BatchInfo(vtkIdType id) : Id(id) {}
};
// A list of batch information. Used by threads to record the
// batches they've processed.
using BatchInfoType = std::vector<BatchInfo>;

//======= These structs and methods are for capturing data within a
//======= thread.

// Track local data on a per-thread basis. In the data compositing / Reduce()
// method this information will be used to combine the data from each thread
// into the requested VTK filter outputs. Note that not all of this data is
// used (especially the vector types) depending on the output type requested.
struct LocalDataType
{
  int MaxPoints; //the maximum number of points in any hull
  int MaxFaces; //the maximum number of faces in any hull
  int NumPrunes; //total number of pruning operations
  BatchInfoType LocalBatches; //the list of batches of generating points processed by this thread
  SpokeType LocalSpokes;  //connecting edges/spokes for each hull
  HullVertexType LocalPoints; //coordinates defining the hull vertices
  TopoCoordType LocalTuples; //points represented in topological tuple space
  CellConnType LocalCellConn; //cell connectivity generated by this thread prior to compositing
  DataInformationType LocalDataInformation; //information about surface output
  VHull Hull; //computational 3D Voronoi hull algorithm
  vtkIdType ThreadId; //assign a thread id [0,NumThreadsUsed). Used later for processing.

  LocalDataType()
    : MaxPoints(0)
    , MaxFaces(0)
    , NumPrunes(0)
    , ThreadId(-1)
  {
    this->LocalBatches.reserve(2048);
    this->LocalSpokes.reserve(2048);
    this->LocalPoints.reserve(2048);
    this->LocalCellConn.reserve(2048);
  }
};

// Map thread local data to thread id. This may be used later as part of the
// data compositing / reduction process.
using ThreadMapType = std::vector<LocalDataType*>;

// Functor used to generate the adjacency (wheels and spokes) structure.
// It basically copies thread local data into a global vector.
// Execution is across the n threads used to compute the spokes--so this is
// a parallel generation of the wheels/spokes data structure.
struct ProduceSpokes
{
  const BatchManager &Batcher;
  const ThreadMapType& ThreadMap;
  const WheelsType& Wheels;
  SpokeType& Spokes;

  ProduceSpokes(const BatchManager &batcher, ThreadMapType &threadMap,
                WheelsType& wheels, SpokeType& spokes)
    : Batcher(batcher)
    , ThreadMap(threadMap)
    , Wheels(wheels)
    , Spokes(spokes)
  {
  }

  void operator()(vtkIdType threadId, vtkIdType endThreadId)
  {
    const BatchManager &batcher = this->Batcher;
    const WheelsType &wheels = this->Wheels;
    Spoke *spokes;
    vtkIdType i, numSpokes, ptId;

    // Now copy the spokes into the right spot
    for ( ; threadId < endThreadId; ++threadId )
    {
      LocalDataType& localData = *(this->ThreadMap[threadId]);

      // Loop over all batches in this thread
      auto spItr = localData.LocalSpokes.begin();
      vtkIdType ptId, endPtId;
      for ( auto& batchInfo : localData.LocalBatches )
      {
        batcher.GetBatchItemRange(batchInfo.Id, ptId, endPtId);

        // Loop over all contiguous spokes in this batch
        spokes = this->Spokes.data() + GetWheelOffset(wheels,ptId);
        for ( ; ptId < endPtId; ++ptId )
        {
          vtkIdType numSpokes = GetNumberOfSpokes(wheels,ptId);
          for ( auto i=0; i < numSpokes; ++i, ++spItr, ++spokes)
          {
            spokes->NeiId = spItr->NeiId;
            spokes->Classification = spItr->Classification;
          }
        } // for all contiguous points in this batch
      }   // for all batches
    }     // across all threads in this batch
  }
}; // ProduceSpokes

// The threaded core of the algorithm. It computes the wheel and spokes data
// structure, and gathers information about the tessellation. Later this
// information is used to produce output. This struct could be templated over
// point type, but due to numerical sensitivity we'll just process doubles
// for now.
struct VCore
{
  const BatchManager Batcher; //Controls processing of batches of generating points
  ThreadMapType ThreadMap; //Keep track of local thread data for later access
  vtkPoints *InPoints; //Input points as data array
  vtkIdType NPts; //The number of input (Voronoi hull generation) points
  const double *Points; //Input points as pointer
  vtkIntArray *RegionIds; //Optional region ids to control tessellation
  vtkStaticPointLocator *Locator; //Used to (quickly) find nearby points
  const vtkIdType* NetScalars; //optional input point scalars to produce a surface net
  double PaddedBounds[6]; //the expanded domain over which Voronoi is calculated
  double Bounds[6]; //locator bounds
  int Divisions[3]; //Internal parameters for computing
  double H[3]; //Locator bin spacing
  double Padding; //The padding distance**2 around the bounding box

  int OutputType; //The specified surface output type

  vtkIdType MaxClips; //Control the maximum number of half-space clips
  int NumThreadsUsed; //Keep track of the number of threads used

  // This are used to create the spokes and wheels adjacency graph used to
  // validate the tessllation and produce a Delaunay triangulation.
  int MaxPoints; //Maximum numper of points in a generated Voronoi hull
  int MaxFaces; //Maximum number of faces (i.e., spokes) in a generated Voronoi hull
  WheelsType Wheels; //Wheel/spokes data structure: offset array to spokes
  vtkIdType NumSpokes; //Total number of edges / spokes
  SpokeType Spokes; //Spokes / edges with classification
  bool PruneSpokes; //Indicate whether to prune small edges / spokes
  double PruneTol2; //Specify the square spoke prune tolerance
  vtkIdType NumPrunes; //If pruning is on, keep track of the number of prunes

  // Keep track of information for producing the filter output
  DataInformationType DataInfo;

  // Used for controlling filter abort and accessing filter information
  vtkVoronoi3D* Filter;

  // Control whether points are merged and/or smoothing is performed. Define
  // infrastructure for merging points.
  bool MergePoints;
  MergeTupleType MergeTuples; //temporary array for merging points
  MergeMapType MergeMap; //maps hull vertex point ids to merged point ids
  vtkIdType NumMergedPts; //after merging, the number of points remaining

  // Storage local to each thread. We don't want to allocate working arrays
  // on every thread invocation. Thread local storage saves lots of
  // new/delete (e.g. the PIds).
  vtkSMPThreadLocalObject<vtkIdList> PIds;
  vtkSMPThreadLocalObject<vtkDoubleArray> Radii2;
  vtkSMPThreadLocal<LocalDataType> LocalData;

  VCore(BatchManager &batcher, vtkPoints *inPts, vtkIntArray* regionIds,
        vtkStaticPointLocator* loc, double padding, int outputType,
        vtkIdType maxClips, vtkVoronoi3D* filter)
    : Batcher(batcher)
    , InPoints(inPts)
    , RegionIds(regionIds)
    , Locator(loc)
    , Padding(padding)
    , OutputType(outputType)
    , MaxClips(maxClips)
    , NumThreadsUsed(0)
    , MaxPoints(0)
    , MaxFaces(0)
    , NumSpokes(0)
    , NumPrunes(0)
    , Filter(filter)
  {
    // Set up points for processing
    this->NPts = this->InPoints->GetNumberOfPoints();
    this->Points = vtkDoubleArray::FastDownCast(inPts->GetData())->GetPointer(0);

    // Wheels: one per input generating point. This is transformed
    // into a vector of offsets into the spokes vector for later processing,
    // hence the +1 allocation.
    this->Wheels.resize(this->NPts+1,0); //initialized to zero

    // Information: depending on output, keep track of output size etc.
    // These information vectors are transformed into offsets for
    // later processing.
    this->DataInfo.resize(this->NPts+1);

    // Pre-compute some local data in preparation for processing.
    loc->GetBounds(this->Bounds);
    loc->GetDivisions(this->Divisions);
    this->H[0] = (this->Bounds[1] - this->Bounds[0]) / static_cast<double>(this->Divisions[0]);
    this->H[1] = (this->Bounds[3] - this->Bounds[2]) / static_cast<double>(this->Divisions[1]);
    this->H[2] = (this->Bounds[5] - this->Bounds[4]) / static_cast<double>(this->Divisions[2]);

    // Define the Voronoi domain by padding out from bounds.
    for (int i = 0; i < 3; ++i)
    {
      this->PaddedBounds[2 * i] = this->Bounds[2 * i] - padding;
      this->PaddedBounds[2 * i + 1] = this->Bounds[2 * i + 1] + padding;
    }

    // Control spoke pruning (used if requested)
    this->PruneSpokes = filter->GetPruneSpokes();
    double pruneTol = filter->GetPruneTolerance();
    this->PruneTol2 = pruneTol * pruneTol;

    this->MergePoints = this->Filter->GetMergePoints();
    this->NumMergedPts = 0;
  }

  // Given two wheel ids, determine whether a valid edge connects the wheels.
  // This assumes that the wheel&spokes data structure have been built.
  bool IsValidSpoke(vtkIdType w0Id, vtkIdType w1Id)
  {
    vtkIdType numSpokes = GetNumberOfSpokes(this->Wheels,w0Id);
    Spoke* spokes = &(this->Spokes[this->Wheels[w0Id]]);

    for (vtkIdType i=0; i < numSpokes; ++i, ++spokes)
    {
      if (spokes->NeiId == w1Id && spokes->Classification == Spoke::VALID)
      {
        return true;
      }
    } // for all spokes in this wheel
    return false;
  }

  // Allocate a little bit of memory to get started. Set some initial values
  // for each thread for accelerating computation.
  void Initialize()
  {
    vtkIdList*& pIds = this->PIds.Local();
    pIds->Allocate(128); // allocate some memory

    vtkDoubleArray*& radii2 = this->Radii2.Local();
    radii2->Allocate(128); // allocate some memory

    // Populate the instance of the VHull class
    LocalDataType& localData = this->LocalData.Local();
    localData.Hull.NPts = this->NPts;
    localData.Hull.Points = this->Points;
    localData.Hull.SetRegionIds(this->RegionIds);
    localData.Hull.Locator = this->Locator;
    localData.Hull.Divisions[0] = this->Divisions[0];
    localData.Hull.Divisions[1] = this->Divisions[1];
    localData.Hull.Divisions[2] = this->Divisions[2];
    localData.Hull.PaddedBounds[0] = this->PaddedBounds[0];
    localData.Hull.PaddedBounds[1] = this->PaddedBounds[1];
    localData.Hull.PaddedBounds[2] = this->PaddedBounds[2];
    localData.Hull.PaddedBounds[3] = this->PaddedBounds[3];
    localData.Hull.PaddedBounds[4] = this->PaddedBounds[4];
    localData.Hull.PaddedBounds[5] = this->PaddedBounds[5];
    localData.Hull.Bounds[0] = this->Bounds[0];
    localData.Hull.Bounds[1] = this->Bounds[1];
    localData.Hull.Bounds[2] = this->Bounds[2];
    localData.Hull.Bounds[3] = this->Bounds[3];
    localData.Hull.Bounds[4] = this->Bounds[4];
    localData.Hull.Bounds[5] = this->Bounds[5];
    localData.Hull.H[0] = this->H[0];
    localData.Hull.H[1] = this->H[1];
    localData.Hull.H[2] = this->H[2];
    localData.Hull.Padding2 = this->Padding * this->Padding;
  }

  void operator()(vtkIdType batchId, vtkIdType endBatchId)
  {
    const BatchManager &batcher = this->Batcher;
    vtkIntArray *regionIds = this->RegionIds;
    vtkIdList*& pIds = this->PIds.Local();
    vtkDoubleArray*& radii2 = this->Radii2.Local();
    LocalDataType& localData = this->LocalData.Local();
    HullVertexType& localPts = localData.LocalPoints;
    TopoCoordType& localTups = localData.LocalTuples;
    CellConnType& localConn = localData.LocalCellConn;
    int &maxPoints = localData.MaxPoints;
    int &maxFaces = localData.MaxFaces;
    int &numPrunes = localData.NumPrunes;
    WheelsType& wheels = this->Wheels;
    BatchInfoType& lBatches = localData.LocalBatches;
    SpokeType& lSpokes = localData.LocalSpokes;
    DataInformationType& info = this->DataInfo;
    VHull& hull = localData.Hull;
    bool isFirst = vtkSMPTools::GetSingleThread();
    vtkIdType checkAbortInterval = std::min((endBatchId - batchId) / 10 + 1, (vtkIdType)1000);

    // Process the hullgenerating points in batches. This performs
    // a little better than independent point-by-point processing, and saves
    // some memory as well.
    for (; batchId < endBatchId; ++batchId)
    {
      // Support algorithm interrupts
      if (batchId % checkAbortInterval == 0)
      {
        if (isFirst)
        {
          this->Filter->CheckAbort();
        }
        if (this->Filter->GetAbortOutput())
        {
          break;
        }
      }

      // Process all points in this batch. Record the batch being
      // processed. Remember that the point ids are contiguous in this
      // batch.
      vtkIdType ptId, endPtId;
      batcher.GetBatchItemRange(batchId, ptId, endPtId);
      const double* x = this->Points + 3 * ptId;
      vtkIdType totalHullPts=0;

      for (; ptId < endPtId; ++ptId, x += 3)
      {
        // If the generating point is an outside region, we do not need to
        // process this hull.
        if ( regionIds && regionIds->GetValue(ptId) < 0 )
        {
          continue;
        }

        // Initialize the Voronoi hull
        hull.Initialize(ptId, x);

        // If hull is successfully built, copy the convex hull polyhedron and
        // related information to thread local storage.
        int maxClips = ( this->MaxClips < this->NPts ? this->MaxClips :
                         (this->NPts > 1 ? (this->NPts-1) : 0) );
        if (hull.BuildHull(pIds, radii2, this->Points, maxClips))
        {
          // Prune faces if requested
          if ( this->PruneSpokes )
          {
            numPrunes += hull.PruneFaces(this->PruneTol2, this->Points);
          }

          // Now accumulate the hull-related information in this thread.
          // First gather the neighborhood points (i.e., spokes) that generated this hull.
          // Also classify the spokes - this will be useful later when adding surface and
          // volumetric information.
          int numSpokes;
          const Spoke* spokes = hull.AddAdjacencyInformation(ptId,wheels,lSpokes,numSpokes,
                                                             maxPoints,maxFaces);

          // Accumulate volumetric information only if necessary
          if ( this->OutputType == vtkVoronoi3D::VORONOI ||
               this->OutputType == vtkVoronoi3D::DELAUNAY )
          {
            hull.AddVolumetricInformation(this->OutputType,ptId,spokes,numSpokes,
                                          info, localPts, localTups, localConn);
          }

          // Accumulate surface information only if necessary
          else
          {
            hull.AddSurfaceInformation(this->OutputType,ptId,spokes,numSpokes,
                                       info, localPts, localTups, localConn);
          }

        } // if hull successfully generated
      } // for all points in this batch
      lBatches.emplace_back(BatchInfo(batchId));
    } // for all batches of points
  }

  void Reduce()
  {
    // Build the wheels and spokes adjacency information.
    this->NumThreadsUsed = 0;
    this->NumSpokes = 0;
    this->MaxPoints = 0;
    this->MaxFaces = 0;
    this->NumPrunes = 0;

    // Gather information along with a prefix sum of some information
    // across all the threads.
    for ( auto& localData : this->LocalData)
    {
      this->ThreadMap.push_back(&localData);
      this->NumSpokes += localData.LocalSpokes.size();
      this->NumThreadsUsed++;
      this->MaxPoints = ( localData.MaxPoints > this->MaxPoints ?
                          localData.MaxPoints : this->MaxPoints);
      this->MaxFaces = ( localData.MaxFaces > this->MaxFaces ?
                         localData.MaxFaces : this->MaxFaces);
      this->NumPrunes += localData.NumPrunes;
    } // loop over local thread output

    // Prefix sum over wheels to determine spokes offsets,
    // as well total number of spokes.
    vtkIdType offset, totalSpokes = 0;
    for ( vtkIdType id=0; id < this->NPts; ++id )
    {
      offset = this->Wheels[id];
      this->Wheels[id] = totalSpokes;
      totalSpokes += offset;
    }
    // Cap off the wheels offsets array.
    this->Wheels[this->NPts] = totalSpokes;

    // Now copy the spokes from local thread data to the global
    // spokes array.
    this->Spokes.resize(this->NumSpokes);

    // Parallel build the adjacency (wheel and spokes) structure.
    ProduceSpokes produceSpokes(this->Batcher,this->ThreadMap,this->Wheels,this->Spokes);
    vtkSMPTools::For(0,this->NumThreadsUsed, produceSpokes);

    // Now build offsets/prefix sum for the volumetric and surface output (if
    // needed). The prefix sum converts hull information into offsets for
    // random access of information.
    DataInformation info, totalInfo;
    for ( vtkIdType id=0; id < this->NPts; ++id )
    {
      info = this->DataInfo[id];
      this->DataInfo[id] = totalInfo;
      totalInfo += info;
    }
    this->DataInfo[this->NPts] = totalInfo;

    // If point merging is requested (and required by the output type),
    // composite the topological point 4-tuples, sort them, and then create a
    // point renumbering map. We do this prior to creating output points and
    // primitives so that when they are created, we can renumber them
    // appropriately.
    if ( this->MergePoints && this->OutputType != vtkVoronoi3D::DELAUNAY &&
         this->OutputType != vtkVoronoi3D::ADJACENCY_GRAPH )
    {
      VCore::ProduceMergingInfo pmi(this);
      vtkSMPTools::For(0,this->NumThreadsUsed, pmi);
    }
  } // Reduce

  // If smoothing or merging, build the data structures to support this.
  // We use the topological 4-tuples to merge topologically coincident points.
  // Note that the order in which the tuples are processed here is implicitly
  // related to the subsequent compositing of Voronoi hull points. That is,
  // topological tuples ti corresponds to Voronoi hull vertices vi.
  struct ProduceMergingInfo
  {
    VCore* VC;

    ProduceMergingInfo(VCore* vc)
      : VC(vc)
    {
      // Use the surface information to allocate the merging array.
      const DataInformationType& info = vc->DataInfo;
      vtkIdType totalPts = info[vc->NPts].NumPts;
      vc->MergeTuples.resize(totalPts);
    }

    void Initialize() {}

    void operator()(vtkIdType threadId, vtkIdType endThreadId)
    {
      const BatchManager &batcher = this->VC->Batcher;
      DataInformationType& info = this->VC->DataInfo;
      MergeTupleType& mergeTuples = this->VC->MergeTuples;

      for (; threadId < endThreadId; ++threadId)
      {
        // Get the current local thread data. Also get indices into
        // the local data
        LocalDataType& localData = *(this->VC->ThreadMap[threadId]);
        HullVertexType::iterator pItr = localData.LocalPoints.begin();
        TopoCoordType::iterator tItr = localData.LocalTuples.begin();
        CellConnType::iterator cItr = localData.LocalCellConn.begin();

        // Loop over the batches that the current thread processed earlier. The batches
        // are ordered and consistent with the local data.
        for ( auto& batchInfo : localData.LocalBatches )
        {
          vtkIdType ptId, endPtId;
          vtkIdType numBatchPts = batcher.GetBatchItemRange(batchInfo.Id, ptId, endPtId);
          for (; ptId < endPtId; ++ptId) //produce tuples for this batch
          {
            vtkIdType numPts = (info[ptId+1].NumPts - info[ptId].NumPts);
            if ( numPts > 0 ) // process if tuples were produced by this point generator
            {
              vtkIdType startPtId = info[ptId].NumPts;
              vtkIdType pId = startPtId;
              for (int i=0; i<numPts; ++i, ++pId, ++tItr)
              {
                mergeTuples[pId].Ids = tItr->Ids; //composite 4-tuple
                mergeTuples[pId].HullVId = pId; //assign a global, unmerged point id
              }
            }
          } // for all points in this batch
        }   // for all batches
      }     // for all threads
    }

    // Now create the point renumbering map. It maps from the hull vertices
    // (which are disconnected and coincident) to the topologically merged
    // output points.
    void Reduce()
    {
      // Make sure there is data
      MergeTupleType& mergeTuples = this->VC->MergeTuples;
      if ( mergeTuples.size() <= 0 )
      {
        return;
      }

      // First we sort the tuples. This will create groups of 4-tuples with
      // the same tuple values. Each group is assigned an id (using a prefix
      // sum) to create the final point map.
      vtkSMPTools::Sort(mergeTuples.begin(),mergeTuples.end());

      // Count the number of merged points. Merged points have the same
      // hull vertex tuple ids.
      vtkIdType numMergedPts = 1;
      MergeTuple currentMT = mergeTuples[0];
      for (vtkIdType i=1; i < mergeTuples.size(); ++i)
      {
        if ( currentMT != mergeTuples[i] )
        {
          numMergedPts++;
          currentMT = mergeTuples[i];
        }
      } // for all hull vertex merge tuples

      // Update the total number of points in the output.
      this->VC->NumMergedPts = numMergedPts;

      // Allocate merge map, and populate it with the merged point ids.
      MergeMapType& mergeMap = this->VC->MergeMap;
      mergeMap.resize(mergeTuples.size());

      // Traverse the hull vertex tuples again and record the merged point
      // id of each tuple.
      vtkIdType currentMergeId = 0;
      currentMT = mergeTuples[0];
      for (vtkIdType i=0; i < mergeTuples.size(); ++i)
      {
        if ( currentMT != mergeTuples[i] )
        {
          ++currentMergeId;
          currentMT = mergeTuples[i];
        }
        mergeMap[mergeTuples[i].HullVId] = currentMergeId;
      } // for all hull vertex merge tuples
    } // Reduce

  }; // ProduceMergingInfo

  // A factory method to conveniently instantiate and execute the algorithm core.
  static std::unique_ptr<VCore>
  Execute(vtkStaticPointLocator *loc, vtkPoints *inPts,
          vtkIntArray *regionIds, double padding, int outputType,
          vtkIdType maxClips, int &maxPoints, int &maxFaces,
          int &numThreads, int &numPrunes, vtkVoronoi3D* filter)
  {
    // Set up batch processing and process the data. This
    BatchManager batcher(inPts->GetNumberOfPoints(),filter->GetBatchSize());

    // Generate the Voronoi tessellation. The core algorithm contains output
    // information used later.
    auto vc = std::make_unique<VCore>(batcher, inPts, regionIds, loc,
                                      padding, outputType, maxClips, filter);

    // Threaded process batches of points.
    vtkSMPTools::For(0,batcher.GetNumberOfBatches(), *vc);

    // Update some global information.
    maxPoints = vc->MaxPoints;
    maxFaces = vc->MaxFaces;
    numThreads = vc->NumThreadsUsed;
    numPrunes = vc->NumPrunes;

    // Okay get out
    return vc;
  }
}; // VCore

// Gather spokes into a wheel. Define some basic operators.  Note that every
// wheel is associated with an input (tile generating) point. So access to
// the wheel and its associated spokes is via point id.
struct Wheel
{
  vtkIdType Id; // The associated point/tile id: so wheelId == pointId
  int NumSpokes; // The number of emanating spokes
  Spoke* Spokes; // A pointer to an ordered array of spokes connected to this wheel

  // Default instantiation.
  Wheel() : Id(0), NumSpokes(0), Spokes(nullptr) {}
  // Instantiate a wheel given a point id.
  Wheel(VCore *vc, vtkIdType id) : Id(id)
  {
    this->NumSpokes = GetNumberOfSpokes(vc->Wheels,id);
    this->Spokes = &vc->Spokes[vc->Wheels[id]];
  }
  // Setup the wheel for queries: an efficient form that does not require
  // wheel instantiation.
  void Initialize(VCore *vc, vtkIdType id)
  {
    this->Id = id;
    this->NumSpokes = GetNumberOfSpokes(vc->Wheels,id);
    this->Spokes = &vc->Spokes[vc->Wheels[id]];
  }
}; // Wheel

// Functor to check necessary conditions for valid adjacency graph.
struct ValidateAdjacencyGraph
{
  VCore *VC;
  vtkIdType NumNonBidirectional;
  vtkIdType NumSingletons;
  bool AllValid;
  ValidateAdjacencyGraph(VCore *vc) : VC(vc), AllValid(false) {}

  // Keep track whether threads are non-degenerate.
  vtkSMPThreadLocal<vtkIdType> NonBidirectional;
  vtkSMPThreadLocal<vtkIdType> Singletons;
  vtkSMPThreadLocal<unsigned char> ThreadAllValid;

  // Determine whether a spoke is used by both wheels. Proper edges must
  // be bidirectional and not extend into the boundary. Singleton edges are
  // problematic as well (when building loops later).
  bool IsBiDirectional(vtkIdType wheelId, Wheel& neighborWheel)
  {
    Spoke *spoke = neighborWheel.Spokes;
    for (int i=0; i < neighborWheel.NumSpokes; ++i, ++spoke)
    {
      if ( spoke->NeiId == wheelId )
      {
        return true;
      }
    }
    return false;
  }

  void Initialize()
  {
    this->NonBidirectional.Local() = 0;
    this->Singletons.Local() = 0;
    this->ThreadAllValid.Local() = 1;
  }

  void  operator()(vtkIdType wheelId, vtkIdType endWheelId)
  {
    int spokeNum;
    Wheel wheel, neighborWheel;
    Spoke *spoke;

    // At this point edges should be classified either VALID or
    // BOUNDARY. We only need to examine the VALID edges.
    for ( ; wheelId < endWheelId; ++wheelId )
    {
      wheel.Initialize(this->VC,wheelId);
      spoke = wheel.Spokes;
      for (spokeNum=0; spokeNum < wheel.NumSpokes; ++spokeNum, ++spoke)
      {
        // Make sure to visit this spoke only once when (v0<v1).
        if ( wheelId < spoke->NeiId && (spoke->Classification & Spoke::VALID) )
        {
          neighborWheel.Initialize(this->VC,spoke->NeiId);
          // Check for unidirectional degeneracy or singleton edge.
          if ( ! this->IsBiDirectional(wheelId, neighborWheel) )
          { // Bidirectional connected if (v0->v1) and (v1->v0)
            this->NonBidirectional.Local()++;
            this->ThreadAllValid.Local() = 0;
          }
          else if ( wheel.NumSpokes == 1 || neighborWheel.NumSpokes == 1)
          {
            this->Singletons.Local()++;
            this->ThreadAllValid.Local() = 0;
          }
        } // process valid edge only once
      }   // over all spokes for this wheel
    }     // for all wheels
  }

  // Roll up validation
  void Reduce()
  {
    this->AllValid = 1;
    for (auto& localValid : this->ThreadAllValid)
    {
      if ( localValid == 0 )
      {
        this->AllValid = 0;
        break;
      }
    }

    this->NumNonBidirectional = 0;
    for (auto& localNonBi : this->NonBidirectional)
    {
      this->NumNonBidirectional += localNonBi;
    }

    this->NumSingletons = 0;
    for (auto& localSingletons : this->Singletons)
    {
      this->NumSingletons += localSingletons;
    }
  }

  static void Execute(VCore *vc)
  {
    ValidateAdjacencyGraph validate(vc);
    vtkSMPTools::For(0, vc->NPts, validate);
    cout << "All Valid: " << (validate.AllValid ? "True\n" : "False\n");
    cout << "Not Bidirectional: " << validate.NumNonBidirectional << "\n";
    cout << "Num Prunes: " << vc->NumPrunes << "\n";
  }

}; // ValidateAdjacencyGraph

// Superclass for classes that produce VTK output. Note that the output
// classes must be consistent with the information gathered previously or
// memory issues will result.
struct VOutput
{
  const VCore* VC;
  double* OutPoints;
  vtkIdType* CellConn;
  vtkIdType* CellOffsets;

  int GeneratePointScalars;
  double* PointScalars;
  int GenerateCellScalars;
  vtkIdType* CellScalars;

  // Used for merging points. Only write points one time,
  std::unique_ptr<PtsWrittenFlags> PtsWritten;

  // Optionally generate random numbers for cell scalars.
  vtkSMPThreadLocalObject<vtkMinimalStandardRandomSequence> LocalGenerator;

  VOutput(const VCore* vc, int genPtScalars, int genCellScalars)
    : VC(vc)
    , OutPoints(nullptr)
    , CellConn(nullptr)
    , CellOffsets(nullptr)
    , GeneratePointScalars(genPtScalars)
    , PointScalars(nullptr)
    , GenerateCellScalars(genCellScalars)
    , CellScalars(nullptr)
  {
    // Allocate some point merging related structure if necessary.
    if ( vc->MergePoints && vc->OutputType != vtkVoronoi3D::DELAUNAY &&
         vc->OutputType != vtkVoronoi3D::ADJACENCY_GRAPH)
    {
      this->PtsWritten = std::unique_ptr<PtsWrittenFlags>
        (new PtsWrittenFlags(vc->NumMergedPts,0));
    }
  }

  // Add a point to the output
  void AddPoint(vtkIdType ptId, double *x)
  {
    double *p = this->OutPoints + 3*ptId;
    *p++ = x[0];
    *p++ = x[1];
    *p++ = x[2];
  }

  // Add a merged point to the output. We just write
  // the value of the first vertex hull point - it's possible
  // to average these conincident points - maybe if necessary.
  void AddMergedPoint(const MergeMapType& mergeMap,
                      PtsWrittenFlags& ptsWritten,
                      vtkIdType ptId, double *x)
  {
    vtkIdType pId = mergeMap[ptId];
    if ( !ptsWritten[pId] )
    {
      double *p = this->OutPoints + 3*pId;
      *p++ = x[0];
      *p++ = x[1];
      *p++ = x[2];
      ptsWritten[pId] = 1;
    }
  }

  // Add a primitive cell to the output. This should be
  // followed by AddPrimPoint() calls.
  void AddPrim(vtkIdType primId, vtkIdType connOffset)
  {
    this->CellOffsets[primId] = connOffset;
  }

  // Add a primitive cell point to the output
  void AddPrimPoint(vtkIdType connOffset, vtkIdType pId)
  {
    this->CellConn[connOffset] = pId;
  }

  // Add a merged primitive cell point to the output
  void AddMergedPrimPoint(const MergeMapType& mergeMap,
                          vtkIdType connOffset, vtkIdType ptId)
  {
    vtkIdType pId = mergeMap[ptId];
    this->CellConn[connOffset] = pId;
  }

  // Produce a point attribute scalar
  double ProducePointScalar(vtkIdType ptId, double hullVertX[3])
  {
    const double *generatorX = this->VC->Points + 3*ptId;
    return (std::sqrt(vtkMath::Distance2BetweenPoints(generatorX,hullVertX)));
  }

  // Produce a cell attribute scalar
  vtkIdType ProduceCellScalar(vtkIdType ptId, vtkIdType numSpokes, vtkIdType primId,
                              vtkIdType threadId, bool& firstRandomScalar)
  {
    vtkIdType s=0;
    switch (this->GenerateCellScalars)
    {
      case vtkVoronoi3D::POINT_IDS:
        s = ptId;
        break;
      case vtkVoronoi3D::REGION_IDS:
        s = this->VC->RegionIds->GetValue(ptId);
        break;
      case vtkVoronoi3D::NUMBER_FACES:
        s = numSpokes;
        break;
      case vtkVoronoi3D::PRIM_IDS:
        s = primId;
        break;
      case vtkVoronoi3D::THREAD_IDS:
        s = threadId;
        break;
      case vtkVoronoi3D::RANDOM:
        auto& localGen = this->LocalGenerator.Local();
        // Make this repeatable, seed based on prim id
        if ( firstRandomScalar )
        {
          localGen->Initialize(primId);
          firstRandomScalar = false;
        }
        s = static_cast<vtkIdType>(localGen->GetNextRangeValue(0,64));
        break;
    }
    return s;
  } // Produce cell scalars

  // Count the number of faces/spokes that are boundary faces to the Voronoi
  // tessellation (exterior boundary faces); or are boundary faces to this
  // polyhedron (interior boundary faces). This version examines the
  // spokes (as compared to the similar version found from the
  // VPolyhedron class).
  const Spoke* CountBoundaryFaces(const WheelsType& wheels, const SpokeType& spokes,
                                  vtkIdType ptId, vtkIdType& numSpokes,
                                  int& numInterior, int& numExterior)
  {
    numInterior=0, numExterior=0;
    const Spoke* spoke = GetWheelSpokes(wheels, spokes, ptId, numSpokes);

    return spoke;
  }
}; // VOutput

// Class responsible for generating output polydata.
struct SurfaceOutput : public VOutput
{
  vtkPolyData* Output;

  SurfaceOutput(const VCore* vc, vtkPolyData *output, int genPtScalars, int genCellScalars)
    : VOutput(vc, genPtScalars, genCellScalars)
    , Output(output) {}

  // Retrieve information for a specified hull. Invoke this after the prefix sum.
  void GetSurfaceInformation(vtkIdType ptId, vtkIdType& numPts, vtkIdType& numPrims,
                             vtkIdType& connSize, vtkIdType& startPtId,
                             vtkIdType& startPrimId, vtkIdType& startConn)
  {
    const VCore* vc = this->VC;
    numPts = (vc->DataInfo[ptId+1].NumPts - vc->DataInfo[ptId].NumPts);
    numPrims = (vc->DataInfo[ptId+1].NumCells - vc->DataInfo[ptId].NumCells);
    connSize = (vc->DataInfo[ptId+1].CellConnSize - vc->DataInfo[ptId].CellConnSize);
    startPtId = vc->DataInfo[ptId].NumPts;
    startPrimId = vc->DataInfo[ptId].NumCells;
    startConn = vc->DataInfo[ptId].CellConnSize;
  }

  // Produce polygonal output for the generating point specified.
  void ProduceSurfacePrims(vtkIdType threadId, vtkIdType ptId,
                           HullVertexType::iterator& pItr, CellConnType::iterator& cItr)
  {
    // Retrieve offset information
    const VCore* vc = this->VC;
    int outputType = vc->OutputType;
    vtkIdType numPts, numPrims, connSize;
    vtkIdType startPtId, startPrimId, startConn;
    this->GetSurfaceInformation(ptId, numPts, numPrims, connSize,
                                startPtId, startPrimId, startConn);

    // If nothing is to be produced, return. Note that in some cases
    // no new points are produced, but primitives are (e.g., adjacency graph).
    if ( numPrims <= 0 )
    {
      return;
    }

    // Point merging may be in effect
    bool merging = vc->MergePoints;
    const MergeMapType& mergeMap = vc->MergeMap;
    PtsWrittenFlags& ptsWritten = *(this->PtsWritten);

    // Assess the situation based on the adjacency matrix.
    vtkIdType numSpokes;
    const Spoke* spokes = GetWheelSpokes(vc->Wheels, vc->Spokes, ptId, numSpokes);

    //-----------------------------------------------
    // The adjacency graph is constructed from the wheel & spokes data structure.
    // To prevent adding duplicate edges, we only output forward edges.
    if ( outputType == vtkVoronoi3D::ADJACENCY_GRAPH ) //line prims
    {
      int numDomainBoundaryFaces, numRegionBoundaryFaces, numForwardFaces;
      CountFaces(spokes, numSpokes, numDomainBoundaryFaces,
                 numRegionBoundaryFaces, numForwardFaces);
      // Output the adjacency graph. Note that the input points are
      // reused, no new points are created.
      for (vtkIdType i=0; i < numSpokes; ++i)
      {
        // See if face/spoke is to be output. If so add a line segment
        // representing the spoke / adjacency graph edge.
        if ( spokes[i].Classification == Spoke::REGION_BOUNDARY ||
             spokes[i].Classification == Spoke::FORWARD_SPOKE )
        {
          this->AddPrim(startPrimId++,startConn);
          this->AddPrimPoint(startConn++,ptId);
          this->AddPrimPoint(startConn++,spokes[i].NeiId);
        }
      }
    }

    //-----------------------------------------------
    else //if any other surface type (polygon prims)
    {
      // Start by writing points, merged or unmerged
      vtkIdType pId = startPtId;
      if ( merging )
      {
        for (int i=0; i<numPts; ++i, ++pId, ++pItr)
        {
          this->AddMergedPoint(mergeMap, ptsWritten, pId, pItr->X);
          if ( this->PointScalars )
          { // currently only producing Voronoi flowers
            this->PointScalars[pId] = this->ProducePointScalar(ptId,pItr->X);
          }
        }
      }
      else
      {
        for (int i=0; i<numPts; ++i, ++pId, ++pItr)
        {
          this->AddPoint(pId, pItr->X);
          if ( this->PointScalars )
          { // currently only producing Voronoi flowers
            this->PointScalars[pId] = this->ProducePointScalar(ptId,pItr->X);
          }
        }
      }

      // Output the cell connectivity. Note that the cell point ids need to be
      // transformed into global point id space. Also output optional cell data.
      bool firstRandomScalar = true;
      vtkIdType primId = startPrimId;
      for (int i=0; i<numPrims; ++primId, ++i)
      {
        this->AddPrim(primId,startConn);
        vtkIdType numPrimPts = *cItr++;
        if ( merging )
        {
          for (int j=0; j < numPrimPts; ++j)
          {
            this->AddMergedPrimPoint(mergeMap, startConn++, startPtId + *cItr++);
          }
        }
        else
        {
          for (int j=0; j < numPrimPts; ++j)
          {
            this->AddPrimPoint(startConn++, startPtId + *cItr++);
          }
        }
        if ( this->CellScalars )
        {
          this->CellScalars[startPrimId+i] =
            this->ProduceCellScalar(ptId,numSpokes,primId,threadId,firstRandomScalar);
        } // if cell scalars
      }
    }
  }

  // Each thread transforms and writes its own data.
  void operator()(vtkIdType threadId, vtkIdType endThreadId)
  {
    const BatchManager &batcher = this->VC->Batcher;
    bool isFirst = vtkSMPTools::GetSingleThread();
    vtkIdType checkAbortInterval = std::min((endThreadId - threadId) / 10 + 1, (vtkIdType)1000);

    // Loop over all threads
    for (; threadId < endThreadId; ++threadId)
    {
      // Support algorithm interrupts
      if (threadId % checkAbortInterval == 0)
      {
        if (isFirst)
        {
          this->VC->Filter->CheckAbort();
        }
        if (this->VC->Filter->GetAbortOutput())
        {
          break;
        }
      }

      // Get the current local thread data. Also get indices into
      // the local data
      LocalDataType& localData = *(this->VC->ThreadMap[threadId]);
      HullVertexType::iterator pItr = localData.LocalPoints.begin();
      CellConnType::iterator cItr = localData.LocalCellConn.begin();

      // Loop over the batches that the current thread processed earlier. The batches
      // are ordered and consistent with the local data vectors.
      for ( auto& batchInfo : localData.LocalBatches )
      {
        vtkIdType ptId, endPtId;
        vtkIdType numBatchPts = batcher.GetBatchItemRange(batchInfo.Id, ptId, endPtId);
        for (; ptId < endPtId; ++ptId) //output all data in this batch
        {
          this->ProduceSurfacePrims(threadId, ptId, pItr, cItr);
        } // for all points in this batch
      }   // for all batches
    }     // for all threads
  }       // operator()

  // A factory method to instantiate and threaded execute an instance
  // of SurfaceOutput to produce polygonal output.
  static void Execute(const VCore* vc, vtkPolyData* output, int genPtScalars, int genCellScalars)
  {
    SurfaceOutput so(vc,output,genPtScalars,genCellScalars);

    // The global information
    const DataInformationType& info = vc->DataInfo;

    // Create the output dataset arrays and allocate them.  Construct the
    // polygonal cell output as requested.
    vtkNew<vtkIdTypeArray> conn;
    conn->SetNumberOfTuples(info[vc->NPts].CellConnSize);
    so.CellConn = conn->GetPointer(0);

    vtkNew<vtkIdTypeArray> offsets;
    offsets->SetNumberOfTuples(info[vc->NPts].NumCells+1);
    so.CellOffsets = offsets->GetPointer(0);
    so.CellOffsets[info[vc->NPts].NumCells] = info[vc->NPts].CellConnSize; //cap off the offsets array

    vtkNew<vtkCellArray> prims;
    prims->SetData(offsets,conn);

    if ( vc->OutputType == vtkVoronoi3D::ADJACENCY_GRAPH )
    {
      // No points are created, just edge primitives. Reuse the input points.
      output->SetPoints(vc->InPoints); //pass points through to output
      output->SetLines(prims);
    }
    else
    {
      // Create new points and polygonal primitives
      vtkNew<vtkPoints> outPts;
      outPts->SetDataTypeToDouble();
      if ( vc->MergePoints ) //point merging needed
      {
        outPts->SetNumberOfPoints(vc->NumMergedPts);
      }
      else // no point merging
      {
        outPts->SetNumberOfPoints(info[vc->NPts].NumPts);
      }
      so.OutPoints = vtkDoubleArray::FastDownCast(outPts->GetData())->GetPointer(0);

      output->SetPoints(outPts);
      output->SetPolys(prims);
    }

    // If auxiliary point scalars are to be generated, create the
    // scalars now.
    if ( so.GeneratePointScalars != vtkVoronoi3D::NO_POINT_SCALARS )
    {
      vtkNew<vtkDoubleArray> pointScalars;
      pointScalars->SetNumberOfComponents(1);
      pointScalars->SetName("Voronoi Point Scalars");
      pointScalars->SetNumberOfTuples(info[vc->NPts].NumPts);
      int idx = output->GetPointData()->AddArray(pointScalars);
      output->GetPointData()->SetActiveAttribute(idx, vtkDataSetAttributes::SCALARS);
      so.PointScalars = pointScalars->GetPointer(0);
    }

    // If auxiliary cell scalars are to be generated, create the
    // scalars now.
    if ( so.GenerateCellScalars != vtkVoronoi3D::NO_CELL_SCALARS &&
         (so.GenerateCellScalars != vtkVoronoi3D::REGION_IDS || vc->RegionIds) )
    {
      vtkNew<vtkIdTypeArray> cellScalars;
      cellScalars->SetNumberOfComponents(1);
      cellScalars->SetName("Voronoi Cell Scalars");
      cellScalars->SetNumberOfTuples(info[vc->NPts].NumCells);
      int idx = output->GetCellData()->AddArray(cellScalars);
      output->GetCellData()->SetActiveAttribute(idx, vtkDataSetAttributes::SCALARS);
      so.CellScalars = cellScalars->GetPointer(0);
    }

    // Now parallel thread the creation of the surface output.
    vtkSMPTools::For(0,vc->NumThreadsUsed, so);
  };
}; // SurfaceOutput

// Class responsible for outputting volumetric (unstructured grid) output
// data.
struct VolumetricOutput : public VOutput
{
  vtkUnstructuredGrid* Output;
  // Support for polyhedron
  vtkIdType* FaceConn; //polyhedral faces
  vtkIdType* FaceOffsets; //polyhedral face offsets
  vtkIdType* LocConn; //polyhedral face locations
  vtkIdType* LocOffsets; //polyhedral face location offsets

  VolumetricOutput(const VCore* vc, vtkUnstructuredGrid *output,
                   int genPtScalars, int genCellScalars)
    : VOutput(vc,genPtScalars,genCellScalars)
    , Output(output)
    , FaceConn(0)
    , FaceOffsets(0)
    , LocConn(0)
    , LocOffsets(0) {}

  // Retrieve volumetric information for a specified hull. Invoke this after the prefix sum.
  void GetVolumetricInformation(vtkIdType ptId, vtkIdType& numPts, vtkIdType& numCells,
                                vtkIdType& numFaces, vtkIdType& cellConnSize,
                                vtkIdType& faceConnSize, vtkIdType& startPtId,
                                vtkIdType& startCellId, vtkIdType& startFaceId,
                                vtkIdType& startCellConn, vtkIdType& startFaceConn)
  {
    const VCore* vc = this->VC;
    numPts = (vc->DataInfo[ptId+1].NumPts - vc->DataInfo[ptId].NumPts);
    numCells = (vc->DataInfo[ptId+1].NumCells - vc->DataInfo[ptId].NumCells);
    numFaces = (vc->DataInfo[ptId+1].NumFaces - vc->DataInfo[ptId].NumFaces);
    cellConnSize = (vc->DataInfo[ptId+1].CellConnSize - vc->DataInfo[ptId].CellConnSize);
    faceConnSize = (vc->DataInfo[ptId+1].FaceConnSize - vc->DataInfo[ptId].FaceConnSize);
    startPtId = vc->DataInfo[ptId].NumPts;
    startCellId = vc->DataInfo[ptId].NumCells;
    startFaceId = vc->DataInfo[ptId].NumFaces;
    startCellConn = vc->DataInfo[ptId].CellConnSize;
    startFaceConn = vc->DataInfo[ptId].FaceConnSize;
  }

  // Add a polyhedral face to the output. This should be followed by
  // AddFacePoint() calls.
  void AddFace(vtkIdType faceId, vtkIdType connOffset)
  {
    this->FaceOffsets[faceId] = connOffset;
  }

  // Add a polyhedral face point to the output
  void AddFacePoint(vtkIdType connOffset, vtkIdType pId)
  {
    this->FaceConn[connOffset] = pId;
  }

  // Add a polyhedral face point to the output
  void AddMergedFacePoint(const MergeMapType& mergeMap,
                          vtkIdType connOffset, vtkIdType ptId)
  {
    vtkIdType pId = mergeMap[ptId];
    this->FaceConn[connOffset] = pId;
  }

  // Produce unstructured grid output for the generating point specified.
  void ProduceVolumetricPrims(vtkIdType threadId, vtkIdType ptId,
                              HullVertexType::iterator& pItr,
                              TopoCoordType::iterator& tItr,
                              CellConnType::iterator& cItr)
  {
    // Retrieve offset information
    const VCore* vc = this->VC;
    int outputType = vc->OutputType;
    vtkIdType numPts, numCells, numFaces, cellConnSize, faceConnSize;
    vtkIdType startPtId, startCellId, startFaceId, startCellConn, startFaceConn;
    vtkIdType startFaceLocConn;
    this->GetVolumetricInformation(ptId, numPts, numCells, numFaces,
                                   cellConnSize, faceConnSize, startPtId,
                                   startCellId, startFaceId, startCellConn,
                                   startFaceConn);

    // If nothing is to be produced, return. Note that in some cases
    // no new points are produced, but cells are (e.g., Delaunay).
    if ( numCells <= 0 )
    {
      return;
    }

    // Retrieve the global adjacency graph
    vtkIdType numSpokes;
    const Spoke* spokes = GetWheelSpokes(vc->Wheels, vc->Spokes, ptId, numSpokes);

    //-----------------------------------------------
    if ( outputType == vtkVoronoi3D::DELAUNAY )
    { // populate output vtkUnstructuredGrid with tetrahedral cells
      bool firstRandomScalar = true;
      vtkIdType cellId = startCellId;
      for (int i=0; i<numCells; ++cellId, ++i, ++tItr)
      {
        this->AddPrim(cellId,startCellConn);
        for (int j=0; j < 4; ++j) //four points per tetra
        {
          this->AddPrimPoint(startCellConn++, tItr->Ids[j]);
        }
        if ( this->CellScalars )
        {
          this->CellScalars[startCellId+i] =
            this->ProduceCellScalar(ptId,numSpokes,cellId,threadId,firstRandomScalar);
        } // if cell scalars
      }
    }

    //-----------------------------------------------
    else //if ( outputType == vtkVoronoi3D::VORONOI )
    { // Populate output vtkUnstructuredGrid with vtkPolyhedron cells

      // Start by writing points, merged or unmerged
      bool merging = vc->MergePoints;
      const MergeMapType& mergeMap = vc->MergeMap;
      PtsWrittenFlags& ptsWritten = *(this->PtsWritten);

      // Start by adding the cell points
      vtkIdType pId = startPtId;
      if ( merging )
      {
        for (int i=0; i<numPts; ++i, ++pId, ++pItr)
        {
          this->AddMergedPoint(mergeMap, ptsWritten, pId, pItr->X);
          if ( this->PointScalars )
          { // currently only producing Voronoi flowers
            this->PointScalars[pId] = this->ProducePointScalar(ptId,pItr->X);
          }
        }
      }
      else
      {
        for (int i=0; i<numPts; ++i, ++pId, ++pItr)
        {
          this->AddPoint(pId, pItr->X);
          if ( this->PointScalars )
          { // currently only producing Voronoi flowers
            this->PointScalars[pId] = this->ProducePointScalar(ptId,pItr->X);
          }
        }
      }

      // Output the polyhedral cell connectivity. Note that the cell point
      // ids need to be transformed into global point id space. Also output
      // optional cell data. At most only one cell can be written. The
      // connectivity is just the points previously output.
      bool firstRandomScalar = true;
      vtkIdType cellId = startCellId;
      this->AddPrim(cellId,startCellConn);
      if ( merging )
      {
        for (int j=0; j < numPts; ++j)
        {
          this->AddMergedPrimPoint(mergeMap, startCellConn++, startPtId + j);
        }
      }
      else
      {
        for (int j=0; j < numPts; ++j)
        {
          this->AddPrimPoint(startCellConn++, startPtId + j);
        }
      }

      if ( this->CellScalars )
      {
        this->CellScalars[startCellId] =
          this->ProduceCellScalar(ptId,numSpokes,cellId,threadId,firstRandomScalar);
      } // if cell scalars

      // Update the face locations.
      this->LocOffsets[startCellId] = startFaceId;

      // Output the polyhedral faces.
      for (int fId=0; fId < numFaces; ++fId)
      {
        this->AddFace(startFaceId+fId,startFaceConn);
        vtkIdType numFacePts = *cItr++;
        if ( merging )
        {
          for (int j=0; j < numFacePts; ++j)
          {
            this->AddMergedFacePoint(mergeMap, startFaceConn++, startPtId + *cItr++);
          }
        }
        else
        {
          for (int j=0; j < numFacePts; ++j)
          {
            this->AddFacePoint(startFaceConn++, startPtId + *cItr++);
          }
        }
      }

    } // vtkPolyhedron output
  }   // ProduceVolumetricPrims

  // Each thread transforms and writes its own data.
  void operator()(vtkIdType threadId, vtkIdType endThreadId)
  {
    const BatchManager &batcher = this->VC->Batcher;
    bool isFirst = vtkSMPTools::GetSingleThread();
    vtkIdType checkAbortInterval = std::min((endThreadId - threadId) / 10 + 1, (vtkIdType)1000);

    // Loop over all threads
    for (; threadId < endThreadId; ++threadId)
    {
      // Support algorithm interrupts
      if (threadId % checkAbortInterval == 0)
      {
        if (isFirst)
        {
          this->VC->Filter->CheckAbort();
        }
        if (this->VC->Filter->GetAbortOutput())
        {
          break;
        }
      }

      // Get the current local thread data. Also get indices into
      // the local data
      LocalDataType& localData = *(this->VC->ThreadMap[threadId]);
      HullVertexType::iterator pItr = localData.LocalPoints.begin();
      TopoCoordType::iterator tItr = localData.LocalTuples.begin();
      CellConnType::iterator cItr = localData.LocalCellConn.begin();

      // Loop over the batches that the current thread processed earlier. The batches
      // are ordered and consistent with the local data vectors.
      for ( auto& batchInfo : localData.LocalBatches )
      {
        vtkIdType ptId, endPtId;
        vtkIdType numBatchPts = batcher.GetBatchItemRange(batchInfo.Id, ptId, endPtId);
        for (; ptId < endPtId; ++ptId) //output all data in this batch
        {
          this->ProduceVolumetricPrims(threadId, ptId, pItr, tItr, cItr);
        } // for all points in this batch
      }   // for all batches
    }     // for all threads
  }       // operator()

  // A factory method to instantiate and threaded execute an instance of
  // VolumetricOutput. This method populates the vtkUnstructuredGrid output.
  static void Execute(const VCore* vc, vtkUnstructuredGrid* output,
                      int genPtScalars, int genCellScalars)
  {
    VolumetricOutput vo(vc,output,genPtScalars,genCellScalars);

    // The global volume information
    const DataInformationType& info = vc->DataInfo;

    // Create the output dataset arrays and allocate them.
    // Construct the cell output. Depending on type of output,
    // other arrays are constructed as well.
    if ( vc->OutputType == vtkVoronoi3D::DELAUNAY )
    {
      // No points are created, just tetrahedral cells which reuse the input
      // points.
      output->SetPoints(vc->InPoints); //pass points through to output

      vtkNew<vtkIdTypeArray> cellConn;
      cellConn->SetNumberOfTuples(info[vc->NPts].CellConnSize);
      vo.CellConn = cellConn->GetPointer(0);

      vtkNew<vtkIdTypeArray> offsets;
      offsets->SetNumberOfTuples(info[vc->NPts].NumCells+1);
      vo.CellOffsets = offsets->GetPointer(0);
      vo.CellOffsets[info[vc->NPts].NumCells] = info[vc->NPts].CellConnSize; //cap off the offsets array

      vtkNew<vtkCellArray> tets;
      tets->SetData(offsets,cellConn);

      output->SetCells(VTK_TETRA,tets);
    }

    // Otherwise, polyhedral cells are created. Polyhedral cells are defined
    // by several arrays etc. that need to be pieced together.
    else // if ( vc->OutputType == vtkVoronoi3D::VORONOI )
    {
      // New points
      vtkNew<vtkPoints> outPts;
      outPts->SetDataTypeToDouble();
      if ( vc->MergePoints ) //point merging specified
      {
        outPts->SetNumberOfPoints(vc->NumMergedPts);
      }
      else // no point merging
      {
        outPts->SetNumberOfPoints(info[vc->NPts].NumPts);
      }
      vo.OutPoints = vtkDoubleArray::FastDownCast(outPts->GetData())->GetPointer(0);
      output->SetPoints(outPts);

      // Cell types - they are all polyhedra. This can be created here.
      vtkNew<vtkUnsignedCharArray> cellTypes;
      cellTypes->SetNumberOfTuples(info[vc->NPts].NumCells);
      vtkSMPTools::Fill(cellTypes->GetPointer(0),
                        cellTypes->GetPointer(0) + info[vc->NPts].NumCells,
                        static_cast<unsigned char>(VTK_POLYHEDRON));

      // Define polyhedral cells to be filled in later.
      vtkNew<vtkIdTypeArray> cellConn;
      cellConn->SetNumberOfTuples(info[vc->NPts].CellConnSize);
      vo.CellConn = cellConn->GetPointer(0);

      vtkNew<vtkIdTypeArray> offsets;
      offsets->SetNumberOfTuples(info[vc->NPts].NumCells+1);
      vo.CellOffsets = offsets->GetPointer(0);
      vo.CellOffsets[info[vc->NPts].NumCells] = info[vc->NPts].CellConnSize; //cap off the offsets array

      vtkNew<vtkCellArray> cells;
      cells->SetData(offsets,cellConn);

      // Polyhedral faces - each face of the polyhedra
      vtkNew<vtkIdTypeArray> faceConn;
      faceConn->SetNumberOfTuples(info[vc->NPts].FaceConnSize);
      vo.FaceConn = faceConn->GetPointer(0);

      vtkNew<vtkIdTypeArray> faceOffsets;
      faceOffsets->SetNumberOfTuples(info[vc->NPts].NumFaces+1);
      vo.FaceOffsets = faceOffsets->GetPointer(0);
      vo.FaceOffsets[info[vc->NPts].NumFaces] = info[vc->NPts].FaceConnSize;

      vtkNew<vtkCellArray> faces;
      faces->SetData(faceOffsets,faceConn);

      // The face locations are basically an enumeration of the face ids. We
      // can partially complete this cell array (using the lambda) here.
      // The offsets are filled in later.
      vtkNew<vtkIdTypeArray> locConn;
      locConn->SetNumberOfTuples(info[vc->NPts].NumFaces);
      vo.LocConn = locConn->GetPointer(0);
      vtkIdType pId=0;
      std::generate(vo.LocConn,vo.LocConn+info[vc->NPts].NumFaces,
        [&pId] {return pId++;});

      vtkNew<vtkIdTypeArray> locOffsets;
      locOffsets->SetNumberOfTuples(info[vc->NPts].NumCells+1);
      vo.LocOffsets = locOffsets->GetPointer(0);
      vo.LocOffsets[info[vc->NPts].NumCells] = info[vc->NPts].NumFaces;

      vtkNew<vtkCellArray> faceLocs;
      faceLocs->SetData(locOffsets,locConn);

      // Finally, assemble the output.
      output->SetPoints(outPts);
      output->SetPolyhedralCells(cellTypes, cells, faceLocs, faces);
    } // VORONOI output type

    // If auxiliary point scalars are to be generated, create the
    // scalars now.
    if ( vo.GeneratePointScalars != vtkVoronoi3D::NO_POINT_SCALARS )
    {
      vtkNew<vtkDoubleArray> pointScalars;
      pointScalars->SetNumberOfComponents(1);
      pointScalars->SetName("Voronoi Point Scalars");
      pointScalars->SetNumberOfTuples(info[vc->NPts].NumPts);
      int idx = output->GetPointData()->AddArray(pointScalars);
      output->GetPointData()->SetActiveAttribute(idx, vtkDataSetAttributes::SCALARS);
      vo.PointScalars = pointScalars->GetPointer(0);
    }

    // If auxiliary cell scalars are to be generated, create the
    // scalars now.
    if ( vo.GenerateCellScalars != vtkVoronoi3D::NO_CELL_SCALARS &&
         (vo.GenerateCellScalars != vtkVoronoi3D::REGION_IDS || vc->RegionIds) )
    {
      vtkNew<vtkIdTypeArray> cellScalars;
      cellScalars->SetNumberOfComponents(1);
      cellScalars->SetName("Voronoi Cell Scalars");
      cellScalars->SetNumberOfTuples(info[vc->NPts].NumCells);
      int idx = output->GetCellData()->AddArray(cellScalars);
      output->GetCellData()->SetActiveAttribute(idx, vtkDataSetAttributes::SCALARS);
      vo.CellScalars = cellScalars->GetPointer(0);
    }

    // Now parallel thread the creation of the volume output.
    vtkSMPTools::For(0,vc->NumThreadsUsed, vo);
  }

}; // VolumetricOutput

} // anonymous namespace


//================= Begin VTK class proper =====================================
//------------------------------------------------------------------------------
// Construct object
vtkVoronoi3D::vtkVoronoi3D()
{
  this->OutputType = vtkVoronoi3D::BOUNDARY;
  this->Padding = 0.001;
  this->Validate = false;
  this->Locator = vtkSmartPointer<vtkStaticPointLocator>::New();
  this->Locator->SetNumberOfPointsPerBucket(2);
  this->PassPointData = true;
  this->GeneratePointScalars = NO_POINT_SCALARS;
  this->GenerateCellScalars = POINT_IDS;
  this->MergePoints = true;

  this->PointOfInterest = (-1);
  // this->PointsOfInterest empty on instantiation
  this->MaximumNumberOfHullClips = VTK_ID_MAX;
  this->PruneSpokes = false;
  this->PruneTolerance = 0.000001;
  this->BatchSize = 1000;

  this->MaximumNumberOfPoints = 0;
  this->MaximumNumberOfFaces = 0;
  this->NumberOfThreadsUsed = 0;
  this->NumberOfPrunes = 0;

  // By default process active point scalars to obtain region ids
  this->SetInputArrayToProcess(
    0, 0, 0, vtkDataObject::FIELD_ASSOCIATION_POINTS, vtkDataSetAttributes::SCALARS);
}

//------------------------------------------------------------------------------
int vtkVoronoi3D::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector, vtkInformationVector* outputVector)
{
  // get the info objects
  vtkInformation* inInfo = inputVector[0]->GetInformationObject(0);
  vtkInformation* outInfo0 = outputVector->GetInformationObject(0);

  // get the input and output
  vtkPointSet* input = vtkPointSet::SafeDownCast(inInfo->Get(vtkDataObject::DATA_OBJECT()));
  vtkDataSet* output = vtkDataSet::SafeDownCast(outInfo0->Get(vtkDataObject::DATA_OBJECT()));

  // Cast to proper output type
  vtkUnstructuredGrid* volOutput = nullptr;
  vtkPolyData* surfOutput = nullptr;
  if ( this->OutputType == vtkVoronoi3D::VORONOI ||
       this->OutputType == vtkVoronoi3D::DELAUNAY )
  {
    volOutput = vtkUnstructuredGrid::SafeDownCast(output);
  }
  else
  {
    surfOutput = vtkPolyData::SafeDownCast(output);
  }

  vtkDebugMacro(<< "Generating 3D Voronoi Tessellation");

  // Check the input, at least one point is needed.
  vtkIdType pId, numPts;
  vtkPoints* inPoints;
  if ((inPoints = input->GetPoints()) == nullptr || (numPts = inPoints->GetNumberOfPoints()) < 1)
  {
    vtkDebugMacro("Cannot tessellate; need at least 1 input point");
    return 1;
  }

  // Input points must be of type double
  vtkSmartPointer<vtkPoints> tPoints;
  if (inPoints->GetDataType() == VTK_DOUBLE)
  { // fast path no conversion
    tPoints = inPoints;
  }
  else
  { // convert points to double
    tPoints = vtkSmartPointer<vtkPoints>::New();
    tPoints->SetDataTypeToDouble();
    tPoints->SetNumberOfPoints(numPts);
    for (pId = 0; pId < numPts; ++pId)
    {
      tPoints->SetPoint(pId, inPoints->GetPoint(pId));
    }
  }

  // Temporary data object holds points to be tessellated
  vtkNew<vtkPolyData> tInput;
  tInput->SetPoints(tPoints);

  // A locator is used to locate closest points.
  if (!this->Locator)
  {
    vtkErrorMacro(<< "Point locator required\n");
    return 0;
  }
  this->Locator->SetDataSet(tInput);
  this->Locator->BuildLocator();

  // Computational bounds and the padded bounding box
  double length = input->GetLength();
  double padding = this->Padding * length;

  // Region ids can be used to control which input points are processed.
  // A region id < 0 means that the associated point is "outside" and does
  // not contribute to the output. We can use this capability to process a
  // specified "PointOfInterest" (if any). Otherwise, we check the input for
  // segmented regions via a regions ids array.
  //
  // If region ids are provided,  array must be a single component tuple,
  // signed integer of type vtkIntArray with the number of tuples == number
  // of input points. (Implementation note: this could be expanded with
  // templates - not sure its worth the object bloat.)
  vtkSmartPointer<vtkIntArray> regionIds;

  // Limit processing to points of interested if so specified.
  if ( (this->PointOfInterest >= 0 && this->PointOfInterest < numPts) ||
       this->PointsOfInterest )
  {
    regionIds = vtkSmartPointer<vtkIntArray>::New();
    regionIds->SetName("Points of Interest");
    regionIds->SetNumberOfTuples(numPts);
    vtkSMPTools::Fill(regionIds->GetPointer(0), regionIds->GetPointer(0) + numPts, -100);
    if ( this->PointOfInterest >= 0 )
    {
      regionIds->SetValue(this->PointOfInterest,numPts); // mark POI in region numPts
    }
    if ( this->PointsOfInterest )
    {
      vtkIdType numPOI = this->PointsOfInterest->GetNumberOfTuples();
      for ( vtkIdType i=0; i < numPOI; ++i)
      {
        vtkIdType poi = this->PointsOfInterest->GetValue(i);
        if ( poi >= 0 && poi < numPts )
        {
          regionIds->SetValue(poi,numPts); // mark POI in region numPts
        }
      }
    }
  }
  else
  {
    vtkDataArray* rIds = this->GetInputArrayToProcess(0, inputVector);
    regionIds = vtkIntArray::FastDownCast(rIds);
    if ( rIds && !regionIds )
    {
      vtkErrorMacro("Region Ids array must be of type vtkIntArray");
    }
    if ( regionIds )
    {
      if ( regionIds->GetNumberOfComponents() > 1 )
      {
        vtkErrorMacro("Region Ids must have 1 component");
        regionIds = nullptr;
      }
    }
  }

  // Process the points to generate Voronoi information, including the adjaceny
  // (wheels and spokes) data structure. Information is also gathered to allocate
  // memory for the output, and then generate the VTK filter output.
  double* inPtr = vtkDoubleArray::FastDownCast(tPoints->GetData())->GetPointer(0);
  std::unique_ptr<VCore> vc = VCore::
    Execute(this->Locator, tPoints, regionIds, padding, this->OutputType,
            this->MaximumNumberOfHullClips, this->MaximumNumberOfPoints,
            this->MaximumNumberOfFaces, this->NumberOfThreadsUsed,
            this->NumberOfPrunes, this);

  // Optionally validate the adjacency graph (wheels/spokes)
  if ( this->Validate )
  {
    ValidateAdjacencyGraph::Execute(vc.get());
  }

  // With the information gathered, now build the output(s).

  // Let's see if volumetric vtkUnstructuredGrid output is to be created.
  if ( this->OutputType == vtkVoronoi3D::VORONOI ||
       this->OutputType == vtkVoronoi3D::DELAUNAY )
  {
    VolumetricOutput::Execute(vc.get(),volOutput,this->GeneratePointScalars,
                              this->GenerateCellScalars);
  }

  // Create surface vtkPolyData output
  else
  {
    SurfaceOutput::Execute(vc.get(),surfOutput,this->GeneratePointScalars,
                           this->GenerateCellScalars);
  }

  return 1;
}

//------------------------------------------------------------------------------
vtkIdType vtkVoronoi3D::FindHull(double x[3])
{
  // Make sure the filter has executed (i.e., a locator is available), and the
  // request is within the bounding box of the input points.
  if ( this->Locator == nullptr )
  {
    return (-1);
  }

  double bounds[6];
  this->Locator->GetBounds(bounds);
  if ( x[0] < bounds[0] || x[0] > bounds[1] ||
       x[1] < bounds[2] || x[1] > bounds[3] ||
       x[2] < bounds[4] || x[2] > bounds[5] )
  {
    return -1;
  }

  // Now simply request the closest point.
  return this->Locator->FindClosestPoint(x);
}

//------------------------------------------------------------------------------
int vtkVoronoi3D::FillInputPortInformation(int, vtkInformation* info)
{
  info->Set(vtkAlgorithm::INPUT_REQUIRED_DATA_TYPE(), "vtkPointSet");
  return 1;
}

//------------------------------------------------------------------------------
int vtkVoronoi3D::FillOutputPortInformation(int port, vtkInformation* info)
{
  if (this->OutputType == vtkVoronoi3D::VORONOI ||
      this->OutputType == vtkVoronoi3D::DELAUNAY )
  {
    info->Set(vtkDataObject::DATA_TYPE_NAME(), "vtkUnstructuredGrid");
  }
  else
  {
    info->Set(vtkDataObject::DATA_TYPE_NAME(), "vtkPolyData");
  }

  return 1;
}

//------------------------------------------------------------------------------
// Since users have access to the locator we need to take into account the
// locator's modified time.
vtkMTimeType vtkVoronoi3D::GetMTime()
{
  vtkMTimeType mTime = this->vtkObject::GetMTime();
  vtkMTimeType time = this->Locator->GetMTime();
  return (time > mTime ? time : mTime);
}

//------------------------------------------------------------------------------
void vtkVoronoi3D::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);

  os << indent << "Output Type: " << this->OutputType << "\n";
  os << indent << "Padding: " << this->Padding << "\n";
  os << indent << "Validate: " << (this->Validate ? "On\n" : "Off\n");
  os << indent << "Locator: " << this->Locator << "\n";
  os << indent << "Pass Point Data: " << (this->PassPointData ? "On\n" : "Off\n");
  os << indent << "Generate Point Scalars: " << this->GeneratePointScalars << "\n";
  os << indent << "Generate Cell Scalars: " << this->GenerateCellScalars << "\n";
  os << indent << "Merge Points: " << (this->MergePoints ? "On\n" : "Off\n");
  os << indent << "Point Of Interest: " << this->PointOfInterest << "\n";
  os << indent << "Points Of Interest: " << this->PointsOfInterest << "\n";
  os << indent << "Maximum Number Of Hull Clips: " << this->MaximumNumberOfHullClips << "\n";
  os << indent << "Prune Spokes: " << (this->PruneSpokes ? "On\n" : "Off\n");
  os << indent << "Prune Tolerance: " << this->PruneTolerance << "\n";
  os << indent << "Batch Size: " << this->BatchSize << "\n";
}

VTK_ABI_NAMESPACE_END
