/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkGeneralizedSurfaceNets3D.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkGeneralizedSurfaceNets3D.h"

#include "vtkCellArray.h"
#include "vtkCellData.h"
#include "vtkDoubleArray.h"
#include "vtkIdTypeArray.h"
#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkLabelMapLookup.h"
#include "vtkMath.h"
#include "vtkObjectFactory.h"
#include "vtkPlane.h"
#include "vtkPointData.h"
#include "vtkPointSet.h"
#include "vtkPolyData.h"
#include "vtkSMPThreadLocalObject.h"
#include "vtkSMPTools.h"
#include "vtkStaticPointLocator.h"
#include "vtkStreamingDemandDrivenPipeline.h"
#include "vtkUnstructuredGrid.h"

#include <vector>

VTK_ABI_NAMESPACE_BEGIN
vtkStandardNewMacro(vtkGeneralizedSurfaceNets3D);

namespace //anonymous
{

//======= These structs and methods are for topologically merging points.

// The struct TopoCoord is the coordinate of a Voronoi hull vertex/point
// represented in terms of a 4-tuple: the four neighboring Voronoi generators
// which produce the hull vertex. The components of the 4-tuple are sorted in
// ascending order. It just so happens that due to the dual relationship of
// Voronoi and Delaunay, the TopoCoord is a tetrahedron, and the x-y-z point
// corresponding to the vertex/point position is the circumcenter of the
// tetrahedron.
struct TopoCoord
{
  // Four points defining a topological coord tuple / Delaunay tet. std::array
  // has some nice properties including being easily sortable.
  std::array<vtkIdType,4> Ids;

  // Various flavors of constructors
  TopoCoord() : Ids{0} {}
  TopoCoord(vtkIdType ids[3], vtkIdType ptId)
    : Ids{ids[0],ids[1],ids[2],ptId}
  {
    std::sort(this->Ids.data(),this->Ids.data()+this->Ids.size());
  }

  // Copy constructor assumes that tuple ids are already sorted.
  TopoCoord(const TopoCoord& tt)
  {
    this->Ids = tt.Ids;
  }

  //  Operator< used to support a subsequent sort operation (used for uniquely
  // identifying and producing a Delaunay tetrahedron).
  bool operator<(const TopoCoord& tuple) const
  {
    return this->Ids < tuple.Ids;
  }
}; // TopoCoord

// The MergeTuple is used to keep track of the hull vertices that produce
// a corresponding TopoCoord. All hull vertices with the same TopoCoord are
// coincident.
struct MergeTuple : public TopoCoord
{
  vtkIdType HullVId; // the id of the hull vertex that produced this tuple

  MergeTuple() : HullVId(-1) {}
  bool operator!=(const MergeTuple& mt) const
  {
    return (this->Ids != mt.Ids);
  }
}; // MergeTuple

//======= These structs and methods are for creating and clipping polyhedron.
//------------------------------------------------------------------------------
// Internal data structures and methods to generate a 3D Voronoi
// tessellation. The filter operates by performing polyhedral clipping by
// processing only neighborhood points falling within a distance measure (the
// Voronoi flower).  The clipping operation is done independently in each
// thread, and then composited together to produce the filter output.
//
//------------------------------------------------------------------------------
// Implementation comment:
// The size of the polyhedron is assumed to be typically dozens (or fewer)
// faces.  Consequently the implementation is serial, and relatively
// simple. For very large polyhedra, many of the operations could be
// threaded.

// Represent a point of the Voronoi polyhedron. This includes the position,
// the evaluated value against the current cutting plane, and radius**2 of
// the distance of the point to the generating point (i.e., the Voronoi petal
// radius**2).  Note: the id of the point is implicitly known from its
// position in the PointsArray. Points can be marked (and mapped to
// a new point id) as being used in order to gather information about the
// points. The Faces[3] are the three faces (actually, the three nearby point
// generators) whose separation planes intersected to produce the point. If
// more than three faces meet at the point (i.e., the point is degenerate)
// then Faces[0] is set to a negative value.
struct VPoint
{
  double X[3]; //position
  double Val;  //value against cutting plane
  double R2;   //Voronoi petal radius
  vtkIdType PtMap; //keep track of points that are actually used by selected faces
  vtkIdType Faces[3]; //the three neighboring generating points defining this point
  bool Degenerate; //indicate whether the point is degenerate

  // methods to define new point
  VPoint(double x, double y, double z)
    : X{x,y,z}
    , Val(0)
    , R2(0.0)
    , PtMap(-1)
    , Faces{-1,-1,-1}
    , Degenerate(false)
  {
  }
  VPoint(double x[3])
    : X{x[0],x[1],x[2]}
    , Val(0)
    , R2(0.0)
    , PtMap(-1)
    , Faces{-1,-1,-1}
    , Degenerate(false)
  {
  }
  // Note that besides copying the x-y-z coordinate, the copy constructor
  // copies the R2 as we do not want to recalculate it. The using faces are
  // also copied. (These using faces are in effect a topological coordinate
  // which can be used to merge coincident points and/or produce a Delaunay
  // tetrahedralization.)
  VPoint(const VPoint& p)
    : X{p.X[0],p.X[1],p.X[2]}
    , Val(0)
    , R2(p.R2)
    , PtMap(-1)
    , Faces{p.Faces[0],p.Faces[1],p.Faces[2]}
    , Degenerate(p.Degenerate)
  {
  }
};  // VPoint

// Represent a polyhedral edge intersected by the current cutting plane.  The
// edge tuple (the edge's two end points) plus the id of the point of
// intersection are retained. The edge tuple assumes that the two vertex end
// points are ordered v0 < v1.
//
// Note that after intersection with the cutting plane, the list of EdgeTuple
// intersection points form a new face. However, the intersection points must
// be sorted around the perimeter of the face, hence determining the loop
// index which represents an ordering of points around the face. The loop
// index is computed by defining a coordinate system from the trim plane
// normal, and the fan of diagonals connecting the points of the face loop.
// This can be used to create new planes that cut the loop into pieces. By
// counting the number of points on either side of these each plane, a loop
// index can be determined.

struct EdgeTuple
{
  vtkIdType V0; //min edge vertex id
  vtkIdType V1; //max edge vertex id
  vtkIdType Id; //point id of inserted point
  vtkIdType LoopIdx; //ordering of the point around the new face.
  vtkIdType Faces[2]; //the two faces using the edge.

  EdgeTuple(vtkIdType v0, vtkIdType v1)
    : V0(v0)
    , V1(v1)
    , Id(-1)
    , LoopIdx(-1)
    , Faces{-1,-1}
  {
    if (this->V0 > this->V1)
    {
      std::swap(this->V0, this->V1);
    }
  }
  bool operator==(const EdgeTuple& et) const { return this->V0 == et.V0 && this->V1 == et.V1; }
  bool operator!=(const EdgeTuple& et) const { return this->V0 != et.V0 || this->V1 != et.V1; }
  bool IsEdge(vtkIdType v0, vtkIdType v1) const
  {
    if (v0 < v1) // ordered properly
    {
      return (this->V0 == v0 && this->V1 == v1);
    }
    else // swap comparison required
    {
      return (this->V0 == v1 && this->V1 == v0);
    }
  }
};

using PointsArray = std::vector<VPoint>; //the polyhedral points (and associated attributes)
using InsertedPointsMap = std::vector<vtkIdType>; //maps input point id to output point id
using InsertedEdgePointsArray = std::vector<EdgeTuple>; //collect edge intersection points


// The face array is just a list of faces for the polyhedron, encoded in a
// contiguous array. For each face there are: 1) the number of points in the
// face; 2) the id of the neighboring point generating the face; and 3) the
// list of point ids defining the face.
using FaceArray = std::vector<vtkIdType>; //a list of polyhedral faces

// Information about the output polyhedral faces. This is used to keep track
// of the output size etc. when processing the Voronoi input points.
struct SurfaceInformation
{
  // Initially these are "number of.." that are transformed to offsets
  // via a subsequent prefix sum operation.
  vtkIdType NumPts; // number of point produced
  vtkIdType NumPrims;// number of primitives (lines or polys) produced
  vtkIdType ConnSize;// size of face connectivity array
  SurfaceInformation() : NumPts(0), NumPrims(0), ConnSize(0) {}
  // Operator += provides support for prefix sum
  SurfaceInformation& operator+=(const SurfaceInformation& info)
  {
    this->NumPts += info.NumPts;
    this->NumPrims += info.NumPrims;
    this->ConnSize += info.ConnSize;
    return *this;
  }
};
using SurfaceInformationType = std::vector<SurfaceInformation>;

//======= These structs and methods are for creating the processing the
//======= adjacency graph / wheel and spokes data structure.

// The adjacency graph is composed of spokes and wheels.  Spokes are used to
// track connecting edges between generating points (i.e., a connected edge,
// or spoke, connects two generating Voronoi points that produce a clipping
// half-space). A list of spokes forms a wheel. (Note in 2D, the spokes are
// sorted around each generating point in CCW order.) Each spoke corresponds
// to a Voronoi hull face (3D) or tile edge (2D).  The spoke and wheel edge
// data structure can be used to optionally validate the topology, and
// ultimately generate output (such as a surface net, or a Delaunay
// tetrahedralization, as requested). Note that the Wheels array is
// essentially an offset into the array of spokes, indicating the beginning
// of a group of n spokes that forms the wheel.
//
// Each spoke may be classified differently. The classification takes into
// account the point ids on each end of the spoke and the order/direction in
// which the two end points occur, and the scalar region ids associated with
// the spoke's two end points. In this classification process, a spoke is
// described by its two end points (ptId,neiId), and the scalar region ids
// (p,n). The classification is as follows. (Note that ptId is always
// ptId >= 0, and p always >= 0.):
// + neiId < 0 || (ptId < neiId && n < 0) : BOUNDARY_CAP (e.g., Voronoi domain boundary)
// + ptId < neiId && p is a contour label && p != n : REGION_BOUNDARY (produces surface net primitive)
// + all other spokes are classified SKIP_SPOKE
//
struct Spoke
{
  vtkIdType NeiId; // Id of the wheel that the spoke is connected to (wheelId,Id)
  unsigned char Classification; // Indicate the classification of this spoke
  enum SpokeClass
  {
    SKIP_SPOKE = 0,
    REGION_BOUNDARY = 1,
    BOUNDARY_CAP = 2,
  };
  Spoke() : NeiId(-1), Classification(SKIP_SPOKE) {}
  Spoke(vtkIdType neiId, unsigned char classification) :
    NeiId(neiId), Classification(classification) {}
};

// Typedefs related to edge spoke structure. Note that the WheelsType vector
// is used to keep track of the number of spokes in a Voronoi hull, as well
// as the number of faces in the Voronoi hull (due to dual property, there is
// a one-to-one correspondance between spoke and a hull face).
using WheelsType = std::vector<vtkIdType>;
using SpokeType = std::vector<Spoke>;
using SpokerIterator = SpokeType::iterator;

// Accumulate local points and cells produced by each thread. Later this
// information may be combined to produce filter output.
struct HullVertex
{
  double X[3];
  HullVertex(double x, double y, double z)
    : X{x,y,z} {}
  HullVertex(const double x[3])
    : X{x[0],x[1],x[2]} {}
};

// Used to collect thread local output. Note that the CellConnType is a
// frankenstein list of vtkIdType, with format (numPrimPts, backfaceId, p0, p1,
// ... numPrimPts, backfaceId, p0, p1, ...).  The backfaceId is the region id
// on the other side of the face used to generate the 2-tuple cell scalars.
using HullVertexType = std::vector<HullVertex>;
using TopoCoordType = std::vector<TopoCoord>;
using CellConnType = std::vector<vtkIdType>;
using MergeTupleType = std::vector<MergeTuple>; //support point merging/smoothing
using MergeMapType = std::vector<vtkIdType>; //maps point ids
using PtsWrittenFlags = std::vector<unsigned char>; //track writing merged points

// ========== These helper functions are created to break cyclic dependencies
// ========== between classes.

// Helper function: Given an input point id, return the offset into the spokes
// array.
vtkIdType GetWheelOffset(const WheelsType& offsets, vtkIdType id)
{ return offsets[id]; }

// Helper function: Given an input point id (or hull id, since there is a
// hull produced for each point), return the number of spokes defining the hull
// polyhedron (i.e., equivalent to the number of polyhedron faces).
vtkIdType GetNumberOfSpokes(const WheelsType& offsets, vtkIdType id)
{ return (offsets[id+1] - offsets[id]); }

// Helper function: Given an input point id (or hull id, since there is a
// hull produced from each input point), return a list of spokes for this
// hull.
const Spoke* GetWheelSpokes(const WheelsType& wheels, const SpokeType& spokes,
                            vtkIdType ptId, vtkIdType& numSpokes)
{
  numSpokes = (wheels[ptId+1] - wheels[ptId]);
  return (spokes.data() + GetWheelOffset(wheels,ptId));
}

// This class does the actual work of building and operating on the 3D
// Voronoi polyhedron cell. It presumes that it is convex, and intersection
// operations are performed against infinite clipping planes. The class is
// used in tandem with another instance of VPolyhedron to repeatedly clip an
// input polyhedron and then output a new polyhedron (i.e., a double buffering
// approach).
struct VPolyhedron
{
  vtkIdType PtId; // Generating point id
  double X[3]; // Generating point position
  double R2; // Radius**2 of the circumflower
  vtkIntArray *RegionIds; // Optional region information for classification

  // These represent the constructed polyhedron
  PointsArray Points; // Array of points defining this polyhedron
  InsertedPointsMap InsertedPoints; // Lookup map between input / output points
  InsertedEdgePointsArray InsertedEdgePoints; // New points generated on intersected edges
  vtkIdType NumFaces; // The number of faces in the face array
  FaceArray Faces; // A list of faces forming this polyhedron.

  // These are for managing degeneracies
  double Tol;
  double Tol2;

  // The polyhedron is initialized with Initialize().
  VPolyhedron()
    : PtId(-1)
    , X{0,0,0}
    , R2(0)
  {
  }

  // Empty out the polyhedron.
  void Clear()
  {
    // Empty out the points and faces
    this->Points.clear();
    this->Faces.clear();
    this->NumFaces = 0;
  }

  // Update the generating point position and id.
  void Update(vtkIdType genPtId, const double genPt[3])
  {
    this->PtId = genPtId;
    this->X[0] = genPt[0];
    this->X[1] = genPt[1];
    this->X[2] = genPt[2];
  }

  // Create the initial Voronoi polyhedron based on the given
  // bounding box (xMin,xMax,yMin,yMax,zMin,zMax). This creates
  // eight points and six faces.
  void Initialize(double bds[6])
  {
    // Empty out the points and faces
    this->Clear();

    // Add the eight points of the initial bounding polyhedron (box). Also
    // add the three faces that intersect at each point. These three faces
    // are the bounding polyhedron faces (hence their negated value).
    this->AddNewPoint(bds[0],bds[2],bds[4]); //point 0
    this->SetPointFaces(0, -1, -3, -5);
    this->AddNewPoint(bds[1],bds[2],bds[4]); //point 1
    this->SetPointFaces(1, -2, -3, -5);
    this->AddNewPoint(bds[0],bds[3],bds[4]); //point 2
    this->SetPointFaces(2, -1, -4, -5);
    this->AddNewPoint(bds[1],bds[3],bds[4]); //point 3
    this->SetPointFaces(3, -2, -4, -5);
    this->AddNewPoint(bds[0],bds[2],bds[5]); //point 4
    this->SetPointFaces(4, -1, -3, -6);
    this->AddNewPoint(bds[1],bds[2],bds[5]); //point 5
    this->SetPointFaces(5, -2, -3, -6);
    this->AddNewPoint(bds[0],bds[3],bds[5]); //point 6
    this->SetPointFaces(6, -1, -4, -6);
    this->AddNewPoint(bds[1],bds[3],bds[5]); //point 7
    this->SetPointFaces(7, -2, -4, -6);

    // Add the six outside faces of the initial boundng polyhedron
    // (box). Note that the "neighboring" points are outside of the bounding
    // box (indicated by <0 values), representing the infinite space bounding
    // the Voroni cell.
    this->AddNewFace(4,-1); //face 0 of the polyhedron, marked as -1
    this->AddFacePoint(0);
    this->AddFacePoint(4);
    this->AddFacePoint(6);
    this->AddFacePoint(2);

    this->AddNewFace(4,-2); //face 1
    this->AddFacePoint(1);
    this->AddFacePoint(3);
    this->AddFacePoint(7);
    this->AddFacePoint(5);

    this->AddNewFace(4,-3); //face 2
    this->AddFacePoint(0);
    this->AddFacePoint(1);
    this->AddFacePoint(5);
    this->AddFacePoint(4);

    this->AddNewFace(4,-4); //face 3
    this->AddFacePoint(2);
    this->AddFacePoint(6);
    this->AddFacePoint(7);
    this->AddFacePoint(3);

    this->AddNewFace(4,-5); //face 4
    this->AddFacePoint(0);
    this->AddFacePoint(2);
    this->AddFacePoint(3);
    this->AddFacePoint(1);

    this->AddNewFace(4,-6); //face 5
    this->AddFacePoint(4);
    this->AddFacePoint(5);
    this->AddFacePoint(7);
    this->AddFacePoint(6);

    // The initial tolerance is determined during the first half-space clip.
    // Here we indicate that the tolerance is uninitialized.
    this->Tol = this->Tol2 = (-1); //mark unset
  }

  // Copy tolerancing information from the sister VPolyhedron. The tolerance
  // is initially determined from the first half space clip of the bounding
  // box, and then is used until the hull is completed.
  void CopyTolerance(VPolyhedron* vpoly)
  {
    this->Tol = vpoly->Tol;
    this->Tol2 = vpoly->Tol2;
  }

  // Compute the bounding Voronoi flower circumcircle (i.e., contains all
  // petals of the Voronoi flower).
  double ComputeCircumFlower()
  {
    double r2;
    this->R2=VTK_FLOAT_MIN; // Initialize

    // Check against the flower petals
    for (auto pItr=this->Points.begin(); pItr != this->Points.end(); ++pItr)
    {
      this->R2 = ( pItr->R2 > this->R2 ? pItr->R2 : this->R2 );
    }
    this->R2 = (4.0 * this->R2); // 2 * (max petal radius)

    return this->R2;
  }

  // Add a new point to this polyhedron. It also computes the Voronoi
  // flower petal radius, and returns its implicit point id (i..e., location
  // in the points array).
  vtkIdType AddNewPoint(double x, double y, double z)
  {
    vtkIdType id = this->Points.size();
    this->Points.emplace_back(VPoint(x,y,z));
    this->Points.back().R2 =
      vtkMath::Distance2BetweenPoints(this->Points.back().X,this->X);
    return id;
  }
  vtkIdType AddNewPoint(double x[3])
  {
    return this->AddNewPoint(x[0], x[1], x[2]);
  }

  // Copy a point from an input polyhedron to this output polyhedron.  Return
  // the new id of the point in the output polyhedron points list. Do not
  // recompute (i.e., reuse) the Voronoi flower petal radius.
  vtkIdType CopyPoint(PointsArray& inPts, vtkIdType inId)
  {
    vtkIdType id = this->Points.size();
    this->Points.emplace_back(inPts[inId]);
    return id;
  }

  // Each point of the polyhedron is created by the intersection of three
  // or more half-space clipping planes (planes defined by neighboring point
  // generators). Note that in degenerate situations, more than three faces
  // may join at a point, in which case a special value indicates a
  // degenerate point.
  // Specify the three faces which produced the given point.
  void SetPointFaces(vtkIdType pId, vtkIdType f0, vtkIdType f1, vtkIdType f2)
  {
    this->Points[pId].Faces[0] = f0;
    this->Points[pId].Faces[1] = f1;
    this->Points[pId].Faces[2] = f2;
  }

  // Mark this point degenerate.
  void MarkPointDegenerate(vtkIdType pId)
  {
    this->Points[pId].Degenerate = true;
  }

  // Add a new face to the list of polyhedron faces. Provide the number of
  // faces defining the face, plus the associated neighbor point id which
  // generates the Voronoi face. Return an offset into the faces array so
  // that the number of points in the face can be updated if necessary.
  vtkIdType AddNewFace(vtkIdType npts, vtkIdType neiPtId)
  {
    vtkIdType offset = this->Faces.size();
    this->Faces.emplace_back(npts);
    this->Faces.emplace_back(neiPtId);
    this->NumFaces++;
    return offset;
  }

  // Add a point id defining the current face. This method is called
  // after AddNewFace().
  void AddFacePoint(vtkIdType ptId)
  {
    this->Faces.emplace_back(ptId);
  }

  // Update the number of points in the current face. Provide the offset
  // into the current face (from AddNewFace()).
  void UpdateFaceCount(vtkIdType offset, vtkIdType npts)
  {
    this->Faces[offset] = npts;
  }

  // Prepare to add a batch of N points.
  vtkIdType AddNFacePoints(vtkIdType batchSize)
  {
    vtkIdType offset = this->Faces.size();
    this->Faces.resize(offset+batchSize);
    return offset;
  }

  // Add a point to previously allocated batch of N points. The loop index
  // loopIdx must be < batchSize.
  void AddNthFacePoint(vtkIdType offset, vtkIdType loopIdx, vtkIdType ptId)
  {
    this->Faces[offset+loopIdx] = ptId;
  }

  // Evaluate the 3D plane equation for a given point x. Normal n is expected
  // to be a unit normal to the plane; o is a plane origin (i.e., point on
  // the plane).
  double EvaluatePlane(double x[3], double o[3], double n[3])
  {
    return (((x[0]-o[0])*n[0]) + ((x[1]-o[1])*n[1]) + ((x[2]-o[2])*n[2]));
  }

  // Intersect this Voronoi polyhedron with the plane defined by an origin
  // and unit normal. The Voronoi neighbor point id associated with this
  // plane must also be provided. The result of the operation is placed
  // into a new Voronoi polyhedron.
  bool IntersectWithPlane(double origin[3], double normal[3], vtkIdType neiPtId,
                          VPolyhedron *outPoly)
  {
    // Check to see if a tolerance has been determined. The first clip plane
    // (i.e., the first time this method is invoked for the current
    // polyhedron) is defined by the closest point to the initial polyhedron,
    // so it is a good time to define the tolerance for this
    // polyhedron. Note, however, this means the tolerance is adaptive to
    // point spacing.
    if ( this->Tol < 0 )
    {
      double e = std::numeric_limits<double>::epsilon();
      this->Tol = 10 * e * sqrt(vtkMath::Distance2BetweenPoints(origin,this->X));
      this->Tol2 = this->Tol * this->Tol;
    }

    // Begin by evaluating all the polyhedron points against the plane.  If
    // all points are "inside" (<=0 based on the tolerance) then there is no
    // intersection.
    double val;
    bool intersects=false;
    for (auto pItr=this->Points.begin(); pItr != this->Points.end(); ++pItr)
    {
      val = this->EvaluatePlane(pItr->X, origin, normal);
      val = ( (-this->Tol <= val && val <= this->Tol) ? 0.0 : val);
      if ( val > 0 )
      {
        intersects = true;
      }
      pItr->Val = val;
    }
    if ( !intersects )
    {
      return false;
    }

    // An intersection has occurred. Prepare this polyhedron for further
    // processing, as well as the output polyhedron.
    this->InsertedPoints.resize(this->Points.size());
    std::fill_n(this->InsertedPoints.begin(), this->Points.size(), (-1));
    this->InsertedEdgePoints.clear();
    outPoly->Clear();

    // Now process the faces that compose the current polyhedron. Output
    // points and faces to the output polyhedron.
    vtkIdType *faces = this->Faces.data();
    vtkIdType npts, *pids, neiId;
    vtkIdType offset=0, fOffset;
    while ( offset < this->Faces.size() )
    {
      vtkIdType *face = faces + offset;
      npts = face[0];
      neiId = face[1];
      pids = face + 2;

      // Process the current face. Points in face with Val<=0 are retained.
      // Edges that intersect the plane produce new points. These new points
      // are gathered into a loop to produce one new face.
      vtkIdType i, v0, v1, pid, numNewPts=0;
      double val0, val1, t, x[3];
      for ( i=0; i < npts; ++i)
      {
        v0 = pids[i];
        v1 = pids[(((i+1)==npts) ? 0 : (i+1))];

        val0 = this->Points[v0].Val;
        val1 = this->Points[v1].Val;

        // If the plane evaluated value is <=0, add the exisiting point to a
        // new face.
        if ( val0 <= 0.0 )
        {
          // See if this point has been copied previously. If not, insert it.
          if ( (pid = this->InsertedPoints[v0]) < 0 )
          {
            pid = outPoly->CopyPoint(this->Points,v0);
            this->InsertedPoints[v0] = pid;
          }
          // Create a new face if necessary
          if (!numNewPts)
          {
            fOffset = outPoly->AddNewFace(0,neiId);
          }
          // Add the point to the current face.
          outPoly->AddFacePoint(pid);
          numNewPts++;
        } //copying point

        // Now check to see if there is a face edge intersection. If so, create
        // a new point if not already created.
        if ( (val0 <= 0.0 && val1 > 0.0) || (val0 > 0.0 && val1 <= 0.0) )
        {
          InsertedEdgePointsArray::iterator insItr =
            std::find(this->InsertedEdgePoints.begin(), this->InsertedEdgePoints.end(),EdgeTuple(v0,v1));
          if ( insItr != this->InsertedEdgePoints.end() )
          {
            pid = insItr->Id;
            insItr->Faces[1] = neiId; //second face
          }
          else
          {
            t = -val0 / (val1-val0);
            x[0] = this->Points[v0].X[0] + t*(this->Points[v1].X[0]-this->Points[v0].X[0]);
            x[1] = this->Points[v0].X[1] + t*(this->Points[v1].X[1]-this->Points[v0].X[1]);
            x[2] = this->Points[v0].X[2] + t*(this->Points[v1].X[2]-this->Points[v0].X[2]);
            pid = outPoly->AddNewPoint(x);
            this->InsertedEdgePoints.emplace_back(EdgeTuple(v0,v1));
            this->InsertedEdgePoints.back().Id = pid;
            this->InsertedEdgePoints.back().Faces[0] = neiId; //first face
          }

          // Create a new face if necessary
          if (!numNewPts)
          {
            fOffset = outPoly->AddNewFace(0,neiId);
          }
          outPoly->AddFacePoint(pid);
          numNewPts++;
        } // Create a new face intersection point
      } //for all points and edges in this face

      // See whether the face was copied, or clipped. If so, a new face was created.
      // If not, this face was clipped away.
      if ( numNewPts > 0 )
      {
        outPoly->UpdateFaceCount(fOffset,numNewPts);
      }
      else
      {
        this->NumFaces--; //this face entirely removed / clipped away
      }

      // Advance to the next face
      offset += (npts+2);
    } // while processing faces

    // All the faces have been clipped by the plane. We've gathered all of
    // the intersection points. These now must be circumferentially ordered
    // to create a final capping polygon. Fortunately, the polygon is convex,
    // so the ordering (i.e., assigning of loop index/order) can be performed
    // by counting the number of points on one side of the polygon fan
    // diagonals. (This cute method works well for small number of vertices
    // in the capping polygon--no sqrt nor atan functions, nor angle
    // sort. However, for large numbers of points, which is rare, the method
    // will not scale well and may require an alternative method.

    // The number of vertices forming the capping polygon.
    npts = this->InsertedEdgePoints.size();
    if ( npts < 3 )
    {
      cout << "Degenerate loop with: (" << this->PtId << "," << npts << ") verts\n";
      return true;
    }

    // The fan pivot point, assigned loop index 0. All diagonals across the
    // capping polygon start from the pivot point.
    this->InsertedEdgePoints[0].LoopIdx = 0;
    double *xO = outPoly->Points[this->InsertedEdgePoints[0].Id].X;

    // Now create lines between the pivot point and other points in the
    // capping polygon. These lines define separation planes orthogonal to
    // the clipped polygon. By counting the number of points on one side of
    // the separation plane, we can assign a loop index value.
    double xxO[3], sepN[3];
    for (vtkIdType i=1; i < npts; ++i)
    {
      // Compute the separation plane.
      double *x = outPoly->Points[this->InsertedEdgePoints[i].Id].X;
      xxO[0] = x[0] - xO[0];
      xxO[1] = x[1] - xO[1];
      xxO[2] = x[2] - xO[2];
      vtkMath::Cross(xxO,normal,sepN);

      // Now evaluate all other points against the separation plane. We
      // don't worry about normalizing the separation plane normal because
      // we just counting values >0.
      vtkIdType idx=1; //the index is one to begin with.
      for (vtkIdType j=1; j < npts; ++j)
      {
        if ( i == j )
        {
          continue;
        }
        double *x = outPoly->Points[this->InsertedEdgePoints[j].Id].X;
        if ( this->EvaluatePlane(x, xO, sepN) > 0 )
        {
          idx++;
        }
      } //point evaluation

      // Assign loop index
      this->InsertedEdgePoints[i].LoopIdx = idx;
    } //for all non-origin points forming the capping polygon.

    // Finally, create a new face (the capping polygon) with the points
    // inserted in the correct order (using the loop index).
    outPoly->AddNewFace(npts,neiPtId);
    offset = outPoly->AddNFacePoints(npts);
    for (vtkIdType i=0; i < npts; ++i)
    {
      outPoly->AddNthFacePoint(offset,this->InsertedEdgePoints[i].LoopIdx,
                               this->InsertedEdgePoints[i].Id);
      outPoly->SetPointFaces(this->InsertedEdgePoints[i].Id,
                             this->InsertedEdgePoints[i].Faces[0],
                             this->InsertedEdgePoints[i].Faces[1],
                             neiPtId);
    }

    return true;
  }

  // See VHull documentation.
  const Spoke*
  AddAdjacencyInformation(vtkIdType ptId, WheelsType& wheels,
                          SpokeType& spokes, int& numSpokes,
                          vtkLabelMapLookup<int>* lMap,
                          int bCapping)
  {
    // Keep track of the starting position at which
    // spokes may be added.
    vtkIdType startPos = spokes.size();

    // Is the current ptId a segmentation label?
    int labelVal = this->RegionIds->GetValue(ptId);
    bool isSelectedLabel = lMap->IsLabelValue(labelVal);

    // Loop over faces, classify associated spokes
    vtkIdType numPts = this->Points.size();
    vtkIdType *faces = this->Faces.data();
    vtkIdType offset = 0;
    numSpokes = 0;
    while ( offset < this->Faces.size() )
    {
      vtkIdType *face = faces + offset;
      vtkIdType npts = face[0];
      vtkIdType neiId = face[1];

      // Classify the spoke - this information is very useful for later
      // processing, e.g., identifying face primitives to output.
      unsigned char spokeClass = Spoke::SKIP_SPOKE;
      if ( bCapping && neiId < 0 && isSelectedLabel )
      {
        spokeClass = Spoke::BOUNDARY_CAP;
      }

      else if ( ptId < neiId )
      {
        int neiVal = this->RegionIds->GetValue(neiId);
        if ( labelVal != neiVal &&
             (isSelectedLabel || lMap->IsLabelValue(neiVal)) )
        {
          spokeClass = Spoke::REGION_BOUNDARY;
        }
      }

      // Create the spoke and add it the spokes vector
      spokes.emplace_back(Spoke(neiId,spokeClass));
      offset += (npts+2);
      ++numSpokes;
    } // while processing polyhedral faces

    wheels[ptId] = numSpokes; //numFaces == numSpokes

    // Spokes are added with emplace_back(), so may cause
    // reallocations. So we wait for all spokes to be added
    // before returning the pointer to the list of spokes.
    return spokes.data() + startPos;
  }

  // See VHull documentation.
  void AddSurfaceInformation(vtkIdType ptId, const Spoke* spokes,
                             int numSpokes, SurfaceInformationType& surfInfo,
                             HullVertexType& localPts, TopoCoordType& localTups,
                             CellConnType& localConn)
  {
    // Generate output only if hull faces exist
    if ( this->NumFaces <= 0 )
    {
      return;
    }

    // Output domain boundary faces
    int numPts, numFaces, connSize;
    this->AddSurfaceFaces(spokes, numSpokes, numPts, numFaces, connSize,
                          localPts, localTups, localConn);
    surfInfo[ptId].NumPts = numPts;
    surfInfo[ptId].NumPrims = numFaces;
    surfInfo[ptId].ConnSize = connSize;

  } // AddSurfaceInformation()

  // Output local points and faces from current polyhedron, as specified by
  // faceTypes.  The face types are an "ORed |" combination of output types
  // as contained in the provided spokes. Faces matching the faceTypes are
  // output: their points and connectivity. Also, information is gathered
  // about the number of output points and faces, and size of connectivity.
  void AddSurfaceFaces(const Spoke* spokes, int numSpokes,
                       int &numOutputPts, int& numOutputFaces, int& connSize,
                       HullVertexType& hullPts, TopoCoordType& tups,
                       CellConnType& faceConn)
  {
    // Loop over all the polyhedron faces, checking against the specified type(s).
    vtkIdType *faces = this->Faces.data();
    vtkIdType *faceIds, backfaceId, numFacePts, ptId, offset = 0;
    numOutputFaces = 0;
    connSize = 0;
    numOutputPts = 0;

    for (int faceNum=0; faceNum < this->NumFaces; ++faceNum)
    {
      vtkIdType *face = faces + offset;
      numFacePts = face[0];
      backfaceId = face[1];

      if ( spokes[faceNum].Classification )
      {
        numOutputFaces++;
        faceConn.emplace_back(numFacePts);
        faceConn.emplace_back(backfaceId);
        faceIds = face + 2;
        connSize += numFacePts;
        for (vtkIdType i=0; i < numFacePts; ++i)
        {
          if (this->Points[faceIds[i]].PtMap < 0)
          {
            this->Points[faceIds[i]].PtMap = numOutputPts++;
            hullPts.emplace_back(HullVertex(this->Points[faceIds[i]].X));
            tups.emplace_back(TopoCoord(this->Points[faceIds[i]].Faces,this->PtId));
          }
          faceConn.emplace_back(this->Points[faceIds[i]].PtMap);
        }
      } // specified face type matches
      offset += (numFacePts + 2);
    } // while processing polyhedral faces
  }   // AddSurfaceFaces
}; // VPolyhedron

// The incremental Voronoi class VHull is used to create a 3D Voronoi cell,
// v_i, from incremental insertion of neighboring points around a Voronoi
// generating point p_i. (Note that the Voronoi cell v_i is defined by a
// convex polyhedron whose interior is closer to p_i than any od the the
// neighboring points p_j.) To use this class, define the Voronoi generator
// point p_i, and a 3D bounding box B in which the Voronoi cell exists.
// Then, incrementally inject neighboring points p_j to produce the Voronoi
// cell. Note, that inserted points p_j are not required to intersect the
// v_i; but the bounding box B must strictly contain the generator point
// p_i. Because Voronoi cells may be semi-infinite in extent, B may bound the
// v_i.
//
// The algorithm proceeds as follows. The p_i is initially placed interior to
// B, so that v_i == B. Then as each point p_j is inserted, a Voronoi
// half-space is created by defining a plane which equally separates p_i from
// p_j. This plane is then used to intersect each face of the polyhedron v_i,
// producing a new v_i. The process continues until all neighboring points
// have been inserted.
//
// Note that the class supports related methods such as computing the Voronoi
// flower, and the Voronoi circumflower. It can also output 1) a surface net
// based on specified labels, and 2) the boundary faces of the data.
// Internally, it uses two instances of VPolyhedron (via a double buffering
// approach) to iteratively build the v_j.
struct VHull
{
  // Define the Voronoi generating point including its associated id.
  vtkIdType NPts;                   // total number of points in dataset
  const double *Points;             // input dataset points
  vtkIntArray *RegionIds;           // Optional region information for classification
  vtkIdType PointId;                // generating hull point id (for this hull)
  double TileX[3];                  // generating hull point - x-y-z coordinates
  vtkStaticPointLocator* Locator;   // locator
  double PaddedBounds[6];           // the domain over which Voronoi is calculated
  double Bounds[6];                 // locator bounds
  int Divisions[3];                 // locator binning dimensions
  double H[3];                      // locator spacing
  double Padding2;                  // Bounding box padding distance
  vtkIdType NumClips;               // The number of clips so far
  bool BoundaryCapping;             // whether to cap surface nets on the boundary

  // Use a double buffering approach to incrementally construct the Voronoi
  // cells as the cell is repeatedly clipped by the addition of neighboring
  // points.  VPO is the current polyhedron; when a neighboring point is used
  // to trim VPO, the result is written into VP1. Then VP0 and VP1 are
  // swapped. We use double buffering to avoid repeated memory allocation and
  // deletion.
  VPolyhedron *VP0;
  VPolyhedron *VP1;

  // This is used to identify and map the points of the polyhedron to be output.
  vtkIdType NumMarkedPts;

  VHull()
  {
    this->VP0 = new VPolyhedron();
    this->VP1 = new VPolyhedron();
  }
  // Copy constructor needed for SMP
  VHull(const VHull& v)
  {
    this->VP0 = new VPolyhedron();
    this->VP1 = new VPolyhedron();
  }
  ~VHull()
  {
    delete this->VP0;
    delete this->VP1;
  }

  // Convenience methods for setting region segmenation labals
  void SetRegionIds(vtkIntArray* regionIds)
  {
    this->VP0->RegionIds = regionIds;
    this->VP1->RegionIds = regionIds;
  }

  /**
   * Initialize the Voronoi cell v_i by setting it to the provided bounding
   * box B, with generating point p_i. Note that p_i must be strictly
   * contained within B.
   */
  void Initialize(vtkIdType genPtId, const double genPt[3])
  {
    // Update the hull
    this->PointId = genPtId;
    this->TileX[0] = genPt[0];
    this->TileX[1] = genPt[1];
    this->TileX[2] = genPt[2];

    // Initialize the number of clips
    this->NumClips = 0;

    // Start the incremental polyhedral generation process.
    this->VP0->Update(genPtId,genPt);
    this->VP1->Update(genPtId,genPt);
    this->VP0->Initialize(this->PaddedBounds);
  }

  /**
   * Insert the next point neighboring point p_j. The method will return
   * true if the v_i is modified as a result of inserting the point. Otherwise
   * the v_i is not modified. Make sure that Initialize() has been invoked
   * prior to calling this method.
   */
  bool InsertPoint(vtkIdType neiPtId, const double neiPt[3])
  {
    double origin[3], normal[3];
    bool negate = false;
    if ( neiPtId < this->PointId )
    {
      origin[0] = (neiPt[0] + this->TileX[0]) / 2.0;
      origin[1] = (neiPt[1] + this->TileX[1]) / 2.0;
      origin[2] = (neiPt[2] + this->TileX[2]) / 2.0;
      normal[0] = neiPt[0] - this->TileX[0];
      normal[1] = neiPt[1] - this->TileX[1];
      normal[2] = neiPt[2] - this->TileX[2];
    }
    else
    {
      origin[0] = (this->TileX[0] + neiPt[0]) / 2.0;
      origin[1] = (this->TileX[1] + neiPt[1]) / 2.0;
      origin[2] = (this->TileX[2] + neiPt[2]) / 2.0;
      normal[0] = this->TileX[0] - neiPt[0];
      normal[1] = this->TileX[1] - neiPt[1];
      normal[2] = this->TileX[2] - neiPt[2];
      negate = true;
    }
    vtkMath::Normalize(normal);
    if (negate)
    {
      normal[0] = (-normal[0]);
      normal[1] = (-normal[1]);
      normal[2] = (-normal[2]);
    }

    // If an intersection occurs, swap the Voronoi polyhedra. Otherwise,
    // retain the current Voronoi polygon. Also update the circumflower
    // radius.
    bool intersects =
      this->VP0->IntersectWithPlane(origin,normal,neiPtId, this->VP1);
    if ( intersects )
    {
      // Forward the computed tolerance.
      this->VP1->CopyTolerance(this->VP0);
      std::swap(this->VP0, this->VP1);
    }

    return intersects;
  }

  // Get the circumflower**2 for thr the current polyhedron.
  double GetCircumFlower()
  {
    return this->VP0->R2;
  }

  // Determine whether the provided point is within the Voronoi circumflower.
  // Return true if it is; false otherwise.
  bool InCircumFlower(const double p[3])
  {
    double r2 = vtkMath::Distance2BetweenPoints(p,this->VP0->X);
    if ( r2 <= this->VP0->R2 )
    {
      return true;
    }

    // Point not in the circumflower
    return false;
  }

  // Determine whether the provided point is within the Voronoi flower
  // neighborhood metric (i.e., inside one of the petals). Return true if it
  // is; false otherwise.
  bool InFlower(const double p[3])
  {
    // Check against the flower petals
    for (auto pItr=this->VP0->Points.begin(); pItr != this->VP0->Points.end(); ++pItr)
    {
      double fr2 = pItr->R2;
      double r2 = vtkMath::Distance2BetweenPoints(p,pItr->X);
      if ( r2 <= fr2 )
      {
        return true;
      }
    }

    // Point not in the flower
    return false;
  }

  // Compute the Voronoi circumflower
  double ComputeCircumFlower()
  {
    return this->VP0->ComputeCircumFlower();
  }

  // If spoke pruning is requested, then edges that are "small" relative to
  // the length of the spoke are deleted.
  void Prune(double pruneTol2)
  {
  }

  // Generate a 3D Voronoi hull by iterative clipping of the hull with nearby
  // points.  Termination of the clipping process occurs when the neighboring
  // points become "far enough" away from the generating point (i.e., the
  // Voronoi Flower error metric is satisfied).
  bool BuildHull(vtkIdList* pIds, vtkDoubleArray *radii2, const double* pts,
                 vtkIdType maxClips)
  {
    // Ensure there are clips to be performed.
    if ( maxClips <= 0 )
    {
      return true;
    }

    const double* v;
    vtkIdType& numClips=this->NumClips;
    vtkIdType ptId,  numPts=this->NPts;

    // Request neighboring points around the generating point in annular
    // shells. The rings are defined by an inner and outer radius
    // (min,max]. The requested points fall within the shell, with shell
    // radii r:(min<r<=max). The neighboring points are used to perform
    // half-space clipping of the Voronoi hull. (The original polyhedral hull
    // around the generating point is defined from the bounding box of the
    // domain.) The Voronoi Flower and Circumflower neighborhood metrics are
    // used to terminate the clipping process. The Voronoi Flower is the set
    // of all flower (i.e., Delaunay circumspheres) located at the Voronoi
    // hull vertices. The Circumflower is the sphere that bounds all petals
    // of the Voronoi Flower.
    constexpr int INIT_QUERY_SIZE = 16;
    constexpr int QUERY_SIZE = 8;
    double R2 = VTK_FLOAT_MAX;
    double annulusMin2 = 0.0;
    double annulusMax2 =
      this->Locator->FindNPointsInShell(INIT_QUERY_SIZE, this->TileX, pIds, radii2, annulusMin2);
    vtkIdType numPtIds, *pIdsPtr;
    double *radii2Ptr;
    bool initQuery=true;

    // Now add additional points until they are outside of the Voronoi
    // flower. For speed, we use the bounding Voronoi circumcircle to
    // determine whether points are outside of the flower. Note that in the
    // while() loop below, if the number of points pIds<=0, then all points
    // have been exhausted and the loop is exited.
    while ( (numPtIds = pIds->GetNumberOfIds()) > 0 && annulusMin2 <= R2 && numClips < maxClips )
    {
      pIdsPtr = pIds->GetPointer(0);
      radii2Ptr = radii2->GetPointer(0);
      for ( vtkIdType i=0; i < numPtIds && numClips < maxClips; ++i )
      {
        ptId = pIdsPtr[i];
        v = pts + 3 * ptId;
        // Only check CircumFlower and InFlower() after the first pass
        if ( initQuery || (radii2Ptr[i] <= R2 && this->InFlower(v)) )
        {
          if ( this->InsertPoint(ptId,v) )
          {
            numClips++;
          }
        }
      } // process all points in requested annulus

      // Okay start checking against the CircumFlower and Voronoi Flower
      initQuery = false;
      R2 = this->ComputeCircumFlower();

      // See if circumflower radius is less then radius of annulus request; if so, the
      // Voronoi hull has been formed.
      if ( R2 < annulusMax2 )
      {
        break;
      }

      // Grab the next ring / annulus of points
      annulusMin2 = annulusMax2;
      annulusMax2 = this->Locator->FindNPointsInShell(QUERY_SIZE, this->TileX, pIds, radii2, annulusMin2);
    } // while points still in Voronoi circumflower

    return true;
  } // BuildHull

  // Add the adjacency information for the specified generating point ptId,
  // i.e., the neighboring points (or spokes) that created this hull. Place
  // the information into the local hull's wheels and spokes data structure,
  // and return a list of this hull's spokes and the number of spokes found.
  // Also keep track of the maximum number of points and faces created across
  // all hulls processed by this thread. Note: the spokes is a thread local
  // SpokeType vector, the returned list const Spoke* is ephemeral and only
  // valid until AddAdjacency() is called again (within the current thread).
  const Spoke*
  AddAdjacencyInformation(vtkIdType ptId, WheelsType& globalWheels,
                          SpokeType& localSpokes, int& numSpokes,
                          vtkLabelMapLookup<int>* lMap)
  {
    return this->VP0->
      AddAdjacencyInformation(ptId,globalWheels,localSpokes,numSpokes,
                              lMap, this->BoundaryCapping);
  }

  // Use this method to provide information about the output polygonal mesh
  // or surface net (i.e., used to construct the second output #1). This
  // method is called at the end of processing the hull; i.e., once the Voronoi
  // polyhedron is computed. This method records the number of points, number
  // of faces/lines, and the size of the connectivity list representing the
  // faces/lines. It also adds cell connectivity and generated points to the
  // provided thread local data vectors (for later compositing). This method
  // is useful especially in parallel computing, as it can be used to keep
  // track of the output size of the Voronoi tessellation, and then filled in
  // later (i.e., prefer one-time memory allocation versus incremental memory
  // allocation).
  void AddSurfaceInformation(vtkIdType ptId, const Spoke* spokes,
                             int numSpokes, SurfaceInformationType& surfInfo,
                             HullVertexType& localPts, TopoCoordType& localTups,
                             CellConnType& localConn)
  {
    this->VP0->AddSurfaceInformation(ptId,spokes,numSpokes,surfInfo,localPts,
                                     localTups, localConn);
  }

}; //VHull

//======= These structs and methods are for managing the threading of
//======= batches of data.

// Class to manage batches of points. This is used to improve threaded
// performance and reduce memory. Note the "exception" when a subset of items
// is to be processed in a single batch (e.g., single point of interest).
struct BatchManager
{
  vtkIdType Num; //Number of total items to process
  vtkIdType BatchSize; //The desired batch size (clamped by Num)
  vtkIdType NumBatches; //The total number of batches to process
  BatchManager(vtkIdType num, vtkIdType batchSize) : Num(num), BatchSize(batchSize)
  {
    this->NumBatches = static_cast<vtkIdType>(ceil(static_cast<double>(num) / batchSize));
  }
  BatchManager(const BatchManager& batchManager)
  {
    this->Num = batchManager.Num;
    this->BatchSize = batchManager.BatchSize;
    this->NumBatches = batchManager.NumBatches;
  }
  vtkIdType GetNumberOfBatches() const
  {
    return this->NumBatches;
  }
  vtkIdType GetBatchItemRange(vtkIdType batchNum, vtkIdType &startId, vtkIdType &endId) const
  {
    startId = batchNum * this->BatchSize;
    endId = startId + this->BatchSize;
    endId = (endId > this->Num ? this->Num : endId);
    return (endId - startId);
  }
}; //BatchManager

// This is used to track information about each batch that is processed.
struct BatchInfo
{
  vtkIdType Id; //the batch id
  BatchInfo(vtkIdType id) : Id(id) {}
};
// A list of batch information. Used by threads to record the
// batches they've processed.
using BatchInfoType = std::vector<BatchInfo>;

//======= These structs and methods are for capturing data within a
//======= thread.

// Track local data on a per-thread basis. In the data compositing / Reduce()
// method this information will be used to combine the data from each thread
// into the requested VTK filter outputs. Note that not all of this data is
// used (especially the vector types) depending on the output type requested.
struct LocalDataType
{
  BatchInfoType LocalBatches; //the list of batches of generating points processed by this thread
  SpokeType LocalSpokes;  //connecting edges/spokes for each hull
  HullVertexType LocalPoints; //point coordinates defining the hull vertices
  TopoCoordType LocalTuples; //points represented in topological tuple space
  CellConnType LocalCellConn; //cell connectivity generated by this thread prior to compositing
  SurfaceInformationType LocalSurfaceInformation; //information about surface output
  VHull Hull; //computational 3D Voronoi hull algorithm
  vtkIdType ThreadId; //assign a thread id [0,NumThreadsUsed). Used later for processing.

  LocalDataType() : ThreadId(-1)
  {
    this->LocalBatches.reserve(2048);
    this->LocalSpokes.reserve(2048);
    this->LocalPoints.reserve(2048);
    this->LocalTuples.reserve(2048);
    this->LocalCellConn.reserve(2048);
  }
};

// Map thread local data to thread id. This may be used later as part of the
// data compositing / reduction process.
using ThreadMapType = std::vector<LocalDataType*>;

// Functor used to generate the adjacency (wheels and spokes) structure.
// It basically copies thread local data into a global vector.
// Execution is across the n threads used to compute the spokes--so this is
// a parallel generation of the wheels/spokes data structure.
struct ProduceSpokes
{
  const BatchManager &Batcher;
  const ThreadMapType& ThreadMap;
  const WheelsType& Wheels;
  SpokeType& Spokes;

  ProduceSpokes(const BatchManager &batcher, ThreadMapType &threadMap,
                WheelsType& wheels, SpokeType& spokes)
    : Batcher(batcher)
    , ThreadMap(threadMap)
    , Wheels(wheels)
    , Spokes(spokes)
  {
  }

  void operator()(vtkIdType threadId, vtkIdType endThreadId)
  {
    const BatchManager &batcher = this->Batcher;
    const WheelsType &wheels = this->Wheels;
    Spoke *spokes;
    vtkIdType i, numSpokes, ptId;

    // Now copy the spokes into the right spot
    for ( ; threadId < endThreadId; ++threadId )
    {
      LocalDataType& localData = *(this->ThreadMap[threadId]);

      // Loop over all batches in this thread
      auto spItr = localData.LocalSpokes.begin();
      vtkIdType ptId, endPtId;
      for ( auto& batchInfo : localData.LocalBatches )
      {
        batcher.GetBatchItemRange(batchInfo.Id, ptId, endPtId);

        // Loop over all contiguous spokes in this batch
        spokes = this->Spokes.data() + GetWheelOffset(wheels,ptId);
        for ( ; ptId < endPtId; ++ptId )
        {
          vtkIdType numSpokes = GetNumberOfSpokes(wheels,ptId);
          for ( auto i=0; i < numSpokes; ++i, ++spItr, ++spokes)
          {
            spokes->NeiId = spItr->NeiId;
            spokes->Classification = spItr->Classification;
          }
        } // for all contiguous points in this batch
      }   // for all batches
    }     // across all threads in this batch
  }
}; // ProduceSpokes

// The threaded core of the algorithm. It computes the wheel and spokes data
// structure, and gathers information about the tessellation. Later this
// information is used to produce output. This struct could be templated over
// point type, but due to numerical sensitivity we'll just process doubles
// for now.
struct VCore
{
  const BatchManager Batcher; //Controls processing of batches of generating points
  ThreadMapType ThreadMap; //Keep track of local thread data for later access
  vtkPoints *InPoints; //Input points as data array
  vtkIdType NPts; //The number of input (Voronoi hull generation) points
  const double *Points; //Input points as pointer
  vtkIntArray *RegionIds; //Optional region ids to control tessellation
  vtkStaticPointLocator *Locator; //Used to (quickly) find nearby points
  const vtkIdType* NetScalars; //optional input point scalars to produce a surface net
  double PaddedBounds[6]; //the expanded domain over which Voronoi is calculated
  double Bounds[6]; //locator bounds
  int Divisions[3]; //Internal parameters for computing
  double H[3]; //Locator bin spacing
  double Padding; //The padding distance**2 around the bounding box

  vtkIdType MaxClips; //Control the maximum number of half-space clips
  int NumThreadsUsed; //Keep track of the number of threads used

  // This are used to create the spokes and wheels adjacency graph used to
  // validate the tessllation and produce a Delaunay triangulation.
  WheelsType Wheels; //Wheel/spokes data structure: offset array to spokes
  vtkIdType NumSpokes; //Total number of edges / spokes
  SpokeType Spokes; //Spokes / edges with classification

  // Keep track of information for producing the filter output
  SurfaceInformationType SurfInfo;

  // Used for controlling filter abort and accessing filter information
  vtkGeneralizedSurfaceNets3D* Filter;

  // Indicate whether to cap the surface net along the domain boundary.
  bool BoundaryCapping;

  // Used to manage contour labels
  vtkIdType NumLabels;
  const double* LabelValues;
  int BackgroundLabel; // used to label tuples with one side on the boudnary

  // Control whether points are merged and/or smoothing is performed. Define
  // infrastructure for merging points.
  bool MergePoints;
  bool Smoothing;
  MergeTupleType MergeTuples; //temporary array for merging points
  MergeMapType MergeMap; //maps hull vertex point ids to merged point ids
  vtkIdType NumMergedPts; //after merging, the number of points remaining

  // Storage local to each thread. We don't want to allocate working arrays
  // on every thread invocation. Thread local storage saves lots of
  // new/delete (e.g. the PIds).
  vtkSMPThreadLocalObject<vtkIdList> PIds;
  vtkSMPThreadLocalObject<vtkDoubleArray> Radii2;
  vtkSMPThreadLocal<LocalDataType> LocalData;
  vtkSMPThreadLocal<vtkLabelMapLookup<int>*> LMap;

  VCore(BatchManager &batcher, vtkPoints *inPts, vtkIntArray* regionIds,
        vtkStaticPointLocator* loc, double padding, vtkIdType maxClips,
        vtkGeneralizedSurfaceNets3D* filter)
    : Batcher(batcher)
    , InPoints(inPts)
    , RegionIds(regionIds)
    , Locator(loc)
    , Padding(padding)
    , MaxClips(maxClips)
    , NumThreadsUsed(0)
    , NumSpokes(0)
    , Filter(filter)
  {
    // Set up points for processing
    this->NPts = this->InPoints->GetNumberOfPoints();
    this->Points = vtkDoubleArray::FastDownCast(inPts->GetData())->GetPointer(0);

    // Wheels: one per input generating point. This is transformed
    // into a vector of offsets into the spokes vector for later processing,
    // hence the +1 allocation.
    this->Wheels.resize(this->NPts+1,0); //initialized to zero

    // Keep track of output size etc.
    this->SurfInfo.resize(this->NPts+1);

    // Pre-compute some local data in preparation for processing.
    loc->GetBounds(this->Bounds);
    loc->GetDivisions(this->Divisions);
    this->H[0] = (this->Bounds[1] - this->Bounds[0]) / static_cast<double>(this->Divisions[0]);
    this->H[1] = (this->Bounds[3] - this->Bounds[2]) / static_cast<double>(this->Divisions[1]);
    this->H[2] = (this->Bounds[5] - this->Bounds[4]) / static_cast<double>(this->Divisions[2]);

    // Define the Voronoi domain by padding out from bounds.
    for (int i = 0; i < 3; ++i)
    {
      this->PaddedBounds[2 * i] = this->Bounds[2 * i] - padding;
      this->PaddedBounds[2 * i + 1] = this->Bounds[2 * i + 1] + padding;
    }

    // Set up boundary capping
    this->BoundaryCapping = this->Filter->GetBoundaryCapping();

    // Set up the contour labels used to extract surfaces
    this->NumLabels = this->Filter->GetNumberOfLabels();
    this->LabelValues = this->Filter->GetValues();
    this->BackgroundLabel = this->Filter->GetBackgroundLabel();

    // Control point merging and smoothing
    this->MergePoints = this->Filter->GetMergePoints();
    this->Smoothing = this->Filter->GetSmoothing();
  }

  // Allocate a little bit of memory to get started. Set some initial values
  // for each thread for accelerating computation.
  void Initialize()
  {
    vtkIdList*& pIds = this->PIds.Local();
    pIds->Allocate(128); // allocate some memory

    vtkDoubleArray*& radii2 = this->Radii2.Local();
    radii2->Allocate(128); // allocate some memory

    // Populate the instance of the VHull class
    LocalDataType& localData = this->LocalData.Local();
    localData.Hull.NPts = this->NPts;
    localData.Hull.Points = this->Points;
    localData.Hull.SetRegionIds(this->RegionIds);
    localData.Hull.Locator = this->Locator;
    localData.Hull.Divisions[0] = this->Divisions[0];
    localData.Hull.Divisions[1] = this->Divisions[1];
    localData.Hull.Divisions[2] = this->Divisions[2];
    localData.Hull.PaddedBounds[0] = this->PaddedBounds[0];
    localData.Hull.PaddedBounds[1] = this->PaddedBounds[1];
    localData.Hull.PaddedBounds[2] = this->PaddedBounds[2];
    localData.Hull.PaddedBounds[3] = this->PaddedBounds[3];
    localData.Hull.PaddedBounds[4] = this->PaddedBounds[4];
    localData.Hull.PaddedBounds[5] = this->PaddedBounds[5];
    localData.Hull.Bounds[0] = this->Bounds[0];
    localData.Hull.Bounds[1] = this->Bounds[1];
    localData.Hull.Bounds[2] = this->Bounds[2];
    localData.Hull.Bounds[3] = this->Bounds[3];
    localData.Hull.Bounds[4] = this->Bounds[4];
    localData.Hull.Bounds[5] = this->Bounds[5];
    localData.Hull.H[0] = this->H[0];
    localData.Hull.H[1] = this->H[1];
    localData.Hull.H[2] = this->H[2];
    localData.Hull.Padding2 = this->Padding * this->Padding;
    localData.Hull.BoundaryCapping = this->BoundaryCapping;

    // Specify the contour label set. These are used to determine
    // if a label is part of the background, or a used to produce
    // output surface net contours.
    this->LMap.Local() =
      vtkLabelMapLookup<int>::CreateLabelLookup(this->LabelValues, this->NumLabels);
  }

  void operator()(vtkIdType batchId, vtkIdType endBatchId)
  {
    const BatchManager &batcher = this->Batcher;
    vtkIntArray *regionIds = this->RegionIds;
    vtkIdList*& pIds = this->PIds.Local();
    vtkDoubleArray*& radii2 = this->Radii2.Local();
    LocalDataType& localData = this->LocalData.Local();
    HullVertexType& localPts = localData.LocalPoints;
    TopoCoordType& localTups = localData.LocalTuples;
    CellConnType& localConn = localData.LocalCellConn;
    WheelsType& wheels = this->Wheels;
    BatchInfoType& lBatches = localData.LocalBatches;
    SpokeType& lSpokes = localData.LocalSpokes;
    SurfaceInformationType& surfInfo = this->SurfInfo;
    VHull& hull = localData.Hull;
    bool isFirst = vtkSMPTools::GetSingleThread();
    vtkLabelMapLookup<int>* lMap = this->LMap.Local();
    vtkIdType checkAbortInterval = std::min((endBatchId - batchId) / 10 + 1, (vtkIdType)1000);

    // Process the hullgenerating points in batches. This performs
    // a little better than independent point-by-point processing, and saves
    // some memory as well.
    for (; batchId < endBatchId; ++batchId)
    {
      // Support algorithm interrupts
      if (batchId % checkAbortInterval == 0)
      {
        if (isFirst)
        {
          this->Filter->CheckAbort();
        }
        if (this->Filter->GetAbortOutput())
        {
          break;
        }
      }

      // Process all points in this batch. Record the batch being
      // processed. Remember that the point ids are contiguous in this
      // batch.
      vtkIdType ptId, endPtId;
      batcher.GetBatchItemRange(batchId, ptId, endPtId);
      const double* x = this->Points + 3 * ptId;
      vtkIdType totalHullPts=0;

      for (; ptId < endPtId; ++ptId, x += 3)
      {
        // If the generating point is an outside region, we do not need to
        // process this hull.
        if ( regionIds && regionIds->GetValue(ptId) < 0 )
        {
          continue;
        }

        // Initialize the Voronoi hull
        hull.Initialize(ptId, x);

        // If hull is successfully built, copy the convex hull polyhedron and
        // related information to thread local storage.
        int maxClips = ( this->MaxClips < this->NPts ? this->MaxClips :
                         (this->NPts > 1 ? (this->NPts-1) : 0) );
        if (hull.BuildHull(pIds, radii2, this->Points, maxClips))
        {
          // Now accumulate the hull-related information in this thread.
          // First gather the neighborhood points (i.e., spokes) that
          // generated this hull.  Also classify the spokes - this will be
          // useful later when adding surface primitive related information.
          int numSpokes;
          const Spoke* spokes =
            hull.AddAdjacencyInformation(ptId,wheels,lSpokes,numSpokes,lMap);

          // Add information about generated surface primitives.
          hull.AddSurfaceInformation(ptId,spokes,numSpokes, surfInfo,
                                     localPts, localTups, localConn);

        } // if hull successfully generated
      } // for all points in this batch
      lBatches.emplace_back(BatchInfo(batchId));
    } // for all batches of points
  }

  void Reduce()
  {
    // Delete all of the label map lookups. Need auto_ptr eventually.
    for (auto lmItr = this->LMap.begin(); lmItr != this->LMap.end(); ++lmItr)
    {
      delete *lmItr;
    } // over all threads

    // Build the wheels and spokes adjacency information.
    this->NumThreadsUsed = 0;
    this->NumSpokes = 0;

    // Gather information along with a prefix sum of some information
    // across all the threads.
    for ( auto& localData : this->LocalData)
    {
      this->ThreadMap.push_back(&localData);
      this->NumSpokes += localData.LocalSpokes.size();
      this->NumThreadsUsed++;
    } // loop over local thread output

    // Prefix sum over wheels to determine spokes offsets,
    // as well total number of spokes.
    vtkIdType offset, totalSpokes = 0;
    for ( vtkIdType id=0; id < this->NPts; ++id )
    {
      offset = this->Wheels[id];
      this->Wheels[id] = totalSpokes;
      totalSpokes += offset;
    }
    // Cap off the wheels offsets array.
    this->Wheels[this->NPts] = totalSpokes;

    // Now copy the spokes from local thread data to the global
    // spokes array.
    this->Spokes.resize(this->NumSpokes);

    // Parallel build the adjacency (wheel and spokes) structure.
    ProduceSpokes produceSpokes(this->Batcher,this->ThreadMap,this->Wheels,this->Spokes);
    vtkSMPTools::For(0,this->NumThreadsUsed, produceSpokes);

    // Now build offsets/prefix sum for the surface output.  The prefix sum
    // converts hull information into offsets for random access of
    // information.
    SurfaceInformation info, totalInfo;
    for ( vtkIdType id=0; id < this->NPts; ++id )
    {
      info = this->SurfInfo[id];
      this->SurfInfo[id] = totalInfo;
      totalInfo += info;
    }
    this->SurfInfo[this->NPts] = totalInfo;

    // If smoothing and/or point merging is requested, composite the
    // topological point 4-tuples, sort them, and then create a point
    // renumbering map. We do this prior to creating output points and
    // surface primitives so that when they are created, we can renumber them
    // appropriately.
    if ( this->MergePoints || this->Smoothing )
    {
      VCore::ProduceMergingInfo pmi(this);
      vtkSMPTools::For(0,this->NumThreadsUsed, pmi);
    }
  } // Reduce

  // If smoothing or merging, build the data structures to support this.
  // We use the topological 4-tuples to merge topologically coincident points.
  // Note that the order in which the tuples are processed here is implicitly
  // related to the subsequent compositing of Voronoi hull points. That is,
  // topological tuples ti corresponds to Voronoi hull vertices vi.
  struct ProduceMergingInfo
  {
    VCore* VC;

    ProduceMergingInfo(VCore* vc)
      : VC(vc)
    {
      // Use the surface information to allocate the merging array.
      const SurfaceInformationType& info = vc->SurfInfo;
      vtkIdType totalPts = info[vc->NPts].NumPts;
      vc->MergeTuples.resize(totalPts);
    }

    void Initialize() {}

    void operator()(vtkIdType threadId, vtkIdType endThreadId)
    {
      const BatchManager &batcher = this->VC->Batcher;
      const SurfaceInformationType& info = this->VC->SurfInfo;
      MergeTupleType& mergeTuples = this->VC->MergeTuples;

      for (; threadId < endThreadId; ++threadId)
      {
        // Get the current local thread data. Also get indices into
        // the local data
        LocalDataType& localData = *(this->VC->ThreadMap[threadId]);
        HullVertexType::iterator pItr = localData.LocalPoints.begin();
        TopoCoordType::iterator tItr = localData.LocalTuples.begin();
        CellConnType::iterator cItr = localData.LocalCellConn.begin();

        // Loop over the batches that the current thread processed earlier. The batches
        // are ordered and consistent with the local data.
        for ( auto& batchInfo : localData.LocalBatches )
        {
          vtkIdType ptId, endPtId;
          vtkIdType numBatchPts = batcher.GetBatchItemRange(batchInfo.Id, ptId, endPtId);
          for (; ptId < endPtId; ++ptId) //produce tuples for this batch
          {
            vtkIdType numPts = (info[ptId+1].NumPts - info[ptId].NumPts);
            if ( numPts > 0 ) // process if tuples were produced by this point generator
            {
              vtkIdType startPtId = info[ptId].NumPts;
              vtkIdType pId = startPtId;
              for (int i=0; i<numPts; ++i, ++pId, ++tItr)
              {
                mergeTuples[pId].Ids = tItr->Ids; //composite 4-tuple
                mergeTuples[pId].HullVId = pId; //assign a global, unmerged point id
              }
            }
          } // for all points in this batch
        }   // for all batches
      }     // for all threads
    }

    // Now create the point renumbering map. It maps from the hull vertices
    // (which are disconnected and coincident) to the topologically merged
    // output points.
    void Reduce()
    {
      // Make sure there is data
      MergeTupleType& mergeTuples = this->VC->MergeTuples;
      if ( mergeTuples.size() <= 0 )
      {
        return;
      }

      // First we sort the tuples. This will create groups of 4-tuples with
      // the same tuple values. Each group is assigned an id (using a prefix
      // sum) to create the final point map.
      vtkSMPTools::Sort(mergeTuples.begin(),mergeTuples.end());

      // Count the number of merged points. Merged points have the same
      // hull vertex tuple ids.
      vtkIdType numMergedPts = 1;
      MergeTuple currentMT = mergeTuples[0];
      for (vtkIdType i=1; i < mergeTuples.size(); ++i)
      {
        if ( currentMT != mergeTuples[i] )
        {
          numMergedPts++;
          currentMT = mergeTuples[i];
        }
      } // for all hull vertex merge tuples

      // Update the total number of points in the output.
      this->VC->NumMergedPts = numMergedPts;

      // Allocate merge map, and populate it with the merged point ids.
      MergeMapType& mergeMap = this->VC->MergeMap;
      mergeMap.resize(mergeTuples.size());

      // Traverse the hull vertex tuples again and record the merged point
      // id of each tuple.
      vtkIdType currentMergeId = 0;
      currentMT = mergeTuples[0];
      for (vtkIdType i=0; i < mergeTuples.size(); ++i)
      {
        if ( currentMT != mergeTuples[i] )
        {
          ++currentMergeId;
          currentMT = mergeTuples[i];
        }
        mergeMap[mergeTuples[i].HullVId] = currentMergeId;
      } // for all hull vertex merge tuples
    } // Reduce

  }; // ProduceMergingInfo

  // A factory method to conveniently instantiate and execute the algorithm core.
  static std::unique_ptr<VCore>
  Execute(vtkStaticPointLocator *loc, vtkPoints *inPts, vtkIntArray *regionIds,
          double padding, vtkIdType maxClips, int &numThreads,
          vtkGeneralizedSurfaceNets3D* filter)
  {
    // Set up batch processing and process the data. This
    BatchManager batcher(inPts->GetNumberOfPoints(),filter->GetBatchSize());

    // Generate the Voronoi tessellation. The core algorithm contains output
    // information used later.
    auto vc = std::make_unique<VCore>(batcher, inPts, regionIds, loc,
                                      padding, maxClips, filter);

    // Threaded process batches of points.
    vtkSMPTools::For(0,batcher.GetNumberOfBatches(), *vc);

    // Update some global information.
    numThreads = vc->NumThreadsUsed;

    // Okay get out
    return vc;
  }
}; // VCore

// Superclass for classes that produce VTK output. Note that the output
// classes must be consistent with the information gathered previously or
// memory issues will result.
struct VOutput
{
  const VCore* VC;
  double* OutPoints;
  vtkIdType* Conn;
  vtkIdType* ConnOffsets;
  vtkIdType* CellScalars; //2-tuple regions on either side

  VOutput(const VCore* vc)
    : VC(vc)
    , OutPoints(nullptr)
    , Conn(nullptr)
    , ConnOffsets(nullptr)
    , CellScalars(nullptr) {}

  // Add a point to the output
  void AddPoint(vtkIdType ptId, double *x)
  {
    double *p = this->OutPoints + 3*ptId;
    *p++ = x[0];
    *p++ = x[1];
    *p++ = x[2];
  }

  // Add a merged point to the output. We just write
  // the value of the first vertex hull point - it's possible
  // to average these conincident points - maybe if necessary.
  void AddMergedPoint(const MergeMapType& mergeMap,
                      PtsWrittenFlags& ptsWritten,
                      vtkIdType ptId, double *x)
  {
    vtkIdType pId = mergeMap[ptId];
    if ( !ptsWritten[pId] )
    {
      double *p = this->OutPoints + 3*pId;
      *p++ = x[0];
      *p++ = x[1];
      *p++ = x[2];
      ptsWritten[pId] = 1;
    }
  }

  // Add a primitive cell to the output. This should be
  // followed by AddPrimPoint() calls.
  void AddPrim(vtkIdType primId, vtkIdType connOffset)
  {
    this->ConnOffsets[primId] = connOffset;
  }

  // Add a primitive cell point to the output
  void AddPrimPoint(vtkIdType connOffset, vtkIdType ptId)
  {
    this->Conn[connOffset] = ptId;
  }

  // Add a merged primitive cell point to the output
  void AddMergedPrimPoint(const MergeMapType& mergeMap,
                          vtkIdType connOffset, vtkIdType ptId)
  {
    vtkIdType pId = mergeMap[ptId];
    this->Conn[connOffset] = pId;
  }

}; // VOutput

// Class responsible for generating output polydata.
struct SurfaceOutput : public VOutput
{
  vtkPolyData* Output;
  std::unique_ptr<PtsWrittenFlags> PtsWritten;

  SurfaceOutput(const VCore* vc, vtkPolyData *output)
    : VOutput(vc)
    , Output(output)
  {
    // Allocate some point merging related structure if necessary.
    if ( vc->MergePoints || vc->Smoothing )
    {
      this->PtsWritten = std::unique_ptr<PtsWrittenFlags>
        (new PtsWrittenFlags(vc->NumMergedPts,0));
    }
  }

  // Retrieve information for a specified hull. Invoke this after the prefix sum.
  void GetSurfaceInformation(vtkIdType ptId, vtkIdType& numPts, vtkIdType& numPrims,
                             vtkIdType& connSize, vtkIdType& startPtId,
                             vtkIdType& startPrimId, vtkIdType& startConn)
  {
    const VCore* vc = this->VC;
    numPts = (vc->SurfInfo[ptId+1].NumPts - vc->SurfInfo[ptId].NumPts);
    numPrims = (vc->SurfInfo[ptId+1].NumPrims - vc->SurfInfo[ptId].NumPrims);
    connSize = (vc->SurfInfo[ptId+1].ConnSize - vc->SurfInfo[ptId].ConnSize);
    startPtId = vc->SurfInfo[ptId].NumPts;
    startPrimId = vc->SurfInfo[ptId].NumPrims;
    startConn = vc->SurfInfo[ptId].ConnSize;
  }

  // Produce polygonal output for the generating point specified.
  void ProduceSurfacePrims(vtkIdType threadId, vtkIdType ptId,
                           HullVertexType::iterator& pItr, CellConnType::iterator& cItr)
  {
    // Retrieve offset information
    const VCore* vc = this->VC;
    vtkIdType numPts, numPrims, connSize;
    vtkIdType startPtId, startPrimId, startConn;
    this->GetSurfaceInformation(ptId, numPts, numPrims, connSize,
                                startPtId, startPrimId, startConn);

    // If nothing is to be produced, return.
    if ( numPrims <= 0 )
    {
      return;
    }

    // Might need this to mark one side of a surface primitive outside.
    int background = this->VC->BackgroundLabel;

    // Point merging may be in effect
    bool merging = vc->Smoothing || vc->MergePoints;
    const MergeMapType& mergeMap = vc->MergeMap;
    PtsWrittenFlags& ptsWritten = *(this->PtsWritten);

    // Output the points
    vtkIdType pId = startPtId;
    if ( merging )
    {
      for (int i=0; i<numPts; ++i, ++pId, ++pItr)
      {
        this->AddMergedPoint(mergeMap, ptsWritten, pId, pItr->X);
      }
    }
    else
    {
      for (int i=0; i<numPts; ++i, ++pId, ++pItr)
      {
        this->AddPoint(pId, pItr->X);
      }
    }

    // Output the cell connectivity. Note that the cell point ids need to be
    // transformed into global point id space. Also output optional cell data.
    vtkIdType primId = startPrimId;
    for (int i=0; i<numPrims; ++primId, ++i)
    {
      this->AddPrim(primId,startConn);
      vtkIdType numPrimPts = *cItr++;
      vtkIdType backfaceId = *cItr++;
      if ( merging )
      {
        for (int j=0; j < numPrimPts; ++j)
        {
          this->AddMergedPrimPoint(mergeMap, startConn++, startPtId + *cItr++);
        }
      }
      else
      {
        for (int j=0; j < numPrimPts; ++j)
        {
          this->AddPrimPoint(startConn++, startPtId + *cItr++);
        }
      }

      // Scalars are 2-tuples, region ids on either side of face prim.
      // Order the tuples so that (s0<s1).
      vtkIdType s0 = this->VC->RegionIds->GetValue(ptId);
      vtkIdType s1 = ( backfaceId >= 0 ? this->VC->RegionIds->GetValue(backfaceId) : background );
      if ( s0 > s1 && s1 >= 0 ) //make sure s1 is non-outside region
      {
        std::swap(s0,s1);
      }
      this->CellScalars[2*primId] = s0;
      this->CellScalars[2*primId + 1] = s1;
    } // for all output cell primitives
  }

  // Each thread transforms and writes its own data.
  void operator()(vtkIdType threadId, vtkIdType endThreadId)
  {
    const BatchManager &batcher = this->VC->Batcher;
    bool isFirst = vtkSMPTools::GetSingleThread();
    vtkIdType checkAbortInterval = std::min((endThreadId - threadId) / 10 + 1, (vtkIdType)1000);

    // Loop over all threads
    for (; threadId < endThreadId; ++threadId)
    {
      // Support algorithm interrupts
      if (threadId % checkAbortInterval == 0)
      {
        if (isFirst)
        {
          this->VC->Filter->CheckAbort();
        }
        if (this->VC->Filter->GetAbortOutput())
        {
          break;
        }
      }

      // Get the current local thread data. Also get indices into
      // the local data
      LocalDataType& localData = *(this->VC->ThreadMap[threadId]);
      HullVertexType::iterator pItr = localData.LocalPoints.begin();
      CellConnType::iterator cItr = localData.LocalCellConn.begin();

      // Loop over the batches that the current thread processed earlier. The batches
      // are ordered and consistent with the local data vectors.
      for ( auto& batchInfo : localData.LocalBatches )
      {
        vtkIdType ptId, endPtId;
        vtkIdType numBatchPts = batcher.GetBatchItemRange(batchInfo.Id, ptId, endPtId);
        for (; ptId < endPtId; ++ptId) //output all data in this batch
        {
          this->ProduceSurfacePrims(threadId, ptId, pItr, cItr);
        } // for all points in this batch
      }   // for all batches
    }     // for all threads
  }       // operator()

  // A factory method to instantiate and threaded execute an instance
  // of SurfaceOutput to produce polygonal output.
  static void Execute(const VCore* vc, vtkPolyData* output)
  {
    // Grab the global surface information.
    const SurfaceInformationType& info = vc->SurfInfo;

    // Create the output dataset arrays (points and cells) and allocate them.
    // The number of points varies depending on whether point merging has been
    // performed.
    vtkNew<vtkPoints> outPts;
    outPts->SetDataTypeToDouble();
    if ( vc->MergePoints || vc->Smoothing ) //point merging needed
    {
      outPts->SetNumberOfPoints(vc->NumMergedPts);
    }
    else // no point merging
    {
      outPts->SetNumberOfPoints(info[vc->NPts].NumPts);
    }

    // Instantiate the surface output class.
    SurfaceOutput so(vc,output);
    so.OutPoints = vtkDoubleArray::FastDownCast(outPts->GetData())->GetPointer(0);

    // The polygonal faces are assembled manually from the connectivity list and
    // offsets.
    vtkNew<vtkIdTypeArray> conn;
    conn->SetNumberOfTuples(info[vc->NPts].ConnSize);
    so.Conn = conn->GetPointer(0);
    vtkNew<vtkIdTypeArray> offsets;
    offsets->SetNumberOfTuples(info[vc->NPts].NumPrims+1);
    so.ConnOffsets = offsets->GetPointer(0);
    so.ConnOffsets[info[vc->NPts].NumPrims] = info[vc->NPts].ConnSize; //cap off the offsets array

    // Assemble the output
    vtkNew<vtkCellArray> prims;
    prims->SetData(offsets,conn);
    output->SetPoints(outPts);
    output->SetPolys(prims);

    // Generate the output cell data 2-tuple, noting region ids on either
    // side of each polygonal face.
    vtkNew<vtkIdTypeArray> cellScalars;
    cellScalars->SetName("Surface Net Scalars");
    cellScalars->SetNumberOfComponents(2);
    cellScalars->SetNumberOfTuples(info[vc->NPts].NumPrims);
    int idx = output->GetCellData()->AddArray(cellScalars);
    output->GetCellData()->SetActiveAttribute(idx, vtkDataSetAttributes::SCALARS);
    so.CellScalars = cellScalars->GetPointer(0);

    // Now parallel thread the creation of the surface output.
    vtkSMPTools::For(0,vc->NumThreadsUsed, so);

  };

}; //SurfaceOutput

} // anonymous namespace


//================= Begin VTK class proper =====================================
//------------------------------------------------------------------------------
// Construct object
vtkGeneralizedSurfaceNets3D::vtkGeneralizedSurfaceNets3D()
{
  this->Labels = vtkSmartPointer<vtkContourValues>::New();

  this->BoundaryCapping = true;
  this->MergePoints = true;

  this->Smoothing = true;
  this->Smoother = vtkSmartPointer<vtkConstrainedSmoothingFilter>::New();
  this->Smoother->SetNumberOfIterations(16);
  this->Smoother->SetRelaxationFactor(0.5);

  this->Padding = 0.001;
  this->Locator = vtkSmartPointer<vtkStaticPointLocator>::New();
  this->Locator->SetNumberOfPointsPerBucket(2);
  this->PointOfInterest = (-1);
  // this->PointsOfInterest empty on instantiation
  this->MaximumNumberOfHullClips = VTK_ID_MAX;
  this->BatchSize = 1000;
  this->NumberOfThreadsUsed = 0;

  // By default process active point scalars to obtain region ids
  this->SetInputArrayToProcess(
    0, 0, 0, vtkDataObject::FIELD_ASSOCIATION_POINTS, vtkDataSetAttributes::SCALARS);
}

//------------------------------------------------------------------------------
int vtkGeneralizedSurfaceNets3D::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector, vtkInformationVector* outputVector)
{
  // get the info objects
  vtkInformation* inInfo = inputVector[0]->GetInformationObject(0);
  vtkInformation* outInfo0 = outputVector->GetInformationObject(0);

  // get the input and output
  vtkPointSet* input = vtkPointSet::SafeDownCast(inInfo->Get(vtkDataObject::DATA_OBJECT()));
  vtkPolyData* output = vtkPolyData::SafeDownCast(outInfo0->Get(vtkDataObject::DATA_OBJECT()));

  vtkDebugMacro(<< "Generating 3D Generalized Surface Net");

  // Check the input, at least one point is needed.
  vtkIdType pId, numPts;
  vtkPoints* inPoints;
  if ((inPoints = input->GetPoints()) == nullptr || (numPts = inPoints->GetNumberOfPoints()) < 1)
  {
    vtkDebugMacro("Cannot tessellate; need at least 1 input point");
    return 1;
  }

  // Input points must be of type double
  vtkSmartPointer<vtkPoints> tPoints;
  if (inPoints->GetDataType() == VTK_DOUBLE)
  { // fast path no conversion
    tPoints = inPoints;
  }
  else
  { // convert points to double
    tPoints = vtkSmartPointer<vtkPoints>::New();
    tPoints->SetDataTypeToDouble();
    tPoints->SetNumberOfPoints(numPts);
    for (pId = 0; pId < numPts; ++pId)
    {
      tPoints->SetPoint(pId, inPoints->GetPoint(pId));
    }
  }

  // Temporary data object holds points to be tessellated
  vtkNew<vtkPolyData> tInput;
  tInput->SetPoints(tPoints);

  // A locator is used to locate closest points.
  if (!this->Locator)
  {
    vtkErrorMacro(<< "Point locator required\n");
    return 0;
  }
  this->Locator->SetDataSet(tInput);
  this->Locator->BuildLocator();

  // Computational bounds and the padded bounding box
  double length = input->GetLength();
  double padding = this->Padding * length;

  // Region ids can be used to control which input points are processed.
  // A region id < 0 means that the associated point is "outside" and does
  // not contribute to the output. We can use this capability to process a
  // specified "PointOfInterest" (if any). Otherwise, we check the input for
  // segmented regions via a regions ids array.
  //
  // If region ids are provided,  array must be a single component tuple,
  // signed integer of type vtkIntArray with the number of tuples == number
  // of input points. (Implementation note: this could be expanded with
  // templates - not sure its worth the object bloat.)
  vtkSmartPointer<vtkIntArray> regionIds;

  // Limit processing to points of interested if so specified.
  if ( (this->PointOfInterest >= 0 && this->PointOfInterest < numPts) ||
       this->PointsOfInterest )
  {
    regionIds = vtkSmartPointer<vtkIntArray>::New();
    regionIds->SetName("Points of Interest");
    regionIds->SetNumberOfTuples(numPts);
    vtkSMPTools::Fill(regionIds->GetPointer(0), regionIds->GetPointer(0) + numPts, -100);
    if ( this->PointOfInterest >= 0 )
    {
      regionIds->SetValue(this->PointOfInterest,numPts); // mark POI in region numPts
    }
    if ( this->PointsOfInterest )
    {
      vtkIdType numPOI = this->PointsOfInterest->GetNumberOfTuples();
      for ( vtkIdType i=0; i < numPOI; ++i)
      {
        vtkIdType poi = this->PointsOfInterest->GetValue(i);
        if ( poi >= 0 && poi < numPts )
        {
          regionIds->SetValue(poi,numPts); // mark POI in region numPts
        }
      }
    }
  }
  else
  {
    vtkDataArray* rIds = this->GetInputArrayToProcess(0, inputVector);
    regionIds = vtkIntArray::FastDownCast(rIds);
    if ( rIds && !regionIds )
    {
      vtkErrorMacro("Region Ids array must be of type vtkIntArray");
    }
    if ( regionIds )
    {
      if ( regionIds->GetNumberOfComponents() > 1 )
      {
        vtkErrorMacro("Region Ids must have 1 component");
        regionIds = nullptr;
      }
    }
  }

  // Process the points to generate Voronoi information, including the adjaceny
  // (wheels and spokes) data structure. Information is also gathered to allocate
  // memory for the output, and then generate the VTK filter output.
  double* inPtr = vtkDoubleArray::FastDownCast(tPoints->GetData())->GetPointer(0);
  std::unique_ptr<VCore> vc = VCore::
    Execute(this->Locator, tPoints, regionIds, padding,
            this->MaximumNumberOfHullClips, this->NumberOfThreadsUsed, this);

  // With the information gathered, now build the surface net.
  SurfaceOutput::Execute(vc.get(),output);

  return 1;
}

//------------------------------------------------------------------------------
int vtkGeneralizedSurfaceNets3D::FillInputPortInformation(int, vtkInformation* info)
{
  info->Set(vtkAlgorithm::INPUT_REQUIRED_DATA_TYPE(), "vtkPointSet");
  return 1;
}

//------------------------------------------------------------------------------
// Overload standard modified time function. Since users have access to
// the locator, smoother, and contour labels, we need to take into account
// the modified time of each instance.
vtkMTimeType vtkGeneralizedSurfaceNets3D::GetMTime()
{
  vtkMTimeType mTime = this->Superclass::GetMTime();
  vtkMTimeType labelsTime = this->Labels->GetMTime();
  vtkMTimeType locatorTime = this->Locator->GetMTime();
  vtkMTimeType smootherTime = this->Smoother->GetMTime();

  mTime = (mTime > labelsTime ? mTime : labelsTime);
  mTime = (mTime > locatorTime ? mTime : locatorTime);
  mTime = (mTime > smootherTime ? mTime : smootherTime);

  return mTime;
}

//------------------------------------------------------------------------------
void vtkGeneralizedSurfaceNets3D::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);

  this->Labels->PrintSelf(os, indent.GetNextIndent());

  os << indent << "Boundary Capping: " << (this->BoundaryCapping ? "On\n" : "Off\n");
  os << indent << "Merge Points: " << (this->MergePoints ? "On\n" : "Off\n");

  os << indent << "Smoothing: " << (this->Smoothing ? "On\n" : "Off\n");
  os << indent << "Smoother: " << this->Smoother.Get() << endl;

  os << indent << "Padding: " << this->Padding << "\n";
  os << indent << "Locator: " << this->Locator << "\n";
  os << indent << "Point Of Interest: " << this->PointOfInterest << "\n";
  os << indent << "Points Of Interest: " << this->PointsOfInterest << "\n";
  os << indent << "Maximum Number Of Hull Clips: " << this->MaximumNumberOfHullClips << "\n";
  os << indent << "Batch Size: " << this->BatchSize << "\n";
}

VTK_ABI_NAMESPACE_END
