/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkCentroidalVoronoi3D.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class   vtkCentroidalVoronoi3D
 * @brief   create a centroidal 3D Voronoi tessellation of input points
 *
 * vtkCentroidalVoronoi3D is a filter that builds on vtkVoronoi3D by iteratively
 * adjusting a subset of the input point coordinates so that input points lie at
 * or near the centroid of their Voronoi cells.
 *
 * This iterative process is not guaranteed to terminate, so the filter provides
 * several termination criteria; the first one to be met terminates iteration:
 * + when a maximum number of iterations has been performed;
 * + when all input points are within an absolute distance of their cell center;
 * + when all input points are within a Voronoi-cell-size-relative distance of their
 *   cell center; or
 * + when the maximum distance any input point is moved during the most recent
 *   iteration is smaller than an absolute distance.
 *
 * You may constrain a subset of input points to be fixed (say, in order to
 * preserve the boundary of a region) while allowing other input points to be
 * relocated each iteration.
 *
 * @warning
 * This class has been threaded with vtkSMPTools. Using TBB or other
 * non-sequential type (set in the CMake variable
 * VTK_SMP_IMPLEMENTATION_TYPE) may improve performance significantly.
 *
 * @sa
 * vtkVoronoi3D vtkVoronoi2D
 */

#ifndef vtkCentroidalVoronoi3D_h
#define vtkCentroidalVoronoi3D_h

#include "vtkVoronoi3D.h"
#include "vtkNew.h" // For ivars.

#include <unordered_set>

VTK_ABI_NAMESPACE_BEGIN

class VTKFILTERSCORE_EXPORT vtkCentroidalVoronoi3D : public vtkVoronoi3D
{
public:
  /// A structure holding metrics used to terminate the algorithm's iteration.
  struct TerminationMetrics
  {
    double DistanceToCenter;
    double VoronoiCellSize;
    double NormalizedDistanceToCenter;
    double DistanceMoved;
  };

  ///@{
  /**
   * Standard methods for instantiation, type information, and printing.
   */
  static vtkCentroidalVoronoi3D* New();
  vtkTypeMacro(vtkCentroidalVoronoi3D, vtkVoronoi3D);
  void PrintSelf(ostream& os, vtkIndent indent) override;
  ///@}

  /// Set/get the maximum number of iterations to perform.
  ///
  /// The initial value is 20.
  vtkSetClampMacro(MaximumNumberOfIterations, int, 1, VTK_INT_MAX);
  vtkGetMacro(MaximumNumberOfIterations, int);

  /// Set/get a termination condition based on the largest distance
  /// between any input point and its Voronoi center.
  ///
  /// If the distance above is smaller than AbsoluteDistanceToCenter,
  /// then iteration is terminated.
  ///
  /// The default is VTK_DOUBLE_MAX.
  vtkSetClampMacro(AbsoluteDistanceToCenter, double, 0, VTK_DOUBLE_MAX);
  vtkGetMacro(AbsoluteDistanceToCenter, double);

  /// Set/get a termination condition based on the largest distance
  /// between any input point and its Voronoi center, normalized by
  /// the diameter of its Voronoi cell.
  ///
  /// If the distance above is smaller than NormalizedDistanceToCenter,
  /// then iteration is terminated.
  ///
  /// The initial value is 0.05.
  vtkSetClampMacro(NormalizedDistanceToCenter, double, 0, VTK_DOUBLE_MAX);
  vtkGetMacro(NormalizedDistanceToCenter, double);

  /// Set/get a termination condition based on the largest distance traveled
  /// by any input point during the current iteration.
  ///
  /// If this distance is larger than MovementCutoff, the algorithm will
  /// terminate.
  ///
  /// The initial value is VTK_DOUBLE_MAX.
  vtkSetClampMacro(MovementCutoff, double, 0, VTK_DOUBLE_MAX);
  vtkGetMacro(MovementCutoff, double);

  /// Set/get a scaling factor for moving input points each iteration.
  ///
  /// Each iteration, input points are moved along the line between their
  /// current position (in that iteration) and the centroid of their Voronoi
  /// cell. The MovementCoefficient determines how far along the line each
  /// point moves. A value of 0 prevents motion. A value of 1 moves each
  /// input point to the center of its Voronoi center each iteration.
  /// Usually coefficient in the range of 0.2 to 0.8 is used to prevent
  /// unstable oscillations.
  ///
  /// The default value is 0.25.
  vtkSetClampMacro(MovementCoefficient, double, 0, 1);
  vtkGetMacro(MovementCoefficient, double);

  /// Set/get an array of point IDs to be fixed during iteration.
  ///
  /// Points with these IDs will not be moved.
  vtkSetObjectMacro(FixedInputs, vtkIdTypeArray);
  vtkGetObjectMacro(FixedInputs, vtkIdTypeArray);

  /// Set an array to use as flags indicating whether an input point may be moved or not.
  ///
  /// If specified, \a FixedInputs is ignored.
  void SetConstraintFlagArray(int fieldAssociation, const char* name)
  { this->SetInputArrayToProcess(1, 0, 0, fieldAssociation, name); }
  void SetConstraintFlagArray(const char* fieldAssociation, const char* attributeTypeOrName)
  { this->SetInputArrayToProcess(1, 0, 0, fieldAssociation, attributeTypeOrName); }

protected:
  vtkCentroidalVoronoi3D();
  ~vtkCentroidalVoronoi3D() override;

  int RequestData(vtkInformation*, vtkInformationVector**, vtkInformationVector*) override;

  /// Copy this object's parameters (Padding, Locator, etc) to \a InternalVoronoi.
  ///
  /// If \a isFinal is true, then all of the parameters are copied.
  /// Otherwise, a subset of parameters are configured for iteration
  /// (validation and spoke pruning turned off; no point data passed; etc.)
  /// so that \a InternalVoronoi can iterate quickly.
  void CopyParametersToInternal(bool isFinal);

  /// Update the \a generators with cell center information from \a this->InternalVoronoi.
  bool IterateCenters(vtkPolyData* generators, TerminationMetrics& metrics);

  int MaximumNumberOfIterations;
  double AbsoluteDistanceToCenter;
  double NormalizedDistanceToCenter;
  double MovementCutoff;
  double MovementCoefficient;
  vtkIdTypeArray* FixedInputs;
  vtkNew<vtkVoronoi3D> InternalVoronoi;
  std::unordered_set<vtkIdType> FixedPointIds;

private:
  vtkCentroidalVoronoi3D(const vtkCentroidalVoronoi3D&) = delete;
  void operator=(const vtkCentroidalVoronoi3D&) = delete;
};

VTK_ABI_NAMESPACE_END

#endif
