/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkCentroidalVoronoi3D.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkCentroidalVoronoi3D.h"

#include "vtkCellArray.h"
#include "vtkCellData.h"
#include "vtkDoubleArray.h"
#include "vtkIdTypeArray.h"
#include "vtkInformation.h"
#include "vtkInformationVector.h"
#include "vtkPointData.h"
#include "vtkPolyData.h"
#include "vtkSMPThreadLocalObject.h"
#include "vtkSMPTools.h"
#include "vtkStaticPointLocator.h"
#include "vtkStreamingDemandDrivenPipeline.h"
#include "vtkTriangle.h"
#include "vtkUnstructuredGrid.h"
#include "vtkVector.h"

#include <vector>

VTK_ABI_NAMESPACE_BEGIN

vtkStandardNewMacro(vtkCentroidalVoronoi3D);

namespace {

class UpdateWorker
{
public:
  UpdateWorker(
    vtkCentroidalVoronoi3D* filter,
    vtkDataArray* genPtsArr,
    vtkDataArray* cellCenters,
    vtkCentroidalVoronoi3D::TerminationMetrics* metrics)
    : Self(filter)
    , CurrentCenters(genPtsArr)
    , ComputedCenters(cellCenters)
    , FinalMetrics(metrics)
  {
  }

  void Initialize()
  {
    auto& metrics(this->Metrics.Local());
    metrics = { VTK_DOUBLE_MIN, VTK_DOUBLE_MAX, VTK_DOUBLE_MIN, VTK_DOUBLE_MIN };
  }

  void operator () (vtkIdType begin, vtkIdType end)
  {
    auto& metrics(this->Metrics.Local());
    vtkVector3d x0;
    vtkVector3d x1;
    vtkVector3d xx;
    vtkVector3d delta;
    vtkVector3d wdelta;
    double d2c = VTK_DOUBLE_MIN;
    double vcs = VTK_DOUBLE_MAX;
    double nd2c = VTK_DOUBLE_MIN;
    double dmv = VTK_DOUBLE_MIN;
    double w1 = this->Self->GetMovementCoefficient();
    for (vtkIdType ii = begin; ii < end; ++ii)
    {
      this->CurrentCenters->GetTuple(ii, x0.GetData());
      this->ComputedCenters->GetTuple(ii, x1.GetData());
      delta = (x1 - x0);
      wdelta = delta * w1;
      xx = x0 + wdelta;
      this->CurrentCenters->SetTuple(ii, xx.GetData());
      d2c = delta.Norm();
      if (d2c > metrics.DistanceToCenter) { metrics.DistanceToCenter = d2c; }
      if (dmv > metrics.DistanceMoved) { metrics.DistanceMoved = dmv; }
      // TODO: Handle vcs and nd2c.
    }
  }

  void Reduce()
  {
    *this->FinalMetrics = { VTK_DOUBLE_MIN, VTK_DOUBLE_MAX, VTK_DOUBLE_MIN, VTK_DOUBLE_MIN };
    for (const auto& metrics : this->Metrics)
    {
      if (this->FinalMetrics->DistanceToCenter < metrics.DistanceToCenter)
      {
        this->FinalMetrics->DistanceToCenter = metrics.DistanceToCenter;
      }
      if (this->FinalMetrics->VoronoiCellSize > metrics.VoronoiCellSize)
      {
        this->FinalMetrics->VoronoiCellSize = metrics.VoronoiCellSize;
      }
      if (this->FinalMetrics->NormalizedDistanceToCenter < metrics.NormalizedDistanceToCenter)
      {
        this->FinalMetrics->NormalizedDistanceToCenter = metrics.NormalizedDistanceToCenter;
      }
      if (this->FinalMetrics->DistanceMoved < metrics.DistanceMoved)
      {
        this->FinalMetrics->DistanceMoved = metrics.DistanceMoved;
      }
    }
  }

protected:
  vtkCentroidalVoronoi3D* Self;
  vtkDataArray* CurrentCenters;
  vtkDataArray* ComputedCenters;
  vtkCentroidalVoronoi3D::TerminationMetrics* FinalMetrics;
  vtkSMPThreadLocal<vtkCentroidalVoronoi3D::TerminationMetrics> Metrics;
};

} // anonymous namespace

vtkCentroidalVoronoi3D::vtkCentroidalVoronoi3D()
{
  this->FixedInputs = nullptr;
}

vtkCentroidalVoronoi3D::~vtkCentroidalVoronoi3D()
{
  this->SetFixedInputs(nullptr);
}

void vtkCentroidalVoronoi3D::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);

  os << indent << "InternalVoronoi:\n";
  vtkIndent i2 = indent.GetNextIndent();
  this->InternalVoronoi->PrintSelf(os, i2);

  os << indent << "MaximumNumberOfIterations: " << this->MaximumNumberOfIterations << "\n";
  os << indent << "AbsoluteDistanceToCenter: " << this->AbsoluteDistanceToCenter << "\n";
  os << indent << "NormalizedDistanceToCenter: " << this->NormalizedDistanceToCenter << "\n";
  os << indent << "MovementCutoff: " << this->MovementCutoff << "\n";
  os << indent << "MovementCoefficient: " << this->MovementCoefficient << "\n";
  os << indent << "FixedInputs: " << this->FixedInputs << "\n";
  os << indent << "FixedPointIds: " << this->FixedPointIds.size() << "\n";
}

void vtkCentroidalVoronoi3D::CopyParametersToInternal(bool isFinal)
{
  this->InternalVoronoi->SetOutputType(isFinal ? this->OutputType : vtkCentroidalVoronoi3D::VORONOI);
  this->InternalVoronoi->SetPadding(this->Padding);
  this->InternalVoronoi->SetValidate(isFinal ? this->Validate : false);
  // this->InternalVoronoi->SetLocator(this->Locator);
  this->InternalVoronoi->SetPassPointData(isFinal ? this->PassPointData : false);
  this->InternalVoronoi->SetGeneratePointScalars(isFinal ? this->GeneratePointScalars : NO_POINT_SCALARS);
  this->InternalVoronoi->SetGenerateCellScalars(isFinal ? this->GenerateCellScalars : POINT_IDS);
  this->InternalVoronoi->SetPointOfInterest(isFinal ? this->PointOfInterest : -1);
  this->InternalVoronoi->SetPointsOfInterest(isFinal ?  this->PointsOfInterest : nullptr);
  this->InternalVoronoi->SetMaximumNumberOfHullClips(this->MaximumNumberOfHullClips);
  this->InternalVoronoi->SetPruneSpokes(isFinal ? this->PruneSpokes : false);
  this->InternalVoronoi->SetPruneTolerance(this->PruneTolerance);
  this->InternalVoronoi->SetBatchSize(this->BatchSize);
}

bool vtkCentroidalVoronoi3D::IterateCenters(vtkPolyData* generators, TerminationMetrics& metrics)
{
  auto* genPtsArr = generators->GetPoints()->GetData();
  auto* cellCenters = this->InternalVoronoi->GetOutputDataObject(0)->GetAttributes(vtkDataObject::POINT)->GetScalars();

  UpdateWorker updater(this, genPtsArr, cellCenters, &metrics);
  vtkSMPTools::For(0, genPtsArr->GetNumberOfTuples(), updater);

  if (metrics.DistanceToCenter < this->AbsoluteDistanceToCenter ||
    metrics.DistanceMoved < this->MovementCutoff)
  {
    return true;
  }
#if 0
  // TODO:
  if (metrics.VoronoiCellSize > 0. && metrics.NormalizedDistanceToCenter > this->NormalizedDistanceToCenter)
  {
    return true;
  }
#endif
  return false;
}

int vtkCentroidalVoronoi3D::RequestData(vtkInformation* vtkNotUsed(request),
  vtkInformationVector** inputVector, vtkInformationVector* outputVector)
{
  // get the info objects
  vtkInformation* inInfo = inputVector[0]->GetInformationObject(0);
  vtkInformation* outInfo0 = outputVector->GetInformationObject(0);

  // get the input and output
  vtkPointSet* input = vtkPointSet::SafeDownCast(inInfo->Get(vtkDataObject::DATA_OBJECT()));
  vtkDataSet* output = vtkDataSet::SafeDownCast(outInfo0->Get(vtkDataObject::DATA_OBJECT()));

  vtkNew<vtkPoints> genPoints;
  genPoints->DeepCopy(input->GetPoints());
  vtkNew<vtkPolyData> generators;
  generators->SetPoints(genPoints);
  generators->GetPointData()->ShallowCopy(input->GetPointData());
  // TODO: Do same for cell data?
  this->InternalVoronoi->SetInputDataObject(0, generators);
  this->CopyParametersToInternal(false);
  // By default process active point scalars to obtain region ids
  auto* regionArray = this->GetInputArrayToProcess(0, inputVector);
  if (regionArray)
  {
    // TODO: What if regionArray is cell data?
    this->InternalVoronoi->SetInputArrayToProcess(
      0, 0, 0, vtkDataObject::FIELD_ASSOCIATION_POINTS, regionArray->GetName());
  }

  int iteration;
  TerminationMetrics metrics{ 0., 1., 0., 0. };
  for (iteration = 0; iteration < this->MaximumNumberOfIterations; ++iteration)
  {
    this->InternalVoronoi->Update();
    // Fetch computed centroid locations, then update genPoints and metrics.
    // If any metric should cause the algorithm to terminate, return true.
    if (this->IterateCenters(generators, metrics))
    {
      break;
    }
    generators->Modified(); // Force Update to re-run this->InternalVoronoi
  }

  // Re-run with final point locations and all filter parameters.
  this->CopyParametersToInternal(true);
  output->ShallowCopy(this->InternalVoronoi->GetOutputDataObject(0));
  return 1;
}

VTK_ABI_NAMESPACE_END
