//=============================================================================
//
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//
//=============================================================================
#include "smtk/simulation/truchas/utility/ModelUtils.h"

#include "smtk/common/UUID.h"
#include "smtk/model/Resource.h"

#include <iomanip>
#include <iostream>

namespace smtk
{
namespace simulation
{
namespace truchas
{

ModelUtils::ModelUtils()
{
}

void ModelUtils::renameModelEntities(smtk::model::ResourcePtr modelResource) const
{
  std::cout << "Renaming model volumes" << std::endl;
  this->renameModelEntitiesInternal(modelResource, smtk::model::VOLUME, "block ");

  std::cout << "Renaming model faces" << std::endl;
  this->renameModelEntitiesInternal(modelResource, smtk::model::FACE, "side set ");

  // Mark the resource as modified.
  modelResource->setClean(false);
}

void ModelUtils::renameModelEntitiesInternal(smtk::model::ResourcePtr modelResource,
  smtk::model::EntityTypeBits entType, const std::string& prefix) const
{
  auto entRefs = modelResource->findEntitiesOfType(entType, true);

  // First pass - find the max id
  int maxId = -1;
  for (auto entRef : entRefs)
  {
    auto uuid = entRef.entity();
    auto propList = modelResource->integerProperty(uuid, "pedigree id");
    if (propList.size() == 1)
    {
      int id = propList[0];
      maxId = id > maxId ? id : maxId;
    }
  } // for

  // Sanity check
  if (maxId < 0)
  {
    std::cerr << "No pedigree ids found" << std::endl;
    return;
  }

  // Get number of digits
  int number = maxId;
  int digits = 0;
  while (number)
  {
    number /= 10;
    digits++;
  }

  // Second pass - update entity names
  std::stringstream ss;
  for (auto entRef : entRefs)
  {
    auto uuid = entRef.entity();
    auto propList = modelResource->integerProperty(uuid, "pedigree id");
    if (propList.size() == 1)
    {
      int id = propList[0];
      ss.str("");
      ss << prefix << std::setfill('0') << std::setw(digits) << id << " - " << entRef.name();
      entRef.setName(ss.str());
    } // if
  }   // for (entRefs)
}

// end namespace smtk::simulation::truchas
}
}
}
