//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
// .NAME validators.h - functions for validating material attributes

#include "smtk/PublicPointerDefs.h"

#include "smtk/attribute/Attribute.h"
#include "smtk/attribute/DoubleItem.h"
#include "smtk/attribute/GroupItem.h"
#include "smtk/attribute/Item.h"
#include "smtk/attribute/Resource.h"

#include <set>
#include <string>
#include <vector>

bool isPhaseElementValid(smtk::attribute::GroupItemPtr phaseItem, std::size_t element);
bool isTransitionElementValid(smtk::attribute::GroupItemPtr groupItem, std::size_t element);

bool isAttributeValid(smtk::attribute::AttributePtr att)
{
  auto sharedItem = att->findAs<smtk::attribute::GroupItem>("shared-properties");
  if (!sharedItem->isValid())
  {
    return false;
  }

  auto phasesItem = att->findAs<smtk::attribute::GroupItem>("phases");
  for (std::size_t i = 0; i < phasesItem->numberOfGroups(); ++i)
  {
    if (!isPhaseElementValid(phasesItem, i))
    {
      return false;
    }
  }

  auto transitionsItem = att->findAs<smtk::attribute::GroupItem>("transitions");
  for (std::size_t i = 0; i < transitionsItem->numberOfGroups(); ++i)
  {
    if (!isTransitionElementValid(transitionsItem, i))
    {
      return false;
    }
  }

  return true;
}

bool isPhaseElementValid(smtk::attribute::GroupItemPtr phaseItem, std::size_t element)
{
  // Validating phase items requires some custom considerations:
  //  - Some phase items are excluded by the current categories
  //  - Multiphase items can be superseded by "shared properties".
  //  - Multiphase materials must specify a "name" for each phase.

  // Our logic presumes that the shared-properties group item has the same set of
  // property names as the phase item, and that categories match for each shared-
  // and corresponding phase-item.

  // Get the property item names that are both (i) member of the
  // current category set and (ii) *not* shared
  auto materialAtt = phaseItem->attribute();
  auto attResource = materialAtt->attributeResource();
  const std::set<std::string>& categoriesSet = attResource->categories();
  std::vector<std::string> categories(categoriesSet.begin(), categoriesSet.end());

  // Use the shared-properties item to get the list of names to check
  bool isMultiphase = phaseItem->numberOfGroups() > 1;
  auto sharedItem = materialAtt->findGroup("shared-properties");
  assert(sharedItem);
  std::set<std::string> namesToCheck;
  for (std::size_t i = 0; i < sharedItem->numberOfItemsPerGroup(); ++i)
  {
    auto item = sharedItem->item(i);
    if (!item->isMemberOf(categories))
    {
      continue; // omit if not in current categories
    }
    if (!isMultiphase || !item->isEnabled())
    {
      // If single phase or if not shared, include in the check list
      namesToCheck.insert(item->name());
    }
  } // for (i)

  // For multiphase materials, also add phase "name"
  if (isMultiphase)
  {
    namesToCheck.insert("name");
  }

  // Check items in the set
  for (auto iter = namesToCheck.begin(); iter != namesToCheck.end(); ++iter)
  {
    auto name = *iter;
    auto item = phaseItem->find(element, name, smtk::attribute::RECURSIVE_ACTIVE);
    if (item && !item->isValid())
    {
      return false;
    }
  } // for (iter)

  return true;
}

bool isTransitionElementValid(smtk::attribute::GroupItemPtr groupItem, std::size_t element)
{
  // Check individual items
  for (std::size_t i = 0; i < groupItem->numberOfItemsPerGroup(); ++i)
  {
    if (!groupItem->item(element, i)->isValid())
    {
      return false;
    }
  }

  // Check that low transition temp less than high transition temp
  if (groupItem->name() == "transitions")
  {
    double highTemp =
      groupItem->findAs<smtk::attribute::DoubleItem>(element, "upper-transition-temperature")
        ->value();
    double lowTemp =
      groupItem->findAs<smtk::attribute::DoubleItem>(element, "lower-transition-temperature")
        ->value();
    if (lowTemp >= highTemp)
    {
      return false;
    }
  } // if (transitions item)

  return true;
}
