//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================

#include "smtkTruchasMaterialsView.h"

// Truchas extension includes
#include "smtk/simulation/truchas/qt/qtAttributeListWidget.h"
#include "smtk/simulation/truchas/qt/qtMaterialAttribute.h"
#include "smtk/simulation/truchas/qt/qtPhasePropertyItem.h"
#include "smtk/simulation/truchas/qt/validators.h"

// SMTK includes
#include "smtk/attribute/Attribute.h"
#include "smtk/attribute/Resource.h"
#include "smtk/extension/qt/qtAttribute.h"
#include "smtk/extension/qt/qtItem.h"
#include "smtk/extension/qt/qtTableWidget.h"
#include "smtk/extension/qt/qtTypeDeclarations.h"
#include "smtk/extension/qt/qtUIManager.h"
#include "smtk/view/View.h"

#include <QDebug>
#include <QHBoxLayout>
#include <QHeaderView>
#include <QLabel>
#include <QPointer>
#include <QPushButton>
#include <QSizePolicy>
#include <QSplitter>
#include <QVBoxLayout>
#include <QVariant>

#include <string>
#include <vector>

class smtkTruchasMaterialsViewInternals
{
public:
  // Internal member data
  qtAttributeListWidget* ListFrame; // upper panel displaying list of attributes
  QPointer<QFrame> EditorFrame;     // lower panel displaying attribute editor
  QPointer<qtMaterialAttribute> CurrentAtt;

  smtk::view::View::Component AttComponent;
  std::string AttributeType;
}; // Internals

qtBaseView* smtkTruchasMaterialsView::createViewWidget(const ViewInfo& info)
{
  smtkTruchasMaterialsView* view = new smtkTruchasMaterialsView(info);
  view->buildUI();
  return view;
}

smtkTruchasMaterialsView::smtkTruchasMaterialsView(const ViewInfo& info)
  : qtBaseAttributeView(info)
{
  this->Internals = new smtkTruchasMaterialsViewInternals();
  int typesIndex = info.m_view->details().findChild("AttributeTypes");
  if (typesIndex < 0)
  {
    qWarning() << "ERROR: smtkTruchasMaterialsView missing AttributeTypes element";
    return;
  }
  smtk::view::View::Component typesComp = m_viewInfo.m_view->details().child(typesIndex);
  std::size_t i, n = typesComp.numberOfChildren();
  for (i = 0; i < n; i++)
  {
    smtk::view::View::Component& attComp = typesComp.child(i);
    if (attComp.name() == "Att")
    {
      this->Internals->AttComponent = attComp;
      break;
    }
  } // for (i)

  // Check that attribute Type is specified
  if (!this->Internals->AttComponent.attribute("Type", this->Internals->AttributeType))
  {
    qWarning()
      << "ERROR: smtkTruchasMaterialsView/AttributeTypes missing Att element or Type attribute";
  }
}

smtkTruchasMaterialsView::~smtkTruchasMaterialsView()
{
  delete this->Internals;
}

bool smtkTruchasMaterialsView::isEmpty() const
{
  // Returns true when categories are *not* enabled

  // Sanity check
  if (this->Internals->AttributeType.empty())
  {
    qWarning() << "Missing material attribute type in view specification";
    return true;
  }

  auto attResource = this->uiManager()->attResource();
  auto attDef = attResource->findDefinition(this->Internals->AttributeType);
  auto& categories = attDef->categories();
  bool pass = this->uiManager()->passCategoryCheck(categories);
  return !pass;
}

void smtkTruchasMaterialsView::onShowCategory()
{
  // Overall categories have been changed
  this->Internals->ListFrame->onShowCategory();
}

void smtkTruchasMaterialsView::updateModelAssociation()
{
  this->onShowCategory();
}

void smtkTruchasMaterialsView::createWidget()
{
  // Display a simplified version of qtAttributeView top frame
  auto attResource = this->uiManager()->attResource();
  auto materialDefinition = attResource->findDefinition(this->Internals->AttributeType);

  QSplitter* frame = new QSplitter(this->parentWidget());
  frame->setOrientation(Qt::Vertical);

  this->Internals->ListFrame = new qtAttributeListWidget(frame, this, materialDefinition);
  this->Internals->ListFrame->setAttributeValidator(isAttributeValid);

  // Attribute frame
  this->Internals->EditorFrame = new QFrame(frame);
  new QVBoxLayout(this->Internals->EditorFrame);
  //  this->Internals->EditorFrame->setVisible(0);

  frame->addWidget(this->Internals->ListFrame);
  frame->addWidget(this->Internals->EditorFrame);

  QObject::connect(this->Internals->ListFrame, &qtAttributeListWidget::attributeCreated, this,
    &qtBaseAttributeView::attributeCreated);
  QObject::connect(this->Internals->ListFrame, &qtAttributeListWidget::attributeRemoved, this,
    &qtBaseAttributeView::attributeRemoved);

  // We want the signals that may change the attribute to be displayed Queued instead of
  // Direct so that QLineEdit::editingFinished signals are processed prior to these.
  QObject::connect(this->Internals->ListFrame, &qtAttributeListWidget::attributeSelectionChanged,
    this, &smtkTruchasMaterialsView::onAttributeSelectionChanged, Qt::QueuedConnection);
  QObject::connect(this->Internals->ListFrame, &qtAttributeListWidget::attributeNameChanged, this,
    &qtBaseAttributeView::attributeChanged, Qt::QueuedConnection);

  this->Widget = frame;
} // createWidget()

void smtkTruchasMaterialsView::onAttributeChanged()
{
  auto* qAtt = dynamic_cast<qtMaterialAttribute*>(this->sender());
  assert(qAtt);
  auto att = qAtt->attribute();
  this->Internals->ListFrame->checkStatus(att);
}

void smtkTruchasMaterialsView::onAttributeSelectionChanged(smtk::attribute::AttributePtr att)
{
  this->Internals->EditorFrame->setVisible(true);
  if (this->Internals->CurrentAtt != nullptr)
  {
    this->Internals->EditorFrame->layout()->removeWidget(this->Internals->CurrentAtt->widget());
  }
  delete this->Internals->CurrentAtt;
  this->Internals->CurrentAtt = nullptr;

  if (att != nullptr)
  {
    this->Internals->CurrentAtt = new qtMaterialAttribute(
      att, this->Internals->AttComponent, this->Internals->EditorFrame, this);

    this->Internals->CurrentAtt->createBasicLayout(false);
    if (this->Internals->CurrentAtt->widget())
    {
      this->Internals->EditorFrame->layout()->addWidget(this->Internals->CurrentAtt->widget());
      if (this->advanceLevelVisible())
      {
        this->Internals->CurrentAtt->showAdvanceLevelOverlay(true);
      }
    }

    QObject::connect(this->Internals->CurrentAtt, &qtMaterialAttribute::itemModified, this,
      &smtkTruchasMaterialsView::onItemChanged, Qt::QueuedConnection);
    QObject::connect(this->Internals->CurrentAtt, &qtMaterialAttribute::modified, this,
      &smtkTruchasMaterialsView::onAttributeChanged, Qt::QueuedConnection);
  } // if (att)
}

void smtkTruchasMaterialsView::onItemChanged(smtk::extension::qtItem* qitem)
{
  // qDebug() << "onItemChanged";
  if (qitem == nullptr)
  {
    return;
  }
  auto item = qitem->item();
  if (item == nullptr)
  {
    return;
  }
  auto attPtr = item->attribute();
  if (attPtr == nullptr)
  {
    return;
  }

  this->Internals->ListFrame->checkStatus(attPtr);
  this->valueChanged(item);
  std::vector<std::string> itemNames { item->name() };
  this->attributeChanged(attPtr, itemNames);
}
