//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#include "smtk/io/LoadJSON.h"
#include "smtk/io/Logger.h"
#include "smtk/io/SaveJSON.h"
#include "smtk/io/SaveJSON.txx"

#include "smtk/model/Manager.h"

#include "smtk/common/testing/cxx/helpers.h"
#include "smtk/model/testing/cxx/helpers.h"

#include "smtk/io/jsonEntity.h"
#include "smtk/io/jsonModelCollection.h"

#include "cJSON.h"

#include <fstream>
#include <iostream>
#include <string>

#include <string.h>

using namespace smtk::io;
using namespace smtk::model;
using namespace smtk::common;
using smtk::shared_ptr;

int testExportEntityRef(const EntityRefs& entities, IteratorStyle relations, int correctCount)
{
  cJSON* json = cJSON_CreateObject();
  SaveJSON::forEntities(json, entities, relations, JSON_ENTITIES);
  int numRecords = 0;
  for (cJSON* child = json->child; child; child = child->next)
    ++numRecords;
  if (numRecords != correctCount)
  { // For debugging, print out (names of) what we exported:
    if (!entities.empty())
      entities.begin()->manager()->assignDefaultNames();
    std::cout << SaveJSON::forEntities(entities, relations, JSON_PROPERTIES) << "\n\n"
              << "Exported " << numRecords << ","
              << " expecting " << correctCount << " for related record enum " << relations << "\n";
  }
  cJSON_Delete(json);
  test(numRecords == correctCount, "Exported wrong number of records.");
  return 0;
}

int testModelExport()
{
  using json = nlohmann::json;
  int status = 0;
  ManagerPtr sm = Manager::create();
  UUIDArray uids = smtk::model::testing::createTet(sm);

  json stuff = sm->topology();
  std::cout << "json for manager topo is\n" << stuff.dump(2) << "\n";

  UUIDArray::size_type modelStart = uids.size();
  uids.push_back(sm->addModel().entity());
  sm->findEntity(uids[21])->relations().push_back(uids[modelStart]);
  sm->findEntity(uids[modelStart])->relations().push_back(uids[21]);
  EntityRefs entities;
  entities.insert(EntityRef(sm, uids[8])); // An edge

  status |= testExportEntityRef(entities, smtk::model::ITERATE_BARE, 1);
  status |= testExportEntityRef(entities, smtk::model::ITERATE_CHILDREN, 9);
  status |= testExportEntityRef(entities, smtk::model::ITERATE_MODELS, 79);

  //std::string json = SaveJSON::forEntities(entities, smtk::model::ITERATE_BARE, JSON_DEFAULT);
  std::vector<EntityPtr> entrecs;
  for (auto eref : entities)
  {
    EntityPtr eptr;
    if (eref.isValid(&eptr))
    {
      entrecs.push_back(eptr);
    }
  }
  json arr = entrecs;
  std::cout << "json for vertex is \n" << arr.dump(2) << "\n";
  return status;
}

int testReadWrite(int debug, const std::string& filename)
{
  using njson = nlohmann::json;

  std::ifstream file(filename.c_str());
  std::string data((std::istreambuf_iterator<char>(file)), (std::istreambuf_iterator<char>()));
  cJSON* parsed = cJSON_Parse(data.c_str());
  cJSON* json = cJSON_CreateObject();

  ManagerPtr sm = Manager::create();

  int status = 0;

  status |= LoadJSON::ofLocalSession(parsed->child, sm, /*loadNativeModels*/ true);
  //status |= LoadJSON::intoModelManager(data.c_str(), sm);
  njson jmmgr = sm;
  std::cout << "json for manager is\n" << jmmgr.dump(2) << "\n";

  status |= SaveJSON::fromModelManager(json, sm,
    // Do not export sessions; they will have different UUIDs
    static_cast<JSONFlags>(JSON_ENTITIES | JSON_TESSELLATIONS | JSON_PROPERTIES));

  char* exported = cJSON_Print(json);
  cJSON_Delete(json);
  json = cJSON_CreateObject();
  ManagerPtr sm2 = Manager::create();

  status |= LoadJSON::intoModelManager(exported, sm2);
  status |= SaveJSON::fromModelManager(json, sm2,
    // Do not export sessions; they will have different UUIDs
    static_cast<JSONFlags>(JSON_ENTITIES | JSON_TESSELLATIONS | JSON_PROPERTIES));
  char* exported2 = cJSON_Print(json);

  if (debug || strcmp(exported, exported2))
  {
    std::cout << "====== snip =======\n";
    std::cout << exported << "\n";
    std::cout << "====== snip =======\n";
    std::cout << exported2 << "\n";
    std::cout << "====== snip =======\n";
    test(strcmp(exported, exported2) == 0, "double import/export pass not exact");
  }
  cJSON_Delete(json);
  free(exported);
  free(exported2);

  return status;
}

int main(int argc, char* argv[])
{
  int debug = argc > 2 ? 1 : 0;
  std::string testFile = argc > 1 ? argv[1] : "testOut";
  int status = 0;
  status |= testModelExport();
  status |= testReadWrite(debug, testFile);
  return status;
}
