//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#pragma once

#include "smtk/io/jsonArrangement.h"
#include "smtk/io/jsonTessellation.h"
#include "smtk/io/jsonUUID.h"

#include "smtk/model/Arrangement.h"
#include "smtk/model/EntityIterator.h"
#include "smtk/model/EntityRef.h"
#include "smtk/model/Manager.h"
#include "smtk/model/Model.h"

#include "nlohmann/json.hpp"

// Define how model-entity iterators are serialized.
namespace nlohmann
{
  template<>
  struct adl_serializer<smtk::model::EntityIterator>
  {
    using UUID = smtk::common::UUID;
    using UUIDArray = smtk::common::UUIDArray;
    using Entity = smtk::model::Entity;
    using EntityRef = smtk::model::EntityRef;
    using EntityPtr = smtk::model::EntityPtr;
    using KindsToArrangements = smtk::model::KindsToArrangements;
    using Models = smtk::model::Models;
    using Model = smtk::model::Model;
    using ManagerPtr = smtk::model::ManagerPtr;
    using EntityIterator = smtk::model::EntityIterator;
    using EntityTypeBits = smtk::model::EntityTypeBits;
    using IteratorStyle = smtk::model::IteratorStyle;
    using Tessellation = smtk::model::Tessellation;

    static void to_json(json& j, const EntityIterator& iter)
    {
      using smtk::model::AbbreviationForArrangementKind;
      EntityIterator mit(iter);
      j = json::object();

      for (mit.begin(); !mit.isAtEnd(); ++mit)
      {
        EntityRef ent = mit.current();
        EntityPtr eptr;
        if (ent.isValid(&eptr))
        {
          json jent = {
            { "d", eptr->dimension() },
            { "e", eptr->entityFlags() }, // Entity::flagToSpecifierString(eptr->entityFlags())
            { "r", eptr->relations() }
          };
          // Add arrangement information:
          const KindsToArrangements& amap(eptr->arrangementMap());
          if (!amap.empty())
          {
            json jarr = json::object();
            for (auto ktoa : amap)
            {
              jarr[AbbreviationForArrangementKind(ktoa.first)] = ktoa.second;
            }
            jent["a"] = jarr;
          }
          // Add string/float/int properties:
          if (ent.hasStringProperties())
          {
            jent["s"] = ent.stringProperties();
          }
          if (ent.hasFloatProperties())
          {
            jent["f"] = ent.floatProperties();
          }
          if (ent.hasIntegerProperties())
          {
            jent["i"] = ent.integerProperties();
          }
          /* Do not include tessellation here
          const Tessellation* tess;
          if ((tess = ent.hasTessellation()))
          {
            jent["t"] = *tess;
          }
           */
          j[ent.entity().toString()] = jent;
        }
      }
    }
  };
}
