//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#pragma once

#include "smtk/common/UUID.h"
#include "smtk/mesh/Handle.h"

#include "nlohmann/json.hpp"

namespace smtk
{
namespace mesh
{
namespace detail
{

#define MOAB_TYPE_WIDTH 4
#define MOAB_ID_WIDTH (8 * sizeof(smtk::mesh::Handle) - MOAB_TYPE_WIDTH)
#define MOAB_TYPE_MASK ((smtk::mesh::Handle)0xF << MOAB_ID_WIDTH)
#define MOAB_ID_MASK (~MOAB_TYPE_MASK)

inline ::moab::EntityID to_id(smtk::mesh::Handle handle)
{
  return (handle & MOAB_ID_MASK);
}

inline smtk::mesh::Handle to_handle(::moab::EntityType type, ::moab::EntityID id)
{
  if (type > ::moab::MBMAXTYPE)
  {
    return smtk::mesh::Handle(); //<You've got to return something.  What do you return?
  }
  return (((smtk::mesh::Handle)type) << MOAB_ID_WIDTH) | id;
}

#undef MOAB_TYPE_WIDTH
#undef MOAB_ID_WIDTH
#undef MOAB_TYPE_MASK
#undef MOAB_ID_MASK

} // namespace detail
} // namespace mesh
} // namespace smtk

namespace nlohmann
{
  template<>
  struct adl_serializer<smtk::mesh::HandleRange>
  {
    static void to_json(json& j, const smtk::mesh::HandleRange& range)
    {
      using const_pair_iterator = smtk::mesh::HandleRange::const_pair_iterator;

      j = json::object();
      for (::moab::EntityType i = ::moab::MBVERTEX; i != ::moab::MBMAXTYPE; ++i)
      {
        smtk::mesh::HandleRange subset = range.subset_by_type(i);
        if (subset.empty())
        {
          continue;
        }

        //build the name
        buffer << i;
        buffer >> typeAsString;
        buffer.clear();

        json jsubset = json::array();
        for (const_pair_iterator i = range.begin(); i != range.end(); ++i)
        {
          ::moab::EntityID start = to_id(i->first);
          ::moab::EntityID end = to_id(i->second);

          jsubset.push_back(start);
          jsubset.push_back(end);
        }
        j[typeAsString] = jsubset;
      }
    }

    static void from_json(const json& j, smtk::mesh::HandleRange& result)
    {
      smtk::mesh::HandleRange result;
      if (!json)
      {
        return result;
      }

      std::stringstream buffer;
      // Iterate the children
      for (auto jit = j.begin(); jit != j.end(); ++jit)
      {
        if (jit->is_array())
        {
          //extract the name
          int type;
          buffer << std::string(jit.key());
          buffer >> type;
          buffer.clear();

          smtk::mesh::HandleRange subset;
          ::moab::EntityType et = static_cast< ::moab::EntityType>(type);
          for (auto rit = jit->begin(); rit != jit->end(); ++rit)
          {
            ::moab::EntityID beg = smtk::mesh::detail::to_handle(et, rit->get<::moab::EntityID>());
            ++rit;
            ::moab::EntityID end = smtk::mesh::detail::to_handle(et, rit->get<::moab::EntityID>());
            subset.insert(beg, end);
          }
          result.merge(subset);
        }
      }
      return result;
    }
  };
}
