//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#include "smtk/simulation/windtunnel/qt/qtOpenFoamRunner.h"

#include <QDebug>
#include <QDialog>
#include <QDialogButtonBox>
#include <QDir>
#include <QFile>
#include <QFont>
#include <QMessageBox>
#include <QPlainTextEdit>
#include <QProcess>
#include <QProcessEnvironment>
#include <QTextStream>
#include <QTimer>
#include <QVBoxLayout>
#include <QWidget>

#include <nlohmann/json.hpp>

namespace
{
// Singleton instance
static smtk::simulation::windtunnel::qtOpenFoamRunner* g_instance = nullptr;
} // namespace

namespace smtk
{
namespace simulation
{
namespace windtunnel
{

qtOpenFoamRunner* qtOpenFoamRunner::instance(QObject* parent)
{
  if (!g_instance)
  {
    g_instance = new qtOpenFoamRunner(parent);
  }

  if (g_instance->parent() == nullptr && parent)
  {
    g_instance->setParent(parent);
  }

  return g_instance;
}

qtOpenFoamRunner::qtOpenFoamRunner(QObject* parent)
  : QObject(parent)
  , m_processDir(new QDir)
  , m_logFile(new QFile(this))
  , m_textEdit(new QPlainTextEdit)
{
  m_textEdit->setReadOnly(true);
  m_textEdit->setPlaceholderText("Checking for logfile...");
  const QFont fixedFont = QFontDatabase::systemFont(QFontDatabase::FixedFont);
  m_textEdit->document()->setDefaultFont(fixedFont);
}

qtOpenFoamRunner::~qtOpenFoamRunner()
{
  if (m_logFile->isOpen())
  {
    m_logFile->close();
  }
}

void qtOpenFoamRunner::setParentWidget(QWidget* widget)
{
  m_parentWidget = widget;

  // Initialize m_logDialog
  m_logDialog = new QDialog(widget);
  m_logDialog->resize(640, 480);
  m_logDialog->setModal(false);
  m_logDialog->setSizeGripEnabled(true);
  // Set to always be on top?

  QVBoxLayout* dialogLayout = new QVBoxLayout;
  dialogLayout->addWidget(m_textEdit);
  QDialogButtonBox* buttonBox = new QDialogButtonBox(QDialogButtonBox::Close);
  dialogLayout->addWidget(buttonBox);
  m_logDialog->setLayout(dialogLayout);

  // Set connections
  QObject::connect(buttonBox, &QDialogButtonBox::accepted, m_logDialog, &QDialog::accept);
  QObject::connect(buttonBox, &QDialogButtonBox::rejected, m_logDialog, &QDialog::reject);
  QObject::connect(m_logDialog, &QDialog::finished, this, &qtOpenFoamRunner::onLogDialogClosed);
}

qint64 qtOpenFoamRunner::start(
  const QString& application,
  const QString& projectFolder,
  const QString& caseFolder,
  const QStringList& arguments)
{
  if (this->state() != ProcessState::NotRunning)
  {
    qWarning() << __FILE__ << __LINE__ << "QProcess is already running";
    return 0;
  }

  // Set up process directory
  QDir processDir(projectFolder);
  processDir.mkdir("proc");
  processDir.cd("proc");
  m_processDir->setPath(processDir.absolutePath());

  if (processDir.exists("run.sh") && !processDir.exists("run.exitcode"))
  {
    qWarning() << "Internal Warning: expected run.exitcode file";
  }

  if (!processDir.isEmpty())
  {
    qInfo() << "Clearing \"proc\" directory";
    QFile qfile;
    QStringList filenames = { "run.sh", "run.exitcode", "run.info" };
    Q_FOREACH (QString name, filenames)
    {
      qfile.setFileName(processDir.absoluteFilePath(name));
      qfile.remove();
    }
  }

#if 1
  // Set up logs directory
  QDir logsDir(caseFolder);
  logsDir.mkdir("logs");
  logsDir.cd("logs");

  QString logFile = QString("%1.log").arg(application);
  QString logPath = logsDir.absoluteFilePath(logFile);
  // Should we use a separate file for stderr?
  QString exitcodePath = m_processDir->absoluteFilePath("run.exitcode");

  QFileInfo fileInfo(logPath);
  // qDebug() << __FILE__ << __LINE__ << m_logFile->fileName() << fileInfo.size();

  // Generate run.sh file
  QString runPath =
    this->createRunScript(application, arguments, caseFolder, logPath, exitcodePath);

  // Setup QProcess and launch in detached mode.
  QProcess process;
  process.setProgram(runPath);
  process.setWorkingDirectory(caseFolder);

  // Also set PWD in env - needed for some binary OpenFOAM installs
  QProcessEnvironment env = QProcessEnvironment::systemEnvironment();
  env.insert("PWD", caseFolder);
  process.setProcessEnvironment(env);

  bool started = process.startDetached(&m_pid);
  if (!started)
  {
    QString message = QString("Failed to launch %1").arg(application);
    QMessageBox::critical(m_parentWidget, "Error", message);
    return 0;
  }
  qDebug() << "qtOpenFoamRunner.cxx:" << __LINE__ << "Process started" << m_pid;
  this->createRunInfo(application, caseFolder, logPath);

  m_logDialog->setWindowTitle(logFile);
  m_logFile->setFileName(logPath);
#else
  // This is a hack to use Allrun script instead of hand-generated
  // Biggest problem is that there is no single log file to display
  QProcess process;

  QDir caseDir(caseFolder);
  QStringList argList;
  argList << "/" << QString("%1/Allrun").arg(caseDir.dirName());

  QString exitcodePath = m_processDir->absoluteFilePath("run.exitcode");
  // QString program = QString("openfoam-docker / %1/Allrun").arg(caseDir.dirName());
  QString program("openfoam-docker");
  qDebug() << "Program:" << program;
  process.setProgram(program);

  qDebug() << argList;
  process.setArguments(argList);

  QString foamFolder = QString("./%1/foam").arg(projectFolder);
  qDebug() << "Working directory:" << foamFolder;
  process.setWorkingDirectory(foamFolder);

  // Also set PWD in env - needed for some binary OpenFOAM installs
  QProcessEnvironment env = QProcessEnvironment::systemEnvironment();
  env.insert("PWD", foamFolder);
  process.setProcessEnvironment(env);

  bool started = process.startDetached(&m_pid);
  if (!started)
  {
    QString message = QString("Failed to launch %1").arg(program);
    QMessageBox::critical(m_parentWidget, "Error", message);
    return 0;
  }
  qDebug() << __FILE__ << __LINE__ << "Process started" << m_pid;
  this->createRunInfo(application, caseFolder, foamFolder);

  // m_logDialog->setWindowTitle(logFile);
  // m_logFile->setFileName(logPath);
#endif

  this->onProcessStarted();
  return m_pid;
}

void qtOpenFoamRunner::onProcessStarted()
{
  m_state = ProcessState::Running;
  Q_EMIT this->started(m_pid);

  m_textEdit->clear();
  m_logDialog->show(); // non-modal
  m_logDialog->raise();

  // Start polling for process and logfile updates
  m_logFilePollingOn = true;
  QTimer::singleShot(0, this, &qtOpenFoamRunner::checkStatus);
}

void qtOpenFoamRunner::showLogFile(const QString& path)
{
  if (this->state() == ProcessState::Running)
  {
    // Check if the input path is the m_logFile path
    QFileInfo inputInfo(path);
    QFileInfo runningInfo(m_logFile->fileName());
    if (inputInfo == runningInfo)
    {
      // Resume polling
      m_logFilePollingOn = true;

      // Todo reload m_textEdit?
      m_textEdit->clear();
      m_logDialog->show(); // non-modal
      m_logDialog->raise();
      QTimer::singleShot(0, this, &qtOpenFoamRunner::checkStatus);
      return;
    }
  }

  // (else) Turn off existing polling (if any)
  m_logFilePollingOn = false;
  if (m_logFile->isOpen())
  {
    m_logFile->close();
  }

  QFile file;
  file.setFileName(path);
  if (!file.open(QIODevice::ReadOnly | QIODevice::Text))
  {
    qWarning() << "Failed to open logfile" << path;
    return;
  }

  // We don't expect big (MB-size) files, so just read the whole thing
  QByteArray byteArray = file.readAll();
  QString text = byteArray.data();
  m_textEdit->setPlainText(text);
  m_logDialog->show(); // non-modal
  m_logDialog->raise();
}

void qtOpenFoamRunner::checkStatus()
{
  this->checkLogFile();

  // Check for run.exitcode file
  if ((m_state == ProcessState::Running) && m_processDir->exists("run.exitcode"))
  {
    // Read contents of run.exitcode file
    int exitCode = -999;
    QFile file(m_processDir->absoluteFilePath("run.exitcode"));
    if (file.size() > 0 && file.open(QFile::ReadOnly | QFile::Text))
    {
      QTextStream in(&file);
      QString content = in.readAll();
      bool ok = false;
      exitCode = content.trimmed().toInt(&ok);
      if (!ok)
      {
        qWarning() << "Unable to read exit code from" << content;
      }
    }

    qDebug() << "qtOpenFoamRunner.cxx:" << __LINE__ << "Process finished" << m_pid << exitCode;
    m_state = ProcessState::NotRunning;
    Q_EMIT this->finished(m_pid, exitCode);
  }

  // (else)
  QTimer::singleShot(997, this, &qtOpenFoamRunner::checkStatus);
}

void qtOpenFoamRunner::checkLogFile()
{
  if (!m_logFilePollingOn)
  {
    return;
  }

  // QFileInfo fileInfo(m_logFile->fileName());
  // qDebug() << __FILE__ << __LINE__ << m_logFile->fileName() << fileInfo.size();

  if (!m_logFile->exists())
  {
    qDebug() << __FILE__ << __LINE__;
    // Check later
    QTimer::singleShot(1000, this, &qtOpenFoamRunner::checkLogFile);
    return;
  }

  if (!m_logFile->isOpen())
  {
    // qDebug() << __FILE__ << __LINE__;
    if (!m_logFile->open(QIODevice::ReadOnly | QIODevice::Text))
    {
      qWarning() << "Failed to open logfile" << m_logFile->fileName();
      return;
    }
  }

  if (!m_logFile->isOpen())
  {
    qWarning() << __FILE__ << __LINE__ << "Internal warning; logfile not open";
    return;
  }

  // Check m_logFile for data
  m_logFile->waitForReadyRead(1.0);
  // qDebug() << __LINE__ << m_logFile->bytesAvailable();
  if (m_logFile->bytesAvailable() > 0)
  {
    QByteArray byteArray = m_logFile->readAll();
    QString text = byteArray.data();
    m_textEdit->appendPlainText(text);
    // qDebug() << __FILE__ << __LINE__ << text.size();
  }
}

void qtOpenFoamRunner::onLogDialogClosed()
{
  qDebug() << __LINE__ << "dialog closed";
  m_logFilePollingOn = false;
  if (m_logFile->isOpen())
  {
    m_logFile->close();
  }
}

QString qtOpenFoamRunner::createRunScript(
  const QString& application,
  const QStringList& arguments,
  const QString& caseFolder,
  const QString& logPath,
  const QString& exitcodePath) const
{
  QString appPrefix = m_useDocker ? "openfoam-docker / " : "";
  QString runScript;
  QTextStream qs(&runScript);
  qs << "#!/usr/bin/env bash\n"
     << "cd " << caseFolder << "\n";
  if (application == "refineMesh")
  {
    // Special case
    this->createRefineMeshRunScript(caseFolder, logPath, exitcodePath, qs);
  }
  else
  {
    qs << appPrefix << application << ' ' << arguments.join(" ") << " > " << logPath << " 2>&1 "
       << '\n'
       << "echo $? > " << exitcodePath << '\n';
  }

  QString runPath = m_processDir->absoluteFilePath("run.sh");
  QFile runFile(runPath);
  if (!runFile.open(QFile::WriteOnly))
  {
    QString message = "Error: Cannot create run.sh file.";
    QMessageBox::critical(m_parentWidget, "File System Error", message);
    return "";
  }
  // (else)
  QTextStream out(&runFile);
  out << runScript;
  runFile.close();

  runFile.setPermissions(
    runFile.permissions() | QFileDevice::ExeOwner | QFileDevice::ExeUser | QFileDevice::ExeOther);

  return runPath;
}

void qtOpenFoamRunner::createRefineMeshRunScript(
  const QString& caseFolder,
  const QString& logfilePath,
  const QString& exitcodePath,
  QTextStream& qs) const
{
  // The operation should have already created refineMeshDict and topoSetDict.* files
  // in the system subdirectory.
  QDir systemDir(caseFolder);
  systemDir.cd("system");
  QStringList nameFilters;
  nameFilters << "topoSetDict.*";
  QStringList topoSetList = systemDir.entryList(nameFilters);
  int count = topoSetList.size();
  if (count < 1)
  {
    qWarning() << "Did not find any topoSetDict.* files";
    return;
  }

  qs << "echo '' > " << logfilePath << '\n';

  qs << "for i in";
  for (int i = 1; i <= count; ++i)
  {
    qs << " " << i;
  }
  qs << '\n'
     << "do\n"
     << "    openfoam-docker / topoSet -dict system/topoSetDict.${i} >> " << logfilePath << '\n'
     << "    openfoam-docker / refineMesh -dict system/refineMeshDict -overwrite >> " << logfilePath
     << '\n'
     << "done\n"
     << "echo 0 > " << exitcodePath << '\n';
}

bool qtOpenFoamRunner::createRunInfo(
  const QString& application,
  const QString& caseFolder,
  const QString& logPath) const
{
  QString logfilePath = caseFolder + "/" + application + ".log";
  nlohmann::json j = { { "application", application.toStdString() },
                       { "case_directory", caseFolder.toStdString() },
                       { "logfile", logPath.toStdString() },
                       { "pid", m_pid } };
  QString infoString = j.dump(2).c_str();
  QString infoPath = m_processDir->absoluteFilePath("run.info");
  QFile infoFile(infoPath);
  if (!infoFile.open(QFile::WriteOnly))
  {
    QString message = "Error: Cannot create run.info file.";
    QMessageBox::critical(m_parentWidget, "File System Error", message);
    return false;
  }
  // (else)
  QTextStream infoOut(&infoFile);
  infoOut << infoString << '\n';
  infoFile.close();

  return true;
}

} // namespace windtunnel
} // namespace simulation
} // namespace smtk
