/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkAppendFilterIntIds.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class   vtkAppendFilterIntIds
 * @brief   appends one or more datasets together into a single unstructured grid
 *
 * vtkAppendFilterIntIds is a filter that appends one of more datasets into a single
 * unstructured grid. All geometry is extracted and appended, but point
 * attributes (i.e., scalars, vectors, normals, field data, etc.) are extracted
 * and appended only if all datasets have the point attributes available.
 * (For example, if one dataset has scalars but another does not, scalars will
 * not be appended.)
 *
 * You can decide to merge points that are coincident by setting
 * `MergePoints`. If this flag is set, points are merged if they are within
 * `Tolerance` radius. If a point global id array is available (point data named
 * "GlobalPointIds"), then two points are merged if they share the same point global id,
 * without checking for coincident point.
 *
 * Modified for AEVA to use vtkInt for global ids
 *
 * @sa
 * vtkAppendPolyData
 */

#ifndef vtk_aeva_ext_vtkAppendFilterIntIds_h
#define vtk_aeva_ext_vtkAppendFilterIntIds_h

#include "vtk/aeva/ext/AEVAExtModule.h"
#include "vtkUnstructuredGridAlgorithm.h"

class vtkDataSetAttributes;
class vtkDataSetCollection;

class AEVAEXT_EXPORT vtkAppendFilterIntIds : public vtkUnstructuredGridAlgorithm
{
public:
  static vtkAppendFilterIntIds* New();
  vtkTypeMacro(vtkAppendFilterIntIds, vtkUnstructuredGridAlgorithm);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  vtkAppendFilterIntIds(const vtkAppendFilterIntIds&) = delete;
  vtkAppendFilterIntIds& operator=(const vtkAppendFilterIntIds&) = delete;

  //@{
  /**
   * Get any input of this filter.
   */
  vtkDataSet* GetInput(int idx);
  vtkDataSet* GetInput() { return this->GetInput(0); }
  //@}

  //@{
  /**
   * Get/Set if the filter should merge coincidental points
   * Note: The filter will only merge points if the ghost cell array doesn't exist
   * Defaults to Off
   */
  vtkGetMacro(MergePoints, vtkTypeBool);
  vtkSetMacro(MergePoints, vtkTypeBool);
  vtkBooleanMacro(MergePoints, vtkTypeBool);
  //@}

  //@{
  /**
   * Get/Set the tolerance to use to find coincident points when `MergePoints`
   * is `true`. Default is 0.0.
   *
   * This is simply passed on to the internal vtkLocator used to merge points.
   * @sa `vtkLocator::SetTolerance`.
   */
  vtkSetClampMacro(Tolerance, double, 0.0, VTK_DOUBLE_MAX);
  vtkGetMacro(Tolerance, double);
  //@}

  //@{
  /**
   * Get/Set whether Tolerance is treated as an absolute or relative tolerance.
   * The default is to treat it as an absolute tolerance. When off, the
   * tolerance is multiplied by the diagonal of the bounding box of the input.
   */
  vtkSetMacro(ToleranceIsAbsolute, bool);
  vtkGetMacro(ToleranceIsAbsolute, bool);
  vtkBooleanMacro(ToleranceIsAbsolute, bool);
  //@}

  /**
   * Remove a dataset from the list of data to append.
   */
  void RemoveInputData(vtkDataSet* dataset);

  /**
   * Returns a copy of the input array.  Modifications to this list
   * will not be reflected in the actual inputs.
   */
  vtkDataSetCollection* GetInputList();

  //@{
  /**
   * Set/get the desired precision for the output types. See the documentation
   * for the vtkAlgorithm::Precision enum for an explanation of the available
   * precision settings.
   */
  vtkSetClampMacro(OutputPointsPrecision, int, SINGLE_PRECISION, DEFAULT_PRECISION);
  vtkGetMacro(OutputPointsPrecision, int);
  //@}

protected:
  vtkAppendFilterIntIds();
  ~vtkAppendFilterIntIds() override;

  // Usual data generation method
  int RequestData(vtkInformation* request,
    vtkInformationVector** inputVector,
    vtkInformationVector* outputVector) override;
  int RequestUpdateExtent(vtkInformation* request,
    vtkInformationVector** inputVector,
    vtkInformationVector* outputVector) override;
  int FillInputPortInformation(int port, vtkInformation* info) override;

  // list of data sets to append together.
  // Here as a convenience.  It is a copy of the input array.
  vtkDataSetCollection* InputList;

  // If true we will attempt to merge points. Must also not have
  // ghost cells defined.
  vtkTypeBool MergePoints;

  int OutputPointsPrecision;
  double Tolerance;

  // If true, tolerance is used as is. If false, tolerance is multiplied by
  // the diagonal of the bounding box of the input.
  bool ToleranceIsAbsolute;

private:
  // Get all input data sets that have points, cells, or both.
  // Caller must delete the returned vtkDataSetCollection.
  vtkDataSetCollection* GetNonEmptyInputs(vtkInformationVector** inputVector); // NOLINT

  void AppendArrays(int attributesType,
    vtkInformationVector** inputVector,
    vtkIdType* globalIds,
    vtkUnstructuredGrid* output,
    vtkIdType totalNumberOfElements);
};

#endif
