"""
Demodock utility library for commandline scripts.

The demodock python module implements several utility functions used to
locate configuration files, determine binaries to use for Nginx and Docker, and
perform a few other tasks common to the main script utilities.
"""

import datetime
import getpass
import os
import subprocess


def get_user():
    """
    Return the name of the invoking user.

    This function returns 'root' if the script was invoked with sudo.
    """
    return getpass.getuser()


def get_home_directory():
    """Return the home directory of the invoking user."""
    return os.expandpath("~")


def get_config_dir():
    """
    Query the environment and return the configuration directory.

    This function looks up DEMODOCK_CONFIG in the environment. If it is not set,
    a default directory based on the user's home directory is used.
    """
    config_dir = os.environ.get("DEMODOCK_CONFIG")
    if config_dir is None:
        config_dir = "%s/.config/demodock" % (get_home_directory())
    return config_dir


def get_docker_binary():
    """
    Query the environment and return the Docker binary.

    This function looks up DEMODOCK_DOCKER_BINARY in the environment. If it is
    not set, the default of 'docker' is used.
    """
    docker_binary = os.environ.get("DEMODOCK_DOCKER_BINARY")
    if docker_binary is None:
        docker_binary = "docker"
    return docker_binary


def get_nginx_binary():
    """
    Query the environment and return the Nginx binary.

    This function looks up DEMODOCK_NGINX_BINARY in the environment. If it is
    not set, the default of 'nginx' is used.
    """
    nginx_binary = os.environ.get("DEMODOCK_NGINX_BINARY")
    if nginx_binary is None:
        nginx_binary = "nginx"
    return nginx_binary


def get_timestamp():
    """
    Return a string representing the current moment.

    This function generates unique timestamps that can be used to disambiguate
    docker containers started with the same name.
    """
    return datetime.datetime.now().strftime("%Y%m%d_%H%M%S_%f")


def get_pid():
    """
    Return the PID of the running Nginx instance, or None if Nginx is not running.

    This function queries the Nginx PID file and converts its contents to an
    int.
    """
    pidfile = "%s/nginx/logs/nginx.pid" % (get_config_dir())
    try:
        with open(pidfile) as f:
            pid = int(f.read())
    except ValueError:
        pid = None

    return pid


def get_config_file():
    """Return the path to the active Nginx config file, or None if Nginx is not running."""
    confpointer = "%s/conf-file" % (get_config_dir())
    with open(confpointer) as f:
        conffile = f.read()

    return conffile


def docker_ps_lines():
    """Return docker-formatted text reporting the running demodock containers."""
    docker_bin = get_docker_binary()
    docker = subprocess.Popen([docker_bin, "ps"], stdout=subprocess.PIPE, stderr=subprocess.PIPE)

    code = docker.wait()
    (stdout, stderr) = docker.communicate()

    if code != 0:
        raise RuntimeError(stderr)

    # Split the output into lines, and filter out all lines (after the first)
    # that are not running an image prepended with "demodock/".
    lines = stdout.split("\n")
    demos = filter(lambda x: x and x.split()[1].startswith("demodock/"), lines[1:])

    return [lines[0]] + demos


def get_live_demos():
    """Return list of running demodock container IDs."""
    lines = docker_ps_lines()

    return map(lambda x: x.split()[0], lines[1:])
