/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkAnisotropicLandmarkTransform.h

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
/**
 * @class   vtkAnisotropicLandmarkTransform
 * @brief   a linear transform specified by two corresponding point sets
 *
 * @sa
 * vtkLandmarkTransform
*/

#ifndef vtkAnisotropicLandmarkTransform_h
#define vtkAnisotropicLandmarkTransform_h

#include "vtkCommonTransformsModule.h" // For export macro
#include "vtkLandmarkTransform.h"
#include "vtkMatrix3x3.h"


class VTKCOMMONTRANSFORMS_EXPORT vtkAnisotropicLandmarkTransform : public vtkLandmarkTransform
{
public:
  static vtkAnisotropicLandmarkTransform *New();

  vtkTypeMacro(vtkAnisotropicLandmarkTransform,vtkLandmarkTransform);
  void PrintSelf(ostream& os, vtkIndent indent) override;

  //@{
  /**
   * Set the error threshold to end the iterative anisotropic mode of
   * landmark registration. Root Mean Square Error is used to determine
   * the error between the source and target points. The algorithm will
   * halt when the error value is less than the threshold.
   */
  void SetThreshold(double threshold);
  //@}

  //@{
  /**
   * Get the rotation matrix independent of scaling.
   */
  vtkGetObjectMacro(RotationMatrix, vtkMatrix3x3);
  //@}

  //@{
  /**
   * Get the scaling matrix independent of rotation.
   */
  vtkGetObjectMacro(ScalingMatrix, vtkMatrix3x3);
  //@}

  /**
   * Invert the transformation. This is done by switching the
   * source and target landmarks.
   */
  void Inverse() override;

  /**
   * Get the MTime.
   */
  vtkMTimeType GetMTime() override;

  /**
   * Make another transform of the same type.
   */
  vtkAbstractTransform *MakeTransform() override;

protected:
  vtkAnisotropicLandmarkTransform();
  ~vtkAnisotropicLandmarkTransform() override;

  // Update the matrix from the quaternion.
  void InternalUpdate() override;

  /**
   * This method does no type checking, use DeepCopy instead.
   */
  void InternalDeepCopy(vtkAbstractTransform *transform) override;

  vtkMatrix3x3 *ScalingMatrix;
  vtkMatrix3x3 *RotationMatrix;

  double Threshold;

private:
  vtkAnisotropicLandmarkTransform(const vtkAnisotropicLandmarkTransform&) = delete;
  void operator=(const vtkAnisotropicLandmarkTransform&) = delete;
};


#endif
