/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkCoordinate.cxx
  Language:  C++
  Date:      $Date$
  Version:   $Revision$

Copyright (c) 1993-2000 Ken Martin, Will Schroeder, Bill Lorensen 
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice,
   this list of conditions and the following disclaimer.

 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.

 * Neither name of Ken Martin, Will Schroeder, or Bill Lorensen nor the names
   of any contributors may be used to endorse or promote products derived
   from this software without specific prior written permission.

 * Modified source versions must be plainly marked as such, and must not be
   misrepresented as being the original software.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS ``AS IS''
AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

=========================================================================*/

#include "vtkCoordinate.h"
#include "vtkViewport.h"
#include "vtkObjectFactory.h"



//------------------------------------------------------------------------------
vtkCoordinate* vtkCoordinate::New()
{
  // First try to create the object from the vtkObjectFactory
  vtkObject* ret = vtkObjectFactory::CreateInstance("vtkCoordinate");
  if(ret)
    {
    return (vtkCoordinate*)ret;
    }
  // If the factory was unable to create the object, then create it here.
  return new vtkCoordinate;
}




#define VTK_RINT(x) ((x > 0.0) ? (int)(x + 0.5) : (int)(x - 0.5))

// Creates an Coordinate with the following defaults: 
// value of  0, 0, 0 in world  coordinates
vtkCoordinate::vtkCoordinate()
{
  this->CoordinateSystem = VTK_WORLD;
  this->Value[0] = 0.0;
  this->Value[1] = 0.0;
  this->Value[2] = 0.0;
  this->Viewport = NULL;
  this->ReferenceCoordinate = NULL;
  this->Computing = 0;
}

// Destroy a Coordinate.  
vtkCoordinate::~vtkCoordinate()
{
  // To get rid of references (Refence counting).
  this->SetReferenceCoordinate(NULL);
  this->SetViewport(NULL);
}

void vtkCoordinate::PrintSelf(ostream& os, vtkIndent indent)
{
  this->vtkObject::PrintSelf(os,indent);

  char posString[64];
  switch (this->CoordinateSystem)
    {
    case VTK_DISPLAY:
      strcpy(posString, "Display"); break;
    case VTK_NORMALIZED_DISPLAY:
      strcpy(posString, "Normalized Display"); break;
    case VTK_VIEWPORT:
      strcpy(posString, "Viewport"); break;
    case VTK_NORMALIZED_VIEWPORT:
      strcpy(posString, "Normalized Viewport"); break;
    case VTK_VIEW:
      strcpy(posString, "View"); break;
    case VTK_WORLD:
      strcpy(posString, "World"); break;
    default:
	strcpy(posString, "UNKNOWN!"); break;
    }

  os << indent << "Coordinate System: " << posString << "\n";
  os << indent << "Value: (" << this->Value[0] << "," 
     << this->Value[1] << "," << this->Value[2] << ")\n";
  os << indent << "ReferenceCoordinate: " << this->ReferenceCoordinate << "\n";
  os << indent << "Viewport: " << this->Viewport << "\n";
}


void vtkCoordinate::SetViewport(vtkViewport *viewport)
{
  if (this->Viewport != viewport)
    {
    if (this->Viewport != NULL) 
      {
      this->Viewport->UnRegister(this);
      }
    this->Viewport = viewport;
    if (this->Viewport != NULL) 
      {
      this->Viewport->Register(this);
      }
    this->Modified();
    }
}






float *vtkCoordinate::GetComputedWorldValue(vtkViewport* viewport)
{
  float *val = this->ComputedWorldValue;
  
  // prevent infinite loops
  if (this->Computing)
    {
    return val;
    }
  this->Computing = 1;

  val[0] = this->Value[0];
  val[1] = this->Value[1];
  val[2] = this->Value[2];

  // use our viewport if set
  if (this->Viewport)
    {
    viewport = this->Viewport;
    }
  
  // if viewport is NULL then we can only do minimal calculations
  if (!viewport)
    {
    if (this->CoordinateSystem == VTK_WORLD)
      {
      if (this->ReferenceCoordinate)
	{
	float *RefValue;
	
	RefValue = this->ReferenceCoordinate->GetComputedWorldValue(viewport);
	val[0] += RefValue[0];
	val[1] += RefValue[1];
	val[2] += RefValue[2];
	}
      this->Computing = 0;
      }
    else
      {
      vtkErrorMacro("Attempt to compute world coordinates from another coordinate system without a viewport");
      }
    return val;
    }
  
  if (this->ReferenceCoordinate && this->CoordinateSystem != VTK_WORLD)
    {
    float RefValue[3];
    int *ival;
    
    ival = this->ReferenceCoordinate->GetComputedDisplayValue(viewport);
    RefValue[0] = (float)(ival[0]);
    RefValue[1] = (float)(ival[1]);
    RefValue[2] = (float)(ival[2]);
    
    // convert to current coordinate system
    switch (this->CoordinateSystem)
      {
      case VTK_NORMALIZED_DISPLAY:
	viewport->DisplayToNormalizedDisplay(RefValue[0],RefValue[1]);
	break;
      case VTK_VIEWPORT:
	viewport->DisplayToNormalizedDisplay(RefValue[0],RefValue[1]);
	viewport->NormalizedDisplayToViewport(RefValue[0],RefValue[1]);
	break;
      case VTK_NORMALIZED_VIEWPORT:
	viewport->DisplayToNormalizedDisplay(RefValue[0],RefValue[1]);
	viewport->NormalizedDisplayToViewport(RefValue[0],RefValue[1]);
	viewport->ViewportToNormalizedViewport(RefValue[0],RefValue[1]);
	break;
      case VTK_VIEW:
	viewport->DisplayToNormalizedDisplay(RefValue[0],RefValue[1]);
	viewport->NormalizedDisplayToViewport(RefValue[0],RefValue[1]);
	viewport->ViewportToNormalizedViewport(RefValue[0],RefValue[1]);
	viewport->NormalizedViewportToView(RefValue[0],RefValue[1],RefValue[2]);
	break;
      }
    
    // add to current value
    val[0] += RefValue[0];
    val[1] += RefValue[1];
    val[2] += RefValue[2];
    }
  
  // compute our WC
  switch (this->CoordinateSystem)
    {
    case VTK_DISPLAY:
      viewport->DisplayToNormalizedDisplay(val[0],val[1]);
    case VTK_NORMALIZED_DISPLAY:
      viewport->NormalizedDisplayToViewport(val[0],val[1]);
    case VTK_VIEWPORT:
      viewport->ViewportToNormalizedViewport(val[0],val[1]);
    case VTK_NORMALIZED_VIEWPORT:
      viewport->NormalizedViewportToView(val[0],val[1],val[2]);
    case VTK_VIEW:
      viewport->ViewToWorld(val[0],val[1],val[2]);
    }
  
  if (this->ReferenceCoordinate && this->CoordinateSystem == VTK_WORLD)
    {
    float *RefValue;
    
    RefValue = this->ReferenceCoordinate->GetComputedWorldValue(viewport);
    val[0] += RefValue[0];
    val[1] += RefValue[1];
    val[2] += RefValue[2];
    }
  
  this->Computing = 0;
  vtkDebugMacro("Returning WorldValue of : " << 
		this->ComputedWorldValue[0] << " , " << 
		this->ComputedWorldValue[1] << " , " << 
		this->ComputedWorldValue[2]);
  return val;
}



int *vtkCoordinate::GetComputedViewportValue(vtkViewport* viewport)
{
  // use our viewport if set
  if (this->Viewport)
    {
    viewport = this->Viewport;
    }

  int *d = this->GetComputedDisplayValue(viewport);

  if (!viewport)
    {
    vtkDebugMacro("Attempt to convert to compute viewport coordinates without a viewport, results may not be valid");
    return this->ComputedDisplayValue;
    }

  float f[2];
  
  f[0] = (float)d[0];
  f[1] = (float)d[1];
  
  viewport->DisplayToNormalizedDisplay(f[0],f[1]);
  viewport->NormalizedDisplayToViewport(f[0],f[1]);
  
  this->ComputedViewportValue[0] = (int)VTK_RINT(f[0]);
  this->ComputedViewportValue[1] = (int)VTK_RINT(f[1]);

  return this->ComputedViewportValue;
}

int *vtkCoordinate::GetComputedLocalDisplayValue(vtkViewport* viewport)
{
  float a[2];

  // use our viewport if set
  if (this->Viewport)
    {
    viewport = this->Viewport;
    }
  this->GetComputedDisplayValue(viewport);

  if (!viewport)
    {
    vtkErrorMacro("Attempt to convert to local display coordinates without a viewport");
    return this->ComputedDisplayValue;
    }
    
  a[0] = (float)this->ComputedDisplayValue[0];
  a[1] = (float)this->ComputedDisplayValue[1];
  
  viewport->DisplayToLocalDisplay(a[0],a[1]);

  this->ComputedDisplayValue[0] = (int)VTK_RINT(a[0]);
  this->ComputedDisplayValue[1] = (int)VTK_RINT(a[1]);

  vtkDebugMacro("Returning LocalDisplayValue of : " << 
		this->ComputedDisplayValue[0] << " , " << 
		this->ComputedDisplayValue[1]);
  
  return this->ComputedDisplayValue;
}

int *vtkCoordinate::GetComputedDisplayValue(vtkViewport* viewport)
{
  float val[3];

  // prevent infinite loops
  if (this->Computing)
    {
    return this->ComputedDisplayValue;
    }
  this->Computing = 1;

  val[0] = this->Value[0];
  val[1] = this->Value[1];
  val[2] = this->Value[2];

  // use our viewport if set
  if (this->Viewport)
    {
    viewport = this->Viewport;
    }

  // if viewport is NULL, there is very little we can do
  if (viewport == NULL)
    {
    // for DISPLAY and VIEWPORT just use the value
    if (this->CoordinateSystem == VTK_DISPLAY)
      {
      this->ComputedDisplayValue[0] = (int)VTK_RINT(val[0]);
      this->ComputedDisplayValue[1] = (int)VTK_RINT(val[1]);
      if (this->ReferenceCoordinate)
	{
	int *RefValue;
	
	RefValue = this->ReferenceCoordinate->GetComputedDisplayValue(viewport);
	this->ComputedDisplayValue[0] += RefValue[0];
	this->ComputedDisplayValue[1] += RefValue[1];
	}
      }
    else
      {
      vtkErrorMacro("Request for coordinate transformation without required viewport");
      }
    return this->ComputedDisplayValue;
    }
  
  // compute our DC
  switch (this->CoordinateSystem)
    {
    case VTK_WORLD:
      if (this->ReferenceCoordinate)
	{
	float *RefValue;
	
	RefValue = this->ReferenceCoordinate->GetComputedWorldValue(viewport);
	val[0] += RefValue[0];
	val[1] += RefValue[1];
	val[2] += RefValue[2];
	}
      viewport->WorldToView(val[0],val[1],val[2]);
    case VTK_VIEW:
      viewport->ViewToNormalizedViewport(val[0],val[1],val[2]);
    case VTK_NORMALIZED_VIEWPORT:
      viewport->NormalizedViewportToViewport(val[0],val[1]);
    case VTK_VIEWPORT:
      if ((this->CoordinateSystem == VTK_NORMALIZED_VIEWPORT ||
	   this->CoordinateSystem == VTK_VIEWPORT) &&
	  this->ReferenceCoordinate)
	{
	int *RefValue;
	
	RefValue = 
	  this->ReferenceCoordinate->GetComputedViewportValue(viewport);
	val[0] += RefValue[0];
	val[1] += RefValue[1];
	}
      viewport->ViewportToNormalizedDisplay(val[0],val[1]);
    case VTK_NORMALIZED_DISPLAY:
      viewport->NormalizedDisplayToDisplay(val[0],val[1]);
    }
  
  this->ComputedDisplayValue[0] = (int)VTK_RINT(val[0]);
  this->ComputedDisplayValue[1] = (int)VTK_RINT(val[1]);
  
  // if we have a reference coordinate and we haven't handled it yet
  if (this->ReferenceCoordinate && 
      (this->CoordinateSystem == VTK_DISPLAY ||
       this->CoordinateSystem == VTK_NORMALIZED_DISPLAY))
    {
    int *RefValue;
    
    RefValue = this->ReferenceCoordinate->GetComputedDisplayValue(viewport);
    this->ComputedDisplayValue[0] += RefValue[0];
    this->ComputedDisplayValue[1] += RefValue[1];
    }
  
  this->Computing = 0;
  vtkDebugMacro("Returning DisplayValue of : " << 
		this->ComputedDisplayValue[0] << " , " << 
		this->ComputedDisplayValue[1]);
  return this->ComputedDisplayValue;
}





float *vtkCoordinate::GetComputedValue(vtkViewport* viewport)
{
  // use our viewport if set
  if (this->Viewport)
    {
    viewport = this->Viewport;
    }
  
  switch (this->CoordinateSystem)
    {
    case VTK_WORLD:
      return this->GetComputedWorldValue(viewport);
    case VTK_VIEW:
    case VTK_NORMALIZED_VIEWPORT:
    case VTK_VIEWPORT:
      {
      // result stored in computed world value due to float
      // but is really a viewport value
      int *v = this->GetComputedViewportValue(viewport);
      this->ComputedWorldValue[0] = v[0];
      this->ComputedWorldValue[1] = v[1];
      break;
      }
    case VTK_NORMALIZED_DISPLAY:
    case VTK_DISPLAY:
      {
      // result stored in computed world value due to float
      // but is really a display value
      int *d = this->GetComputedDisplayValue(viewport);
      this->ComputedWorldValue[0] = d[0];
      this->ComputedWorldValue[1] = d[1];
      break;
      }
    }

  return this->ComputedWorldValue;
}

