/*****************************************************************************/
/*                                    XDMF                                   */
/*                       eXtensible Data Model and Format                    */
/*                                                                           */
/*  Id : XdmfGeometryType.hpp                                                */
/*                                                                           */
/*  Author:                                                                  */
/*     Kenneth Leiter                                                        */
/*     kenneth.leiter@arl.army.mil                                           */
/*     US Army Research Laboratory                                           */
/*     Aberdeen Proving Ground, MD                                           */
/*                                                                           */
/*     Copyright @ 2011 US Army Research Laboratory                          */
/*     All Rights Reserved                                                   */
/*     See Copyright.txt for details                                         */
/*                                                                           */
/*     This software is distributed WITHOUT ANY WARRANTY; without            */
/*     even the implied warranty of MERCHANTABILITY or FITNESS               */
/*     FOR A PARTICULAR PURPOSE.  See the above copyright notice             */
/*     for more information.                                                 */
/*                                                                           */
/*****************************************************************************/

#ifndef XDMFGEOMETRYTYPE_HPP_
#define XDMFGEOMETRYTYPE_HPP_

// Includes
#include "Xdmf.hpp"
#include "XdmfItemProperty.hpp"

/**
 * @brief Property describing the types of coordinate values stored in
 * an XdmfGeometry.
 *
 * XdmfGeometryType is a property used by XdmfGeometry to specify the
 * type of coordinate values stored in the XdmfGeometry. A specific
 * XdmfGeometryType can be created by calling one of the static
 * methods in the class, i.e.  XdmfAttributeType::XYZ().
 *
 * Example of use:
 *
 * //Assuming that exampleGeometry is a shared pointer to an XdmfGeometry with its type set
 * if (exampleGeometry->getType() == XdmfGeometry::XYZ())
 * {
 *   //do whatever is to be done if the geometry is xyz
 * }
 *
 * Xdmf supports the following geometry types:
 *   NoGeometryType
 *   XYZ
 *   XY
 */
class XDMF_EXPORT XdmfGeometryType : public XdmfItemProperty {

public:

  virtual ~XdmfGeometryType();

  friend class XdmfGeometry;

  // Supported Xdmf Geometry Types
  static shared_ptr<const XdmfGeometryType> NoGeometryType();
  static shared_ptr<const XdmfGeometryType> XYZ();
  static shared_ptr<const XdmfGeometryType> XY();

  /**
   * Get the dimensions of this geometry type - i.e. XYZ = 3.
   *
   * Example of use:
   * 
   * C++
   *
   * @code {.cpp}
   * unsigned int exampleDimensions = XdmfGeometryType::XYZ()->getDimensions();
   * //The variable exampleDimensions now holds the number of dimensions that XYZ has
   * @endcode
   *
   * Python
   *
   * @code {.py}
   * exampleDimensions = XdmfGeometryType.XYZ().getDimensions()
   * '''
   * The variable exampleDimensions now holds the number of dimensions that XYZ has
   * '''
   * @endcode
   *
   * @return an int containing number of dimensions.
   */
  virtual unsigned int getDimensions() const;

  /**
   * Get the name of this geometry type.
   *
   * Example of use:
   *
   * C++
   *
   * @code {.cpp}
   * std::string exampleName = XdmfGeometryType::XYZ()->getName();
   * //The variable exampleName now holds the name of XYZ
   * @endcode
   *
   * Python
   *
   * @code {.py}
   * exampleName = XdmfGeometryType.XYZ().getName()
   * '''
   * The variable exampleName now holds the name of XYZ
   * '''
   * @endcode
   *
   * @return the name of this geometry type.
   */
  std::string getName() const;

  virtual void
  getProperties(std::map<std::string, std::string> & collectedProperties) const;

protected:

  /**
   * Protected constructor for XdmfGeometryType.  The constructor is
   * protected because all geometry types supported by Xdmf should be
   * accessed through more specific static methods that construct
   * XdmfGeometryTypes - i.e.  XdmfGeometryType::XYZ().
   *
   * @param name a std::string containing the name of the geometry type..
   * @param dimensions an int containing the dimensions of the geometry type.
   */
  XdmfGeometryType(const std::string & name, const int & dimensions);

private:

  XdmfGeometryType(const XdmfGeometryType &); // Not implemented.
  void operator=(const XdmfGeometryType &); // Not implemented.

  static shared_ptr<const XdmfGeometryType>
  New(const std::map<std::string, std::string> & itemProperties);

  unsigned int mDimensions;
  std::string mName;
};

#endif /* XDMFGEOMETRYTYPE_HPP_ */
