/*=========================================================================

  Program:   ParaView
  Module:    vtkPVRenderView.cxx

  Copyright (c) Kitware, Inc.
  All rights reserved.
  See Copyright.txt or http://www.paraview.org/HTML/Copyright.html for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkPVRenderView.h"
#include "vtkPVRenderModule.h"

#include "vtkInstantiator.h"
#include "vtkBMPWriter.h"
#include "vtkCamera.h"
#include "vtkCollectionIterator.h"
#include "vtkCallbackCommand.h"
#include "vtkCommand.h"
#include "vtkErrorCode.h"
#include "vtkImageData.h"
#include "vtkInteractorObserver.h"
#include "vtkJPEGWriter.h"
#include "vtkKWApplicationSettingsInterface.h"
#include "vtkKWChangeColorButton.h"
#include "vtkKWCheckButton.h"
#include "vtkKWComposite.h"
#include "vtkKWCompositeCollection.h"
#include "vtkKWCornerAnnotation.h"
#include "vtkKWFrame.h"
#include "vtkKWLabel.h"
#include "vtkKWLabeledFrame.h"
#include "vtkKWMenu.h"
#include "vtkKWMessageDialog.h"
#include "vtkKWNotebook.h"
#include "vtkPVSourceNotebook.h"
#include "vtkKWPushButton.h"
#include "vtkKWRadioButton.h"
#include "vtkKWScale.h"
#include "vtkKWSplitFrame.h"
#include "vtkKWWindowCollection.h"
#include "vtkMultiProcessController.h"
#include "vtkObjectFactory.h"
#include "vtkPNGWriter.h"
#include "vtkPNMWriter.h"
#include "vtkPVApplication.h"
#include "vtkPVApplicationSettingsInterface.h"
#include "vtkPVAxesWidget.h"
#include "vtkPVCameraControl.h"
#include "vtkPVCameraIcon.h"
#include "vtkPVCompositeRenderModule.h"
#include "vtkPVConfig.h"
#include "vtkPVDataInformation.h"
#include "vtkPVGenericRenderWindowInteractor.h"
#include "vtkPVInteractorStyleControl.h"
#include "vtkPVNavigationWindow.h"
#include "vtkSMPart.h"
#include "vtkSMPartDisplay.h"
#include "vtkPVProcessModule.h"
#include "vtkPVRenderModuleUI.h"
#include "vtkPVRenderView.h"
#include "vtkPVSource.h"
#include "vtkPVSourceCollection.h"
#include "vtkPVSourceList.h"
#include "vtkPVWindow.h"
#include "vtkPolyData.h"
#include "vtkPolyDataMapper.h"
#include "vtkPostScriptWriter.h"
#include "vtkRenderer.h"
#include "vtkRendererCollection.h"
#include "vtkRenderWindow.h"
#include "vtkString.h"
#include "vtkTIFFWriter.h"
#include "vtkTimerLog.h"
#include "vtkToolkits.h"
#include "vtkWindowToImageFilter.h"
#include "vtkClientServerStream.h"
#include "vtkPVOptions.h"


#ifdef _WIN32
#include "vtkWin32OpenGLRenderWindow.h"
#else
#include "vtkOpenGLRenderWindow.h"
#include "vtkOpenGLRenderer.h"
#endif

#define VTK_PV_NAV_FRAME_SIZE_REG_KEY "NavigationFrameSize"

/* 
 * This part was generated by ImageConvert from image:
 *    properties.png (zlib, base64)
 */
#define image_properties_width         16
#define image_properties_height        14
#define image_properties_pixel_size    4
#define image_properties_buffer_length 140

static unsigned char image_properties[] = 
  "eNrdkcENgDAMAxk9gzBMRmCjACqR3NSO4Islv5pz3dTdNxe+FMzjzOJ2x8axU48czSfbZ9"
  "jjtR+yKgP5lGJrBvZPfpoVWWx/OI9dVOa8J1uYqvo+uuuX6njsT//kx/d/ceVPtN4Pmg==";



//===========================================================================
//***************************************************************************
class vtkPVRenderViewObserver : public vtkCommand
{
public:
  static vtkPVRenderViewObserver *New() 
    {return new vtkPVRenderViewObserver;};

  vtkPVRenderViewObserver()
    {
      this->PVRenderView = 0;
    }

  virtual void Execute(vtkObject* wdg, unsigned long event,  
                       void* calldata)
    {
      if ( this->PVRenderView )
        {
        this->PVRenderView->ExecuteEvent(wdg, event, calldata);
        this->AbortFlagOn();
        }
    }

  vtkPVRenderView* PVRenderView;
};
//***************************************************************************
//===========================================================================

//----------------------------------------------------------------------------
vtkStandardNewMacro(vtkPVRenderView);
vtkCxxRevisionMacro(vtkPVRenderView, "1.343");

int vtkPVRenderViewCommand(ClientData cd, Tcl_Interp *interp,
                             int argc, char *argv[]);



//----------------------------------------------------------------------------
vtkPVRenderView::vtkPVRenderView()
{
  this->RenderModule = NULL;

  if (getenv("PV_SEPARATE_RENDER_WINDOW") != NULL)
    {
    this->TopLevelRenderWindow = vtkKWWidget::New();
    this->TopLevelRenderWindow->SetParent(this->Frame);
    this->VTKWidget->SetParent(NULL);
    this->VTKWidget->SetParent(this->TopLevelRenderWindow);
    }
  else
    {
    this->TopLevelRenderWindow = NULL;
    }    

  this->CommandFunction = vtkPVRenderViewCommand;
  this->SplitFrame = vtkKWSplitFrame::New();

  this->BlockRender = 0;

  this->MenuEntryUnderline = 4;
  this->SetMenuEntryName(VTK_PV_VIEW_MENU_LABEL);
  this->SetMenuEntryHelp("Show global view parameters (background color, annoations2 etc.)");

  this->StandardViewsFrame = vtkKWLabeledFrame::New();
  this->XMaxViewButton = vtkKWPushButton::New();
  this->XMinViewButton = vtkKWPushButton::New();
  this->YMaxViewButton = vtkKWPushButton::New();
  this->YMinViewButton = vtkKWPushButton::New();
  this->ZMaxViewButton = vtkKWPushButton::New();
  this->ZMinViewButton = vtkKWPushButton::New();
  
  this->RenderParametersFrame = vtkKWLabeledFrame::New();

  this->TriangleStripsCheck = vtkKWCheckButton::New();
  this->ParallelProjectionCheck = vtkKWCheckButton::New();
  this->ImmediateModeCheck = vtkKWCheckButton::New();

  this->RenderModuleUI = NULL;
 
  this->ManipulatorControl2D = vtkPVInteractorStyleControl::New();
  this->ManipulatorControl2D->SetRegisteryName("2D");
  this->ManipulatorControl3D = vtkPVInteractorStyleControl::New();
  this->ManipulatorControl3D->SetRegisteryName("3D");

  this->CameraControlFrame = vtkKWLabeledFrame::New();
  this->CameraControl = vtkPVCameraControl::New();
  
  this->NavigationFrame = vtkKWLabeledFrame::New();
  this->NavigationWindow = vtkPVNavigationWindow::New();
  this->SelectionWindow = vtkPVSourceList::New();

  this->NavigationWindowButton = vtkKWRadioButton::New();
  this->SelectionWindowButton = vtkKWRadioButton::New();

  this->ShowSelectionWindow = 0;
  this->ShowNavigationWindow = 0;

  this->InterfaceSettingsFrame = vtkKWLabeledFrame::New();
  this->Display3DWidgets = vtkKWCheckButton::New();

  this->OrientationAxesFrame = vtkKWLabeledFrame::New();
  this->OrientationAxesCheck = vtkKWCheckButton::New();
  this->OrientationAxesInteractiveCheck = vtkKWCheckButton::New();
  this->OrientationAxesOutlineColor = vtkKWChangeColorButton::New();
  this->OrientationAxesTextColor = vtkKWChangeColorButton::New();
  this->OrientationAxes = vtkPVAxesWidget::New();

  int cc;
  for ( cc = 0; cc < 6; cc ++ )
    {
    this->CameraIcons[cc] = vtkPVCameraIcon::New();
    }
  this->CameraIconsFrame = vtkKWLabeledFrame::New();

  this->PropertiesButton = vtkKWPushButton::New();
  this->MenuLabelSwitchBackAndForthToViewProperties = 0;

  this->Observer = vtkPVRenderViewObserver::New();
  this->Observer->PVRenderView = this;
  
  this->RenderTimer = vtkTimerLog::New();
  this->TimerToken = NULL;

  this->SourceNotebook = 0;
}



//----------------------------------------------------------------------------
// Only used for corner annotation.
void vtkPVRenderView::Add2DComposite(vtkKWComposite *c)  
{  
  //int fixme;  // this should be in render module.  
   
  c->SetView(this);  
  // never allow a composite to be added twice  
  if (this->Composites->IsItemPresent(c))  
    {  
    return;  
    }  
  this->Composites->AddItem(c);  
  if (c->GetProp() != NULL)  
    {  
    this->GetRenderer2D()->AddProp(c->GetProp());  
    }  
}  
   
//----------------------------------------------------------------------------
// Only used for corner annotation.
void vtkPVRenderView::Remove2DComposite(vtkKWComposite *c)  
{  
  
  c->SetView(NULL);  
  this->GetRenderer2D()->RemoveProp(c->GetProp());  
  this->Composites->RemoveItem(c);  
}  






//----------------------------------------------------------------------------
vtkRenderWindow* vtkPVRenderView::GetRenderWindow()
{
  if (this->RenderModule == NULL)
    {
    vtkErrorMacro("Missing renderModule.");
    return NULL;
    }
  return this->RenderModule->GetRenderWindow();
}

//----------------------------------------------------------------------------
vtkRenderer* vtkPVRenderView::GetRenderer()
{
  if (this->RenderModule == NULL)
    {
    vtkErrorMacro("Missing renderModule.");
    return NULL;
    }
  return this->RenderModule->GetRenderer();
}

//----------------------------------------------------------------------------
vtkRenderer* vtkPVRenderView::GetRenderer2D()
{
  if (this->RenderModule == NULL)
    {
    vtkErrorMacro("Missing renderModule.");
    return NULL;
    }
  return this->RenderModule->GetRenderer2D();
}

//----------------------------------------------------------------------------
void vtkPVRenderView::ShowNavigationWindowCallback(int registery)
{
  if (!this->GetApplication())
    {
    return;
    }
  
  this->AddTraceEntry("$kw(%s) ShowNavigationWindowCallback %d",
                      this->GetTclName(), registery);
  
  this->Script("catch {eval pack forget [pack slaves %s]}",
               this->NavigationFrame->GetFrame()->GetWidgetName());
  this->Script("pack %s -fill both -expand t -side top -anchor n", 
               this->NavigationWindow->GetWidgetName());
  this->NavigationFrame->SetLabel("Navigation Window");
  this->ShowSelectionWindow = 0;
  this->ShowNavigationWindow = 1;

  this->NavigationWindowButton->StateOn();
  this->SelectionWindowButton->StateOff();
  if ( registery )
    {
    this->GetApplication()->SetRegisteryValue(2, "RunTime","SourcesBrowser",
                                         "NavigationWindow");
    }
}

//----------------------------------------------------------------------------
void vtkPVRenderView::ShowSelectionWindowCallback(int registery)
{
  if ( !this->GetApplication() )
    {
    return;
    }
  
  this->AddTraceEntry("$kw(%s) ShowSelectionWindowCallback %d",
                      this->GetTclName(), registery);
  
  this->Script("catch {eval pack forget [pack slaves %s]}",
               this->NavigationFrame->GetFrame()->GetWidgetName());
  this->Script("pack %s -fill both -expand t -side top -anchor n", 
               this->SelectionWindow->GetWidgetName());
  this->NavigationFrame->SetLabel("Selection Window");
  this->ShowNavigationWindow = 0;
  this->ShowSelectionWindow = 1;

  this->NavigationWindowButton->StateOff();
  this->SelectionWindowButton->StateOn();
  if ( registery )
    {
    this->GetApplication()->SetRegisteryValue(2, "RunTime","SourcesBrowser",
                                         "SelectionWindow");
    }
}

//----------------------------------------------------------------------------
vtkPVRenderView::~vtkPVRenderView()
{
  if (this->RenderModule)
    {
    this->RenderModule->UnRegister(this);
    this->RenderModule = NULL;
    }

  this->InterfaceSettingsFrame->Delete();
  this->Display3DWidgets->Delete();
  this->Display3DWidgets = NULL;

  this->OrientationAxesFrame->Delete();
  this->OrientationAxesCheck->Delete();
  this->OrientationAxesInteractiveCheck->Delete();
  this->OrientationAxesOutlineColor->Delete();
  this->OrientationAxesTextColor->Delete();
  this->OrientationAxes->Delete();
  
  if ( this->SelectionWindow )
    {
    this->SelectionWindow->Delete();
    }
  this->SplitFrame->Delete();
  this->SplitFrame = NULL;
 
  this->NavigationFrame->Delete();
  this->NavigationFrame = NULL;
 
  this->NavigationWindowButton->Delete();
  this->SelectionWindowButton->Delete();

  this->NavigationWindow->Delete();
  this->NavigationWindow = NULL;

  if (this->RenderModuleUI)
    {
    this->RenderModuleUI->Delete();
    this->RenderModuleUI = NULL;
    }
  
  // undo the binding we set up
  if ( this->IsCreated() )
    {
    this->Script("bind %s <Motion> {}", this->VTKWidget->GetWidgetName());
    }

  if (this->TopLevelRenderWindow)
    {
    this->TopLevelRenderWindow->Delete();
    this->TopLevelRenderWindow = NULL;
    }
  
  this->StandardViewsFrame->Delete();
  this->StandardViewsFrame = NULL;
  this->XMaxViewButton->Delete();
  this->XMaxViewButton = NULL;
  this->XMinViewButton->Delete();
  this->XMinViewButton = NULL;
  this->YMaxViewButton->Delete();
  this->YMaxViewButton = NULL;
  this->YMinViewButton->Delete();
  this->YMinViewButton = NULL;
  this->ZMaxViewButton->Delete();
  this->ZMaxViewButton = NULL;
  this->ZMinViewButton->Delete();
  this->ZMinViewButton = NULL;

  this->RenderParametersFrame->Delete();
  this->RenderParametersFrame = 0;

  this->ParallelProjectionCheck->Delete();
  this->ParallelProjectionCheck = NULL;

  this->TriangleStripsCheck->Delete();
  this->TriangleStripsCheck = NULL;

  this->ImmediateModeCheck->Delete();
  this->ImmediateModeCheck = NULL;


  this->ManipulatorControl2D->Delete();
  this->ManipulatorControl3D->Delete();

  this->CameraControlFrame->Delete();
  this->CameraControl->Delete();
  
  this->CameraIconsFrame->Delete();
  this->CameraIconsFrame = 0;
  int cc;
  for ( cc = 0; cc < 6; cc ++ )
    {
    if ( this->CameraIcons[cc] )
      {
      this->CameraIcons[cc]->SetRenderView(0);
      this->CameraIcons[cc]->Delete();
      this->CameraIcons[cc] = 0;
      }
    }

  this->Observer->Delete();
  this->Observer = NULL;

  this->PropertiesButton->Delete();
  this->PropertiesButton = NULL;
  this->SetMenuLabelSwitchBackAndForthToViewProperties(NULL);
  
  this->RenderTimer->Delete();
  if (this->TimerToken)
    {
    Tcl_DeleteTimerHandler( this->TimerToken );
    this->TimerToken = NULL;
    }
    
  if (this->SourceNotebook)
    {
    this->SourceNotebook->SetParent(0);
    this->SourceNotebook->Delete();
    this->SourceNotebook = 0;
    }
}

//----------------------------------------------------------------------------
void PVRenderViewAbortCheck(vtkObject*, unsigned long, void* arg, void*)
{
  vtkPVRenderView *me = (vtkPVRenderView*)arg;
  int abort;

  // if we are printing then do not abort
  if (me->GetPrinting())
    {
    return;
    }
  
  abort = me->ShouldIAbort();
  if (abort == 1)
    {
    me->GetRenderWindow()->SetAbortRender(1);
    me->EventuallyRender();
    }
  if (abort == 2)
    {
    //("Abort 2");
    me->GetRenderWindow()->SetAbortRender(2);
    }
}

//----------------------------------------------------------------------------
void vtkPVRenderView::CreateRenderObjects(vtkPVApplication *pvApp)
{
  // Create the call back that looks for events to abort rendering.
  pvApp->GetProcessModule()->GetRenderModule()->RemoveObservers(vtkCommand::AbortCheckEvent);
  vtkCallbackCommand* abc = vtkCallbackCommand::New();
  abc->SetCallback(PVRenderViewAbortCheck);
  abc->SetClientData(this);
  pvApp->GetProcessModule()->GetRenderModule()->AddObserver(vtkCommand::AbortCheckEvent, abc);
  abc->Delete();
}


//----------------------------------------------------------------------------
// Here we are going to change only the satellite procs.
void vtkPVRenderView::PrepareForDelete()
{
  // Get rid of the bindings.
  // We do not want methods called by tcl after this object has been deleted.
  // I was getting intermittent crashes from the expose binding.
  // Expose
  this->Script("bind %s <Expose> {}", 
               this->GetTclName(), this->GetTclName());
  // Configure
  this->Script("bind %s <Configure> {}", 
               this->GetTclName(), this->GetTclName());
  // I do not know if this is neccesary, but I intend to remove all 
  // pending events for this object before it gets deleted.
  this->Script("update");


  vtkPVApplication* pvapp = this->GetPVApplication();
  if (pvapp)
    {
    pvapp->SetRegisteryValue(2, "RunTime", "UseParallelProjection", "%d",
                             this->ParallelProjectionCheck->GetState());
    pvapp->SetRegisteryValue(2, "RunTime", "UseStrips", "%d",
                             this->TriangleStripsCheck->GetState());
    pvapp->SetRegisteryValue(2, "RunTime", "UseImmediateMode", "%d",
                             this->ImmediateModeCheck->GetState());
    double *vp = this->OrientationAxes->GetViewport();
    pvapp->SetRegisteryValue(2, "RunTime", "OrientationAxesViewport",
                             "%lf %lf %lf %lf", vp[0], vp[1], vp[2], vp[3]);

    // If it's the last win, save the size of the nav frame

    if (pvapp->GetWindows()->GetNumberOfItems() <= 1 &&
        pvapp->GetSaveWindowGeometry())
      {
      pvapp->SetRegisteryValue(
        2, "Geometry", VTK_PV_NAV_FRAME_SIZE_REG_KEY, "%d", 
        this->SplitFrame->GetFrame1Size());
      }
    }

  // We must call prepare for delete on the RenderModule !!!!!!!


  // Circular reference.
  if ( this->ManipulatorControl2D )
    {
    this->ManipulatorControl2D->SetManipulatorCollection(0);
    }
  if ( this->ManipulatorControl3D )
    {
    this->ManipulatorControl3D->SetManipulatorCollection(0);
    }
  if ( this->SelectionWindow )
    {
    this->SelectionWindow->PrepareForDelete();
    this->SelectionWindow->Delete();
    this->SelectionWindow = 0;
    }
  int cc;
  for ( cc = 0; cc < 6; cc ++ )
    {
    if ( this->CameraIcons[cc] )
      {
      this->CameraIcons[cc]->SetRenderView(0);
      this->CameraIcons[cc]->Delete();
      this->CameraIcons[cc] = 0;
      }
    }
}


//----------------------------------------------------------------------------
void vtkPVRenderView::Close()
{
  this->PrepareForDelete();
  vtkKWView::Close();
}  


//----------------------------------------------------------------------------
void vtkPVRenderView::Create(vtkKWApplication *app, const char *args)
{
  // Call the superclass to create the widget and set the appropriate flags

  if (!this->vtkKWWidget::Create(app, "frame", "-bd 0"))
    {
    vtkErrorMacro("Failed creating widget " << this->GetClassName());
    return;
    }

  this->ConfigureOptions(args);

  // Need to make sure it destructs right before this view does.
  // It's the whole TKRenderWidget destruction pain.

  if (this->RenderModule == NULL)
    {
    this->RenderModule = this->GetPVApplication()->GetProcessModule()->GetRenderModule();
    this->RenderModule->Register(this);
    }

  // Create the frames

  this->Frame->Create(app,"frame","-bd 3 -relief ridge");
  this->Script("pack %s -expand yes -fill both -side top -anchor nw",
               this->Frame->GetWidgetName());

  this->Frame2->Create(app,"frame","-bd 0");
  this->Script("pack %s -fill x -side top -anchor nw",
               this->Frame2->GetWidgetName());

  this->Label->Create(app, "-fg #fff -text {3D View} -bd 0");

  this->Script("pack %s -side left -anchor w",
               this->Label->GetWidgetName());

  this->Script("bind %s <Any-ButtonPress> {%s MakeSelected}",
               this->Label->GetWidgetName(), this->GetTclName());

  // Properties button

  this->PropertiesButton->SetParent(this->Label->GetParent());
  // The -bg color is hardcoded here since the real -bg of the frame is
  // changed when it's active (it's not active at the moment). But inside
  // paraview tehre is only one view, and it's always active, so use the
  // active color.
  this->PropertiesButton->Create(
    app, "-bd 0 -bg #008 -padx 0 -pady 0 -highlightthickness 0 -relief flat");
  this->PropertiesButton->SetCommand(this, "SwitchBackAndForthToViewProperties");
  this->PropertiesButton->SetBalloonHelpString(
    "Switch back and forth between the current view properties and the" 
    VTK_PV_VIEW_MENU_LABEL ".");

  this->PropertiesButton->SetImageOption(image_properties, 
                                         image_properties_width, 
                                         image_properties_height, 
                                         image_properties_pixel_size,
                                         image_properties_buffer_length);

  this->Script("pack %s %s -side left -anchor w -padx 2",
               this->Label->GetWidgetName(),
               this->PropertiesButton->GetWidgetName());

  this->Script("bind %s <Any-ButtonPress> {%s MakeSelected}",
               this->Frame2->GetWidgetName(), this->GetTclName());

  // Create the control frame - only pack it if support option enabled

  this->ControlFrame->Create(app,"frame","-bd 0");
  if (this->SupportControlFrame)
    {
    this->Script("pack %s -expand t -fill both -side top -anchor nw",
                 this->ControlFrame->GetWidgetName());
    }
  
  // Separate window for the renderer.

  if (getenv("PV_SEPARATE_RENDER_WINDOW") != NULL)
    {
    this->TopLevelRenderWindow->Create(app, "toplevel", "");
    this->Script("wm title %s %s", 
                 this->TopLevelRenderWindow->GetWidgetName(),
                 app->GetApplicationName());
    }

  // Add the -rw argument

  char *local = new char [strlen(args)+100];
  sprintf(local,"%s -rw Addr=%p",args,this->GetRenderWindow());
  this->Script("vtkTkRenderWidget %s %s",
               this->VTKWidget->GetWidgetName(),local);

  this->Script("pack %s -expand yes -fill both -side top -anchor nw",
               this->VTKWidget->GetWidgetName());
  
  // Expose
  this->Script("bind %s <Expose> {%s Exposed}", 
               this->GetTclName(), this->GetTclName());

  // Configure
  this->Script("bind %s <Configure> {%s Configured}", 
               this->GetTclName(), this->GetTclName());

  // Configure the split frame

  this->SplitFrame->SetParent(this->GetPropertiesParent());
  this->SplitFrame->SetOrientationToVertical();
  this->SplitFrame->SetSeparatorSize(5);
  this->SplitFrame->SetFrame1MinimumSize(80);

  if (app->GetSaveWindowGeometry() &&
      app->HasRegisteryValue(
        2, "Geometry", VTK_PV_NAV_FRAME_SIZE_REG_KEY))
    {
    this->SplitFrame->SetFrame1Size(
      app->GetIntRegisteryValue(
        2, "Geometry", VTK_PV_NAV_FRAME_SIZE_REG_KEY));
    }
  else
    {
    this->SplitFrame->SetFrame1Size(80);
    }

  this->SplitFrame->Create(app);

  this->Script("pack %s -fill both -expand t -side top", 
               this->SplitFrame->GetWidgetName());

  // Configure the navigation frame

  this->NavigationFrame->SetParent(this->SplitFrame->GetFrame1());
  this->NavigationFrame->ShowHideFrameOff();
  this->NavigationFrame->Create(app, 0);
  this->NavigationFrame->SetLabel("Navigation");
  this->Script("pack %s -fill both -expand t -side top", 
               this->NavigationFrame->GetWidgetName());

  // Configure the navigation window

  this->NavigationWindow->SetParent(this->NavigationFrame->GetFrame());
  this->NavigationWindow->SetWidth(341);
  this->NavigationWindow->SetHeight(545);
  this->NavigationWindow->Create(app, 0); 

  // Configure the selection window

  this->SelectionWindow->SetParent(this->NavigationFrame->GetFrame());
  this->SelectionWindow->SetWidth(341);
  this->SelectionWindow->SetHeight(545);
  this->SelectionWindow->Create(app, 0); 

  this->SelectionWindowButton->SetParent(
    this->NavigationFrame->GetLabelFrame());
  this->SelectionWindowButton->Create(
    app, 
    "-indicatoron 0 -highlightthickness 0 -image PVSelectionWindowButton "
    "-selectimage PVSelectionWindowButton");
  this->SelectionWindowButton->SetBalloonHelpString(
    "Switch to selection window mode.");
  this->SelectionWindowButton->SetCommand(
    this, "ShowSelectionWindowCallback 1");
  
  this->NavigationWindowButton->SetParent(
    this->NavigationFrame->GetLabelFrame());
  this->NavigationWindowButton->Create(
    app, 
    "-indicatoron 0 -highlightthickness 0 -image PVNavigationWindowButton "
    "-selectimage PVNavigationWindowButton");
  this->NavigationWindowButton->SetBalloonHelpString(
    "Switch to navigation window mode.");
  this->NavigationWindowButton->SetCommand(
    this, "ShowNavigationWindowCallback 1");

  this->Script("pack %s %s -side left -anchor w -before %s -padx 1",
               this->SelectionWindowButton->GetWidgetName(),
               this->NavigationWindowButton->GetWidgetName(),
               this->NavigationFrame->GetLabel()->GetWidgetName());

  if (app->HasRegisteryValue(2, "RunTime", "SourcesBrowser"))
    {
    if (app->BooleanRegisteryCheck(2, "RunTime", 
                                                 "SourcesBrowser",
                                                 "SelectionWindow"))
      {
      this->ShowSelectionWindowCallback(0);
      }
    else
      {
      this->ShowNavigationWindowCallback(0);
      }
    }
  else
    {
    this->ShowSelectionWindowCallback(0);
    }

  this->SetSourcesBrowserAlwaysShowName(
    this->GetPVApplication()->GetSourcesBrowserAlwaysShowName());

  this->GetRenderWindow()->AddObserver(
                 vtkCommand::CursorChangedEvent, this->Observer);


  // This is created in this object only because the GetSourceParent method.
  // These should be create in application or window.
  this->SourceNotebook = vtkPVSourceNotebook::New();
  this->SourceNotebook->SetParent(this->GetSourceParent());
  this->SourceNotebook->Create(app,"");
  this->Script("pack %s -pady 2 -padx 2 -fill both -expand yes -anchor n",
               this->SourceNotebook->GetWidgetName());

  // For initializing the trace of the notebook.
  // Do we really need this?
  this->GetSourceParent()->SetTraceReferenceObject(this);
  this->GetSourceParent()->SetTraceReferenceCommand("GetParametersParent");

  this->EventuallyRender();
  delete [] local;
}

//----------------------------------------------------------------------------
void vtkPVRenderView::SwitchBackAndForthToViewProperties()
{
  vtkKWWindow *win = this->GetPVWindow();
  if (!win)
    {
    return;
    }

  vtkKWMenu *viewmenu = win->GetMenuView();
  if (!viewmenu)
    {
    return;
    }

  if (!viewmenu->HasItem(VTK_PV_VIEW_MENU_LABEL))
    {
    return;
    }
  int prop_position = viewmenu->GetIndex(VTK_PV_VIEW_MENU_LABEL);

  // First check where we are in the view menu

  int position = viewmenu->GetCheckedRadioButtonItem(viewmenu, "Radio");
  if (position < 0)
    {
    return;
    }

  // Now if we are not in the view prop, save the old pos and go to it
  // otherwise restore the old one

  if (position != prop_position)
    {
    this->SetMenuLabelSwitchBackAndForthToViewProperties(this->Script(
      "%s entrycget %d -label", viewmenu->GetWidgetName(), position));
    viewmenu->Invoke(prop_position);
    }
  else
    {
    if (this->MenuLabelSwitchBackAndForthToViewProperties &&
        viewmenu->HasItem(this->MenuLabelSwitchBackAndForthToViewProperties))
      {
      // This entry of the view menu must be in "normal" state to switch
      // between source and 3D views.
      int state = viewmenu->GetState(this->MenuLabelSwitchBackAndForthToViewProperties);
      viewmenu->SetState(this->MenuLabelSwitchBackAndForthToViewProperties,
                         vtkKWMenu::Normal);
      viewmenu->Invoke(
        viewmenu->GetIndex(this->MenuLabelSwitchBackAndForthToViewProperties));
      viewmenu->SetState(this->MenuLabelSwitchBackAndForthToViewProperties,
                         state);
      }
    }
}

//----------------------------------------------------------------------------
vtkKWWidget *vtkPVRenderView::GetSourceParent()
{
  return this->SplitFrame->GetFrame2();
}

//----------------------------------------------------------------------------
void vtkPVRenderView::Display3DWidgetsCallback()
{
  int val = this->Display3DWidgets->GetState();
  this->SetDisplay3DWidgets(val);
  this->GetApplication()->SetRegisteryValue(2, "RunTime","Display3DWidgets",
                                      (val?"1":"0"));
}

//----------------------------------------------------------------------------
void vtkPVRenderView::SetDisplay3DWidgets(int s)
{
  this->Display3DWidgets->SetState(s);
  this->GetPVApplication()->SetDisplay3DWidgets(s);
}

//----------------------------------------------------------------------------
void vtkPVRenderView::SetSourcesBrowserAlwaysShowName(int s)
{
  if (this->NavigationWindow)
    {
    this->NavigationWindow->SetAlwaysShowName(s);
    }
  if (this->SelectionWindow)
    {
    this->SelectionWindow->SetAlwaysShowName(s);
    }
}

//----------------------------------------------------------------------------
void vtkPVRenderView::CreateViewProperties()
{
  this->vtkKWView::CreateViewProperties();

  double rgb[3];
  vtkPVWindow* pvwindow = this->GetPVWindow();
  vtkPVApplication* pvapp = this->GetPVApplication();

  this->RendererBackgroundColor->SetBalloonHelpString(
    "Change the background color of the 3D View window");
  // This used to be in vtkPVWindow.
  // I moved it here because I had to move Save color to this class.
  // I also think it belongs here.
  // Don't you think that save and retrieve color should really be in
  // the class vtkPVApplication?
  pvwindow->RetrieveColor(2, "RenderViewBG", rgb); 
  if (rgb[0] == -1)
    {
    rgb[0] = 0.33;
    rgb[1] = 0.35;
    rgb[2] = 0.43;
    }
  this->SetRendererBackgroundColor(rgb[0], rgb[1], rgb[2]);

  // Render parameters

  this->RenderParametersFrame->SetParent(this->GeneralProperties->GetFrame());
  this->RenderParametersFrame->ShowHideFrameOn();
  this->RenderParametersFrame->Create(this->GetApplication(),0);
  this->RenderParametersFrame->SetLabel("Advanced Render Parameters");
  this->Script("pack %s -padx 2 -pady 2 -fill x -expand yes -anchor w",
               this->RenderParametersFrame->GetWidgetName());

  // Render parameters: parallel projection

  this->ParallelProjectionCheck->SetParent(
    this->RenderParametersFrame->GetFrame());
  this->ParallelProjectionCheck->Create(this->GetApplication(), "");
  this->ParallelProjectionCheck->SetText("Use parallel projection");
  if (pvapp && pvwindow && 
      pvapp->GetRegisteryValue(2, "RunTime", "UseParallelProjection", 0))
    {
    this->ParallelProjectionCheck->SetState(
      pvwindow->GetIntRegisteryValue(2, "RunTime", "UseParallelProjection"));
    this->ParallelProjectionCallback();
    }
  else
    {
    this->ParallelProjectionCheck->SetState(0);
    }
  this->ParallelProjectionCheck->SetCommand(this, 
                                            "ParallelProjectionCallback");
  this->ParallelProjectionCheck->SetBalloonHelpString(
    "Toggle the use of parallel projection "
    "(if parallel project is off, perspective projection is used).");
  
  // Render parameters: triangle strips

  this->TriangleStripsCheck->SetParent(
    this->RenderParametersFrame->GetFrame());
  this->TriangleStripsCheck->Create(this->GetApplication(), "");
  this->TriangleStripsCheck->SetText("Use triangle strips");
  if (pvapp && pvwindow && 
      pvapp->GetRegisteryValue(2, "RunTime", "UseStrips", 0))
    {
    this->TriangleStripsCheck->SetState(
      pvwindow->GetIntRegisteryValue(2, "RunTime", "UseStrips"));
    }
  else
    {
    this->TriangleStripsCheck->SetState(0);
    }
  this->TriangleStripsCheck->SetCommand(this, "TriangleStripsCallback");
  this->TriangleStripsCheck->SetBalloonHelpString(
    "If this option is chosen, all triangles are converted into triangle "
    "strips before rendering. This may improve the rendering perfermance.");
  
  // Render parameters: immediate mode

  this->ImmediateModeCheck->SetParent(this->RenderParametersFrame->GetFrame());
  this->ImmediateModeCheck->Create(this->GetApplication(), 
                                   "-text \"Use immediate mode rendering\"");
  this->ImmediateModeCheck->SetCommand(this, "ImmediateModeCallback");
  if (pvapp && pvwindow && 
      pvapp->GetRegisteryValue(2, "RunTime", "UseImmediateMode", 0))
    {
    this->ImmediateModeCheck->SetState(
      pvwindow->GetIntRegisteryValue(2, "RunTime", "UseImmediateMode"));
    }
  else
    {
    this->ImmediateModeCheck->SetState(1);
    }
  this->ImmediateModeCheck->SetBalloonHelpString(
    "When this option is off, OpenGL display lists are used when rendering."
    "Using display lists improves performance for small datasets "
    "but is not recommended for large datasets due to excessive memory "
    "usage and long display list creating times.");

  // Render parameters: pack

  this->Script("pack %s %s %s -side top -anchor w",
               this->ParallelProjectionCheck->GetWidgetName(),
               this->TriangleStripsCheck->GetWidgetName(),
               this->ImmediateModeCheck->GetWidgetName());

  // Create the render module user interface.
  char* rmuiClassName;
  vtkProcessModule* pm = pvapp->GetProcessModule();
  rmuiClassName = new char[strlen(pm->GetOptions()->GetRenderModuleName()) + 20];
  sprintf(rmuiClassName, "vtkPV%sUI", pm->GetOptions()->GetRenderModuleName());
  vtkObject* rmui = vtkInstantiator::CreateInstance(rmuiClassName);
  vtkPVRenderModuleUI* rmuio = vtkPVRenderModuleUI::SafeDownCast(rmui);
  if ( rmuio )
    {
    this->RenderModuleUI = rmuio;
    this->RenderModuleUI->SetRenderModule(pm->GetRenderModule());
    this->RenderModuleUI->SetParent(this->GeneralProperties->GetFrame());
    this->RenderModuleUI->Create(this->GetApplication(),0);
    this->RenderModuleUI->SetTraceReferenceObject(this);
    this->RenderModuleUI->SetTraceReferenceCommand("GetRenderModuleUI");
    this->Script("pack %s -padx 2 -pady 2 -fill x -expand yes -anchor w",
      this->RenderModuleUI->GetWidgetName());
    // Disable compositing if the server does not support remote rendering.
    rmuio->Initialize();
    }
  else
    {
    this->RenderModuleUI = 0;
    }
  delete [] rmuiClassName;
  rmuiClassName = NULL;

  // Interface settings

  this->InterfaceSettingsFrame->SetParent(this->GeneralProperties->GetFrame());
  this->InterfaceSettingsFrame->ShowHideFrameOn();
  this->InterfaceSettingsFrame->Create(this->GetApplication(),0);
  this->InterfaceSettingsFrame->SetLabel("3D Interface Settings");
  this->Script("pack %s -padx 2 -pady 2 -fill x -expand yes -anchor w",
               this->InterfaceSettingsFrame->GetWidgetName());

  // Interface settings: 3D widgets

  this->Display3DWidgets->SetParent(
    this->InterfaceSettingsFrame->GetFrame());
  this->Display3DWidgets->Create(this->GetApplication(), 0);
  this->Display3DWidgets->SetText("Display 3D widgets automatically");
  this->Display3DWidgets->SetBalloonHelpString(
    "When this advanced option is on, all 3D widgets (manipulators) are "
    "visible when created. Turn this off to avoid unnecessary "
    "renders when working with large data and not using 3D widgets "
    "to adjust parameters.");
  this->Display3DWidgets->SetCommand(this, "Display3DWidgetsCallback");

  if (!this->GetApplication()->GetRegisteryValue(2,"RunTime","Display3DWidgets",0)||
    this->GetPVWindow()->GetIntRegisteryValue(2,"RunTime","Display3DWidgets"))
    {
    this->SetDisplay3DWidgets(1);
    }
  else
    {
    this->SetDisplay3DWidgets(0);
    }
  
  // Interface settings: pack

  this->Script("pack %s -side top -padx 2 -pady 2 -anchor w",
               this->Display3DWidgets->GetWidgetName());

  // Orientation axes settings
  
  this->OrientationAxesFrame->SetParent(this->AnnotationProperties->GetFrame());
  this->OrientationAxesFrame->ShowHideFrameOn();
  this->OrientationAxesFrame->Create(this->GetApplication(), 0);
  this->OrientationAxesFrame->SetLabel("Orientation Axes");
  this->Script("pack %s -padx 2 -pady 2 -fill x -expand yes -anchor w",
               this->OrientationAxesFrame->GetWidgetName());
  
  // Orientation axes settings: visibility check
  
  this->OrientationAxesCheck->SetParent(this->OrientationAxesFrame->GetFrame());
  this->OrientationAxesCheck->Create(this->GetApplication(), 0);
  this->OrientationAxesCheck->SetText("Display orientation axes");
  this->OrientationAxesCheck->SetCommand(this, "OrientationAxesCheckCallback");
  if (pvapp && pvwindow &&
      pvapp->GetRegisteryValue(2, "RunTime", "OrientationAxesVisibility", 0))
    {
    this->OrientationAxesCheck->SetState(
      pvwindow->GetIntRegisteryValue(2, "RunTime", "OrientationAxesVisibility"));
    }
  else
    {
    this->OrientationAxesCheck->SetState(1);
    }
  this->OrientationAxesCheckCallback();
  
  // Orientation axes settings: interactive check
  
  this->OrientationAxesInteractiveCheck->SetParent(
    this->OrientationAxesFrame->GetFrame());
  this->OrientationAxesInteractiveCheck->Create(this->GetApplication(), 0);
  this->OrientationAxesInteractiveCheck->SetText("Interactive");
  this->OrientationAxesInteractiveCheck->SetCommand(this, "OrientationAxesInteractiveCallback");
  if (pvapp && pvwindow &&
      pvapp->GetRegisteryValue(2, "RunTime", "OrientationAxesInteractivity", 0))
    {
    this->OrientationAxesInteractiveCheck->SetState(
      pvwindow->GetIntRegisteryValue(2, "RunTime", "OrientationAxesInteractivity"));
    }
  else
    {
    this->OrientationAxesInteractiveCheck->SetState(0);
    }
  
  this->OrientationAxesInteractiveCallback();
  
  // Orientation axes settings: outline color
  
  this->OrientationAxesOutlineColor->SetParent(
    this->OrientationAxesFrame->GetFrame());
  this->OrientationAxesOutlineColor->SetText("Set Outline Color");
  this->OrientationAxesOutlineColor->Create(this->GetApplication(), 0);
  this->OrientationAxesOutlineColor->SetCommand(this, "SetOrientationAxesOutlineColor");
  if (pvapp && pvwindow)
    {
    pvwindow->RetrieveColor(2, "OrientationAxesOutline", rgb);
    if (rgb[0] == -1)
      {
      rgb[0] = 1.0;
      rgb[1] = 1.0;
      rgb[2] = 1.0;
      }
    }
  else
    {
    rgb[0] = 1.0;
    rgb[1] = 1.0;
    rgb[2] = 1.0;
    }
  this->SetOrientationAxesOutlineColor(rgb[0], rgb[1], rgb[2]);
    
  // Orientation axes settings: text color
  
  this->OrientationAxesTextColor->SetParent(
    this->OrientationAxesFrame->GetFrame());
  this->OrientationAxesTextColor->SetText("Set Axis Label Color");
  this->OrientationAxesTextColor->Create(this->GetApplication(), 0);
  this->OrientationAxesTextColor->SetCommand(this, "SetOrientationAxesTextColor");
  if (pvapp && pvwindow)
    {
    pvwindow->RetrieveColor(2, "OrientationAxesText", rgb);
    if (rgb[0] == -1)
      {
      rgb[0] = 1.0;
      rgb[1] = 1.0;
      rgb[2] = 1.0;
      }
    }
  else
    {
    rgb[0] = 1.0;
    rgb[1] = 1.0;
    rgb[2] = 1.0;
    }
  this->SetOrientationAxesTextColor(rgb[0], rgb[1], rgb[2]);

  // Orientation axes settings: pack
  
  this->Script("pack %s %s -padx 2 -side top -anchor w",
               this->OrientationAxesCheck->GetWidgetName(),
               this->OrientationAxesInteractiveCheck->GetWidgetName());
  this->Script("pack %s %s -padx 2 -fill x -side top -anchor w",
               this->OrientationAxesOutlineColor->GetWidgetName(),
               this->OrientationAxesTextColor->GetWidgetName());

  // Orientation axes widget
  
  this->OrientationAxes->SetParentRenderer(this->GetRenderer());

  char buffer[1024];
  double vp[4];
  if (pvapp && pvwindow &&
      pvapp->GetRegisteryValue(2, "RunTime", "OrientationAxesViewport", buffer))
    {
    if (*buffer)
      {
      sscanf(buffer, "%lf %lf %lf %lf", vp, vp+1, vp+2, vp+3);
      this->OrientationAxes->SetViewport(vp[0], vp[1], vp[2], vp[3]);
      }
    else
      {
      this->OrientationAxes->SetViewport(0, 0, 0.2, 0.2);
      }
    }
  else
    {
    this->OrientationAxes->SetViewport(0, 0, 0.2, 0.2);
    }
  
  // Camera settings
  this->Notebook->AddPage("Camera", 
                          "Camera and viewing navigation properties page");
  vtkKWWidget* page = this->Notebook->GetFrame("Camera");

  vtkKWFrame* frame = vtkKWFrame::New();
  frame->SetParent(page);
  frame->ScrollableOn();
  frame->Create(this->GetApplication(),0);
  this->Script("pack %s -fill both -expand yes", frame->GetWidgetName());

  // Camera: standard views

  this->StandardViewsFrame->SetParent( frame->GetFrame() );
  this->StandardViewsFrame->ShowHideFrameOn();
  this->StandardViewsFrame->Create(this->GetApplication(), 0);
  this->StandardViewsFrame->SetLabel("Standard Views");

  const char *views_grid_settings = " -padx 1 -pady 1 -ipadx 5 -sticky ew";

  this->XMaxViewButton->SetParent(this->StandardViewsFrame->GetFrame());
  this->XMaxViewButton->SetLabel("+X");
  this->XMaxViewButton->Create(this->GetApplication(), "");
  this->XMaxViewButton->SetCommand(this, "StandardViewCallback 1 0 0");
  this->XMaxViewButton->SetBalloonHelpString(
    "Looking down X axis from (1,0,0)");
  this->Script("grid configure %s -column 0 -row 0 %s",
               this->XMaxViewButton->GetWidgetName(), views_grid_settings);

  this->XMinViewButton->SetParent(this->StandardViewsFrame->GetFrame());
  this->XMinViewButton->SetLabel("-X");
  this->XMinViewButton->Create(this->GetApplication(), "");
  this->XMinViewButton->SetCommand(this, "StandardViewCallback -1 0 0");
  this->XMinViewButton->SetBalloonHelpString(
    "Looking down X axis from (-1,0,0)");
  this->Script("grid configure %s -column 0 -row 1 %s",
               this->XMinViewButton->GetWidgetName(), views_grid_settings);

  this->YMaxViewButton->SetParent(this->StandardViewsFrame->GetFrame());
  this->YMaxViewButton->SetLabel("+Y");
  this->YMaxViewButton->Create(this->GetApplication(), "");
  this->YMaxViewButton->SetCommand(this, "StandardViewCallback 0 1 0");
  this->YMaxViewButton->SetBalloonHelpString(
    "Looking down Y axis from (0,1,0)");
  this->Script("grid configure %s -column 1 -row 0 %s",
               this->YMaxViewButton->GetWidgetName(), views_grid_settings);

  this->YMinViewButton->SetParent(this->StandardViewsFrame->GetFrame());
  this->YMinViewButton->SetLabel("-Y");
  this->YMinViewButton->Create(this->GetApplication(), "");
  this->YMinViewButton->SetCommand(this, "StandardViewCallback 0 -1 0");
  this->YMinViewButton->SetBalloonHelpString(
    "Looking down Y axis from (0,-1,0)");
  this->Script("grid configure %s -column 1 -row 1 %s",
               this->YMinViewButton->GetWidgetName(), views_grid_settings);

  this->ZMaxViewButton->SetParent(this->StandardViewsFrame->GetFrame());
  this->ZMaxViewButton->SetLabel("+Z");
  this->ZMaxViewButton->Create(this->GetApplication(), "");
  this->ZMaxViewButton->SetCommand(this, "StandardViewCallback 0 0 1");
  this->ZMaxViewButton->SetBalloonHelpString(
    "Looking down Z axis from (0,0,1)");
  this->Script("grid configure %s -column 2 -row 0 %s",
               this->ZMaxViewButton->GetWidgetName(), views_grid_settings);

  this->ZMinViewButton->SetParent(this->StandardViewsFrame->GetFrame());
  this->ZMinViewButton->SetLabel("-Z");
  this->ZMinViewButton->Create(this->GetApplication(), "");
  this->ZMinViewButton->SetCommand(this, "StandardViewCallback 0 0 -1");
  this->ZMinViewButton->SetBalloonHelpString(
    "Looking down Z axis from (0,0,-1)");
  this->Script("grid configure %s -column 2 -row 1 %s",
               this->ZMinViewButton->GetWidgetName(), views_grid_settings);

  // Camera: stored camera position

  int cc;
  this->CameraIconsFrame->SetParent(frame->GetFrame());
  this->CameraIconsFrame->ShowHideFrameOn();
  this->CameraIconsFrame->Create(this->GetApplication(), 0);
  this->CameraIconsFrame->SetLabel("Stored Camera Positions");

  vtkKWWidget* cframe = this->CameraIconsFrame->GetFrame();
  for ( cc = 0; cc < 6; cc ++ )
    {
    int x, y;
    this->CameraIcons[cc]->SetRenderView(this);
    this->CameraIcons[cc]->SetParent(cframe);
    this->CameraIcons[cc]->Create(this->GetApplication(), "");

    x = cc % 3;
    y = cc / 3;

    this->Script("grid configure %s -column %d -row %d "
                 "-padx 1 -pady 1 -ipadx 0 -ipady 0 -sticky news",
                 this->CameraIcons[cc]->GetWidgetName(),
                 x, y);
                 
    }

  // Camera: manipulators

  this->ManipulatorControl2D->SetParent(frame->GetFrame());
  this->ManipulatorControl2D->Create(pvapp, 0);
  this->ManipulatorControl2D->SetLabel("2D Movements");
  this->ManipulatorControl2D->SetTraceReferenceObject(this);
  this->ManipulatorControl2D->SetTraceReferenceCommand("GetManipulatorControl2D");
  
  this->ManipulatorControl3D->SetParent(frame->GetFrame());
  this->ManipulatorControl3D->Create(pvapp, 0);
  this->ManipulatorControl3D->SetLabel("3D Movements");
  this->ManipulatorControl3D->SetTraceReferenceObject(this);
  this->ManipulatorControl3D->SetTraceReferenceCommand("GetManipulatorControl3D");
  
  // Camera: camera control frame

  this->CameraControlFrame->SetParent(frame->GetFrame());
  this->CameraControlFrame->ShowHideFrameOn();
  this->CameraControlFrame->Create(this->GetApplication(), 0);
  this->CameraControlFrame->SetLabel("Camera Orientation");
  
  // Camera: camera control
  
  this->CameraControl->SetParent(this->CameraControlFrame->GetFrame());
  this->CameraControl->Create(this->GetApplication(), 0);
  this->CameraControl->SetInteractorStyle(this->GetPVWindow()->GetCenterOfRotationStyle());
  this->CameraControl->SetRenderView(this);
  this->CameraControl->SetTraceReferenceObject(this);
  this->CameraControl->SetTraceReferenceCommand("GetCameraControl");
  
  this->Script("pack %s -side top -fill x -expand y", this->CameraControl->GetWidgetName());
  
  // Camera: pack

  this->Script("pack %s %s %s %s %s -padx 2 -pady 2 -fill x -expand 1 -anchor w",
               this->StandardViewsFrame->GetWidgetName(),
               this->CameraIconsFrame->GetWidgetName(),
               this->ManipulatorControl2D->GetWidgetName(),
               this->ManipulatorControl3D->GetWidgetName(),
               this->CameraControlFrame->GetWidgetName());

  frame->Delete();

  this->Notebook->Raise("General");
}

//----------------------------------------------------------------------------
void vtkPVRenderView::StandardViewCallback(float x, float y, float z)
{
  this->AddTraceEntry("$kw(%s) StandardViewCallback %f %f %f",
                      this->GetTclName(), x, y, z);
  
  vtkCamera *cam = this->GetRenderer()->GetActiveCamera();
  cam->SetFocalPoint(0.0, 0.0, 0.0);
  cam->SetPosition(x, y, z);
  if (y == 0.0)
    {
    cam->SetViewUp(0.0, 1.0, 0.0);
    }
  else
    {
    cam->SetViewUp(1.0, 0.0, 0.0);
    }

  this->ResetCamera();
  this->EventuallyRender();
}

//----------------------------------------------------------------------------
void vtkPVRenderView::UpdateNavigationWindow(vtkPVSource *currentSource, 
                                             int nobind)
{
  if (currentSource == 0)
    {
    return;
    }
  if (this->NavigationWindow)
    {
    this->NavigationWindow->SetCreateSelectionBindings(!nobind);
    this->NavigationWindow->Update(currentSource);
    }
  if (this->SelectionWindow)
    {
    this->SelectionWindow->SetCreateSelectionBindings(!nobind);
    this->SelectionWindow->Update(currentSource);
    }
}

//----------------------------------------------------------------------------
void vtkPVRenderView::SetRendererBackgroundColor(double r, double g, double b)
{
  vtkPVApplication *pvApp = this->GetPVApplication();
 
  // Save the color in registry here because there is no good place to do 
  // it upon exiting or destruction.  We have to delete the render window
  // before the TKRenderWidget, so we loose access to the background color.
  // The other option is to change vtkKWView to get the background color
  // locally from the color button.
  // This used to be in vtkPVWindow.  I had to move it here because
  // the renderer was destructing before the vtkPVWindow.
  double rgb[3];
  rgb[0] = r; rgb[1] = g; rgb[2] = b;
  this->GetPVWindow()->SaveColor(2, "RenderViewBG", rgb); 
 
  // Set the color of the interface button.
  this->RendererBackgroundColor->SetColor(r, g, b);
  // Since setting the color of the button from a script does
  // not invoke the callback, We also trace the view.
  this->AddTraceEntry("$kw(%s) SetRendererBackgroundColor %f %f %f",
                      this->GetTclName(), r, g, b);
  pvApp->GetProcessModule()->GetRenderModule()->SetBackgroundColor(r, g, b);
  this->EventuallyRender();
}

//----------------------------------------------------------------------------
// a litle more complex than just "bind $widget <Expose> {%W Render}"
// we have to handle all pending expose events otherwise they que up.
void vtkPVRenderView::Exposed()
{  
  if (this->InExpose)
    {
    return;
    }
  this->InExpose = 1;
  this->Script("update");
  this->EventuallyRender();
  this->InExpose = 0;
}

//----------------------------------------------------------------------------
// called on Configure event. Configure events might not generate an
// expose event if the size of the view gets smaller.
// At the moment, just call Exposed(), which does what we want to do,
// i.e. eventually render. If an Expose event was also generated after
// that Configure event, it will be discard because of the InExpose ivar
// logic (see above)
void vtkPVRenderView::Configured()
{
  vtkPVRenderModule* rm = this->GetPVApplication()->GetProcessModule()->GetRenderModule();
  if (this->BlockRender)
    {
    this->BlockRender = 2;
    return;
    }
  if (rm)
    {
    rm->InteractiveRender();
    }
}


//----------------------------------------------------------------------------
void vtkPVRenderView::ResetCamera()
{
  double bds[6];


  this->GetPVApplication()->GetProcessModule()->GetRenderModule()->ComputeVisiblePropBounds(bds);
  if (bds[0] <= bds[1] && bds[2] <= bds[3] && bds[4] <= bds[5])
    {
    this->GetRenderer()->ResetCamera(bds);
    }
}

//----------------------------------------------------------------------------
void vtkPVRenderView::SetCameraState(float p0, float p1, float p2,
                                     float fp0, float fp1, float fp2,
                                     float up0, float up1, float up2)
{
  vtkCamera *cam; 
  
  // This is to trace effects of loaded scripts. 
  this->AddTraceEntry(
    "$kw(%s) SetCameraState %.3f %.3f %.3f  %.3f %.3f %.3f  %.3f %.3f %.3f", 
    this->GetTclName(), p0, p1, p2, fp0, fp1, fp2, up0, up1, up2); 
  
  cam = this->GetRenderer()->GetActiveCamera(); 
  cam->SetPosition(p0, p1, p2); 
  cam->SetFocalPoint(fp0, fp1, fp2); 
  cam->SetViewUp(up0, up1, up2); 
  
  this->EventuallyRender(); 
}

//----------------------------------------------------------------------------
void vtkPVRenderView::SetCameraParallelScale(float scale)
{
  this->AddTraceEntry("$kw(%s) SetCameraParallelScale %.3f",
                      this->GetTclName(), scale);
  
  this->GetRenderer()->GetActiveCamera()->SetParallelScale(scale);
  
  this->EventuallyRender();
}

//----------------------------------------------------------------------------
void vtkPVRenderView::AddBindings()
{
  this->Script("bind %s <Motion> {%s MotionCallback %%x %%y}",
               this->VTKWidget->GetWidgetName(), this->GetTclName());
}
    
//----------------------------------------------------------------------------
vtkPVApplication* vtkPVRenderView::GetPVApplication()
{
  if (this->GetApplication() == NULL)
    {
    return NULL;
    }
  
  if (this->GetApplication()->IsA("vtkPVApplication"))
    {  
    return (vtkPVApplication*)(this->GetApplication());
    }
  else
    {
    vtkErrorMacro("Bad typecast");
    return NULL;
    } 
}


void vtkPVRenderView::ForceRender()
{
  vtkPVApplication *pvApp = this->GetPVApplication();
  if ( pvApp )
    {
    pvApp->GetProcessModule()->SetGlobalLODFlag(0);
    pvApp->GetProcessModule()->GetRenderModule()->StillRender();
    }
}

//----------------------------------------------------------------------------
void vtkPVRenderView::Render()
{
  int abort;
  if (this->BlockRender)
    {
    this->BlockRender = 2;
    return;
    }

  // Some aborts require us to que another render.
  abort = this->ShouldIAbort();
  if (abort)
    {
    if (abort == 1)
      {
      this->EventuallyRender();
      }
    return;
    }

  vtkPVRenderModule* rm = this->GetPVApplication()->GetProcessModule()->GetRenderModule();
  if (rm)
    {
    rm->InteractiveRender();
    }
}





//----------------------------------------------------------------------------
// There are a couple of ways I could do this.
// I could allow eventually render callback to occur
// and abort the render using the disable rendering flag ...
void vtkPVRenderView::UpdateTclButAvoidRendering()
{
  int saveRender = 0;

  // Remove any pending renders.
  if (this->TimerToken)
    {
    saveRender = 1;
    Tcl_DeleteTimerHandler( this->TimerToken );
    this->TimerToken = NULL;
    }

  this->Script("update");

  // Add render pending back.
  if (saveRender)
    {
    this->EventuallyRender();
    }
}

//----------------------------------------------------------------------------
void vtkPVRenderView::StartBlockingRender()
{
  vtkDebugMacro("Start blocking render requests");
  if ( this->BlockRender > 0 )
    {
    return;
    }
  this->BlockRender = 1;
}

//----------------------------------------------------------------------------
void vtkPVRenderView::EndBlockingRender()
{
  vtkDebugMacro("Stop blocking render requests");
  if ( this->BlockRender > 1 )
    {
    vtkDebugMacro("There was a render request, so call render");
    this->EventuallyRender();
    }
  this->BlockRender = 0;
}


//----------------------------------------------------------------------------
extern "C" { void PVRenderView_IdleRender(ClientData arg); }

void PVRenderView_IdleRender(ClientData arg)
{
  vtkPVRenderView *me = (vtkPVRenderView *)arg;
  me->EventuallyRenderCallBack();
}

//----------------------------------------------------------------------------
void vtkPVRenderView::EventuallyRender()
{
  vtkDebugMacro("Enqueue EventuallyRender request");

  this->RenderTimer->StartTimer();
  if ( !this->TimerToken )
    {
    this->TimerToken = Tcl_CreateTimerHandler(110, 
                                              PVRenderView_IdleRender, 
                                              (ClientData)this);
    }
}

//----------------------------------------------------------------------------
void vtkPVRenderView::EventuallyRenderCallBack()
{
  int abortFlag;
  double elapsedTime;
  
  this->RenderTimer->StopTimer();
  
  elapsedTime = this->RenderTimer->GetElapsedTime();
  abortFlag = this->ShouldIAbort();
  
  this->TimerToken = NULL;
  
  if ( elapsedTime < 0.1 || abortFlag != 0 )
    {
    if ( abortFlag == 1 )
      {
      this->TimerToken = Tcl_CreateTimerHandler(200, 
                                                PVRenderView_IdleRender, 
                                                (ClientData)this);    
      }
    else if ( elapsedTime < 0.1 )
      {
      this->TimerToken = Tcl_CreateTimerHandler(100, 
                                                PVRenderView_IdleRender, 
                                                (ClientData)this);    
      }
    return;
    }
  
  if ( this->BlockRender )
    {
    this->BlockRender = 2;
    return;
    }

  vtkPVApplication *pvApp = this->GetPVApplication();
  pvApp->GetProcessModule()->SetGlobalLODFlag(0);
  pvApp->GetProcessModule()->GetRenderModule()->StillRender();
}

//----------------------------------------------------------------------------
void vtkPVRenderView::TriangleStripsCallback()
{
  if (this->TriangleStripsCheck->GetState())
    {
    vtkTimerLog::MarkEvent("--- Enable triangle strips.");
    }
  else
    {
    vtkTimerLog::MarkEvent("--- Disable triangle strips.");
    }

  this->SetUseTriangleStrips(this->TriangleStripsCheck->GetState());
}

//----------------------------------------------------------------------------
void vtkPVRenderView::SetUseTriangleStrips(int state)
{
  vtkPVWindow *pvWin;
  vtkPVSourceCollection *sources;
  vtkPVSource *pvs;
  vtkPVApplication *pvApp;
  int numParts, partIdx;

  this->AddTraceEntry("$kw(%s) SetUseTriangleStrips %d", this->GetTclName(),
                      state);

  if (this->TriangleStripsCheck->GetState() != state)
    {
    this->TriangleStripsCheck->SetState(state);
    }
  
  if ( ! this->ImmediateModeCheck->GetState() && ! state)
    { // Make sure immediate mode is on when strips are off.
    this->SetUseImmediateMode(1);
    }

  pvApp = this->GetPVApplication();
  vtkPVProcessModule* pm = pvApp->GetProcessModule();
  pvWin = this->GetPVWindow();
  if (pvWin == NULL)
    {
    vtkErrorMacro("Missing window.");
    return;
    }
  sources = pvWin->GetSourceList("Sources");

  // It would be nice to get the displays through the render module.  
  sources->InitTraversal();
  while ( (pvs = sources->GetNextPVSource()) )
    {
    numParts = pvs->GetNumberOfParts();
    for (partIdx = 0; partIdx < numParts; ++partIdx)
      {
      pvs->GetPartDisplay()->SetUseTriangleStrips(state);
      }
    }
  pm->SendStream(vtkProcessModule::DATA_SERVER);
  // Save this selection on the server manager so new
  // part displays will have it a s a default.
  pm->SetUseTriangleStrips(state);

  this->EventuallyRender();
}

//----------------------------------------------------------------------------
void vtkPVRenderView::ParallelProjectionOn()
{
  this->AddTraceEntry("$kw(%s) ParallelProjectionOn", this->GetTclName());

  if (!this->ParallelProjectionCheck->GetState())
    {
    this->ParallelProjectionCheck->SetState(1);
    }
  this->GetRenderer()->GetActiveCamera()->ParallelProjectionOn();
  this->EventuallyRender();
}

//----------------------------------------------------------------------------
void vtkPVRenderView::ParallelProjectionOff()
{
  this->AddTraceEntry("$kw(%s) ParallelProjectionOff", this->GetTclName());
  if (this->ParallelProjectionCheck->GetState())
    {
    this->ParallelProjectionCheck->SetState(0);
    }
  this->GetRenderer()->GetActiveCamera()->ParallelProjectionOff();
  this->EventuallyRender();
}

//----------------------------------------------------------------------------
void vtkPVRenderView::ParallelProjectionCallback()
{

  if (this->ParallelProjectionCheck->GetState())
    {
    vtkTimerLog::MarkEvent("--- Enable parallel projection.");
    this->ParallelProjectionOn();
    }
  else
    {
    vtkTimerLog::MarkEvent("--- Disable parallel projection.");
    this->ParallelProjectionOff();
    }
}

//----------------------------------------------------------------------------
void vtkPVRenderView::ImmediateModeCallback()
{
  if (this->ImmediateModeCheck->GetState())
    {
    vtkTimerLog::MarkEvent("--- Disable display lists.");
    }
  else
    {
    vtkTimerLog::MarkEvent("--- Enable display lists.");
    }
  
  this->SetUseImmediateMode(this->ImmediateModeCheck->GetState());
}

//----------------------------------------------------------------------------
void vtkPVRenderView::SetUseImmediateMode(int state)
{
  vtkPVWindow *pvWin;
  vtkPVSourceCollection *sources;
  vtkPVSource *pvs;
  int partIdx, numParts;

  this->AddTraceEntry("$kw(%s) SetUseImmediateMode %d", this->GetTclName(),
                      state);

  if (this->ImmediateModeCheck->GetState() != state)
    {
    this->ImmediateModeCheck->SetState(state);
    }
  
  if ( ! state && ! this->TriangleStripsCheck->GetState())
    { // Make sure triangle strips are on.
    // When immediate mode is off, triangle strips must be on.
    this->SetUseTriangleStrips(1);
    }

  pvWin = this->GetPVWindow();
  if (pvWin == NULL)
    {
    vtkErrorMacro("Missing window.");
    return;
    }
  sources = pvWin->GetSourceList("Sources");
  
  sources->InitTraversal();
  while ( (pvs = sources->GetNextPVSource()) )
    {
    numParts = pvs->GetNumberOfParts();
    for (partIdx = 0; partIdx < numParts; ++partIdx)
      {
      pvs->GetPartDisplay()->SetUseImmediateMode(state);
      }
    }
  // Save this selection on the server manager so new
  // part displays will have it a s a default.
  this->GetPVApplication()->GetProcessModule()->SetUseImmediateMode(state);

  this->EventuallyRender();
}

//----------------------------------------------------------------------------
vtkPVWindow *vtkPVRenderView::GetPVWindow()
{
  vtkPVWindow *pvWin = vtkPVWindow::SafeDownCast(this->GetParentWindow());

  return pvWin;
}

//----------------------------------------------------------------------------
void vtkPVRenderView::SaveInBatchScript(ofstream* file)
{
  int i;

  *file << "set Ren1 [$proxyManager NewProxy rendering DefaultDisplayWindow]" 
        << endl;
  *file << "  $proxyManager RegisterProxy rendering Ren1 $Ren1" 
        << endl;
  *file << "  $Ren1 UnRegister {}" << endl;

  double* color = this->GetRenderer()->GetBackground();
  *file << "  [$Ren1 GetProperty BackgroundColor] SetElements3 "; 
  for(i=0; i<3; i++)
    {
    *file << color[i] << " ";
    }
  *file << endl;
  int *size = this->GetRenderWindow()->GetSize();
  *file << "  [$Ren1 GetProperty Size] SetElements2 "; 
  for(i=0; i<2; i++)
    {
    *file << size[i] << " ";
    }
  *file << endl;

  vtkCamera *camera;
  camera = this->GetRenderer()->GetActiveCamera();

  double position[3];
  camera->GetPosition(position);
  *file << "  [$Ren1 GetProperty CameraPosition] SetElements3 ";
  for(i=0; i<3; i++)
    {
    *file << position[i] << " ";
    }
  *file << endl;
  
  double focalPoint[3];
  camera->GetFocalPoint(focalPoint);
  *file << "  [$Ren1 GetProperty CameraFocalPoint] SetElements3 ";
  for(i=0; i<3; i++)
    {
    *file << focalPoint[i] << " ";
    }
  *file << endl;

  double viewUp[3];
  camera->GetViewUp(viewUp);
  *file << "  [$Ren1 GetProperty CameraViewUp] SetElements3 "; 
  for(i=0; i<3; i++)
    {
    *file << viewUp[i] << " ";
    }
  *file << endl;

  double viewAngle;
  viewAngle = camera->GetViewAngle();
  *file << "  [$Ren1 GetProperty CameraViewAngle] SetElements1 "
        << viewAngle << endl;

  double clippingRange[2];
  camera->GetClippingRange(clippingRange);
  *file << "  [$Ren1 GetProperty CameraClippingRange] SetElements2 ";
  for(i=0; i<2; i++)
    {
    *file << clippingRange[i] << " ";
    }
  *file << endl;
}


//----------------------------------------------------------------------------
void vtkPVRenderView::SaveState(ofstream* file)
{
  vtkCamera *camera;
  double position[3];
  double focalPoint[3];
  double viewUp[3];
  double *color;

  color = this->GetRenderer()->GetBackground();
  *file << "$kw(" << this->GetTclName() << ") SetRendererBackgroundColor " 
        << color[0] << " " << color[1] << " " << color[2] << endl;

  camera = this->GetRenderer()->GetActiveCamera();
  camera->GetPosition(position);
  camera->GetFocalPoint(focalPoint);
  camera->GetViewUp(viewUp);
  
  if (camera->GetParallelProjection())
    {
    *file << "$kw(" << this->GetTclName() << ") ParallelProjectionOn" << endl;
    *file << "$kw(" << this->GetTclName() << ") SetCameraParallelScale "
          << camera->GetParallelScale() << endl;
    }
  else
    {
    *file << "$kw(" << this->GetTclName() << ") ParallelProjectionOff" << endl;
    }
  
  *file << "$kw(" << this->GetTclName() << ") SetCameraState " 
        << position[0] << " " << position[1] << " " << position[2] << " "
        << focalPoint[0] << " " << focalPoint[1] << " " << focalPoint[2] << " "
        << viewUp[0] << " " << viewUp[1] << " " << viewUp[2] << endl; 

  *file << "$kw(" << this->GetTclName() << ") SetUseTriangleStrips "
        << this->TriangleStripsCheck->GetState() << endl;
  
  *file << "$kw(" << this->GetTclName() << ") SetUseImmediateMode "
        << this->ImmediateModeCheck->GetState() << endl;
  
  if (this->RenderModuleUI)
    {
    *file << "set kw(" << this->RenderModuleUI->GetTclName()
          << ") [$kw(" << this->GetTclName() << ") GetRenderModuleUI]" << endl;
    this->RenderModuleUI->SaveState(file);
    }
  
  *file << "set kw(" << this->CornerAnnotation->GetTclName()
        << ") [$kw(" << this->GetTclName() << ") GetCornerAnnotation]" << endl;
  this->CornerAnnotation->SaveState(file);
  
  *file << "$kw(" << this->GetTclName() << ") SetOrientationAxesVisibility "
        << this->OrientationAxesCheck->GetState() << endl;
  *file << "$kw(" << this->GetTclName() << ") SetOrientationAxesInteractivity "
        << this->OrientationAxesInteractiveCheck->GetState() << endl;
  color = this->OrientationAxesOutlineColor->GetColor();
  *file << "$kw(" << this->GetTclName() << ") SetOrientationAxesOutlineColor "
        << color[0] << " " << color[1] << " " << color[2] << endl;
  color = this->OrientationAxesTextColor->GetColor();
  *file << "$kw(" << this->GetTclName() << ") SetOrientationAxesTextColor "
        << color[0] << " " << color[1] << " " << color[2] << endl;
  
  *file << "set kw(" << this->ManipulatorControl2D->GetTclName()
        << ") [$kw(" << this->GetTclName() << ") GetManipulatorControl2D]"
        << endl;
  this->ManipulatorControl2D->SaveState(file);
  
  *file << "set kw(" << this->ManipulatorControl3D->GetTclName()
        << ") [$kw(" << this->GetTclName() << ") GetManipulatorControl3D]"
        << endl;
  this->ManipulatorControl3D->SaveState(file);
}

//----------------------------------------------------------------------------
int* vtkPVRenderView::GetRenderWindowSize()
{
  if ( this->GetRenderWindow() )
    {
    return this->GetRenderWindow()->GetSize();
    }
  return 0;
}

//----------------------------------------------------------------------------
void vtkPVRenderView::UpdateCameraManipulators()
{
  vtkPVInteractorStyleControl *iscontrol3D = this->GetManipulatorControl3D();
  vtkPVInteractorStyleControl *iscontrol2D = this->GetManipulatorControl2D();
  iscontrol3D->UpdateMenus();
  iscontrol2D->UpdateMenus();
}

//----------------------------------------------------------------------------
void vtkPVRenderView::SetupCameraManipulators()
{
  vtkPVInteractorStyleControl *iscontrol3D = this->GetManipulatorControl3D();
  vtkPVInteractorStyleControl *iscontrol2D = this->GetManipulatorControl2D();

  iscontrol3D->SetCurrentManipulator(0, 0, "Rotate");
  iscontrol3D->SetCurrentManipulator(1, 0, "Pan");
  iscontrol3D->SetCurrentManipulator(2, 0, "Zoom");
  iscontrol3D->SetCurrentManipulator(0, 1, "Roll");
  iscontrol3D->SetCurrentManipulator(1, 1, "Center");
  iscontrol3D->SetCurrentManipulator(2, 1, "Pan");
  iscontrol3D->SetCurrentManipulator(0, 2, "FlyIn");
  iscontrol3D->SetCurrentManipulator(2, 2, "FlyOut");
  iscontrol3D->SetDefaultManipulator("Rotate");
  iscontrol3D->UpdateMenus();

  iscontrol2D->SetCurrentManipulator(0, 1, "Roll");
  iscontrol2D->SetCurrentManipulator(1, 0, "Pan");
  iscontrol2D->SetCurrentManipulator(2, 1, "Pan");
  iscontrol2D->SetCurrentManipulator(2, 0, "Zoom");
  iscontrol2D->SetDefaultManipulator("Pan");
  iscontrol2D->UpdateMenus();
}

//----------------------------------------------------------------------------
void vtkPVRenderView::StoreCurrentCamera(int position)
{
  if ( this->CameraIcons[position] )
    {
    this->CameraIcons[position]->StoreCamera();
    }
}

//----------------------------------------------------------------------------
void vtkPVRenderView::RestoreCurrentCamera(int position)
{
  if ( this->CameraIcons[position] )
    {
    this->CameraIcons[position]->RestoreCamera();
    }
}

//----------------------------------------------------------------------------
void vtkPVRenderView::SaveAsImage(const char* filename) 
{
  this->GetRenderWindow()->SwapBuffersOff();
  this->ForceRender();
  this->Script("update");

  if ( !filename || !*filename )
    {
    vtkErrorMacro("Filename not specified");
    return;
    }
  
  // first get the file name
  vtkWindowToImageFilter *w2i = vtkWindowToImageFilter::New();
  w2i->SetInput(this->GetRenderWindow());
  w2i->ReadFrontBufferOff();
  w2i->ShouldRerenderOff();
  w2i->Update();
  
  this->GetRenderWindow()->SwapBuffersOn();
  this->GetRenderWindow()->Frame();

  int success = 1;
  
  if (!strcmp(filename + strlen(filename) - 4,".bmp"))
    {
    vtkBMPWriter *bmp = vtkBMPWriter::New();
    bmp->SetInput(w2i->GetOutput());
    bmp->SetFileName((char *)filename);
    bmp->Write();
    if (bmp->GetErrorCode() == vtkErrorCode::OutOfDiskSpaceError)
      {
      success = 0;
      }
    bmp->Delete();
    }
  else if (!strcmp(filename + strlen(filename) - 4,".tif"))
    {
    vtkTIFFWriter *tif = vtkTIFFWriter::New();
    tif->SetInput(w2i->GetOutput());
    tif->SetFileName((char *)filename);
    tif->Write();
    if (tif->GetErrorCode() == vtkErrorCode::OutOfDiskSpaceError)
      {
      success = 0;
      }
    tif->Delete();
    }
  else if (!strcmp(filename + strlen(filename) - 4,".ppm"))
    {
    vtkPNMWriter *pnm = vtkPNMWriter::New();
    pnm->SetInput(w2i->GetOutput());
    pnm->SetFileName((char *)filename);
    pnm->Write();
    if (pnm->GetErrorCode() == vtkErrorCode::OutOfDiskSpaceError)
      {
      success = 0;
      }
    pnm->Delete();
    }
  else if (!strcmp(filename + strlen(filename) - 4,".png"))
    {
    vtkPNGWriter *png = vtkPNGWriter::New();
    png->SetInput(w2i->GetOutput());
    png->SetFileName((char *)filename);
    png->Write();
    if (png->GetErrorCode() == vtkErrorCode::OutOfDiskSpaceError)
      {
      success = 0;
      }
    png->Delete();
    }
  else if (!strcmp(filename + strlen(filename) - 4,".jpg"))
    {
    vtkJPEGWriter *jpg = vtkJPEGWriter::New();
    jpg->SetInput(w2i->GetOutput());
    jpg->SetFileName((char *)filename);
    jpg->Write();
    if (jpg->GetErrorCode() == vtkErrorCode::OutOfDiskSpaceError)
      {
      success = 0;
      }
    jpg->Delete();
    }

  w2i->Delete();
  
  if (!success)
    {
    vtkKWMessageDialog::PopupMessage(
      this->GetApplication(), this->ParentWindow, "Write Error",
      "There is insufficient disk space to save this image. The file will be "
      "deleted.");
    }
}


//----------------------------------------------------------------------------
void vtkPVRenderView::ExecuteEvent(vtkObject*, unsigned long event, void* par)
{
  if ( event == vtkCommand::CursorChangedEvent )
    {
    int val = *(static_cast<int*>(par));
    const char* image = "left_ptr";
    switch ( val ) 
      {
      case VTK_CURSOR_ARROW:
        image = "arror";
        break;
      case VTK_CURSOR_SIZENE:
        image = "top_right_corner";
        break;
      case VTK_CURSOR_SIZENW:        
        image = "top_left_corner";
        break;
      case VTK_CURSOR_SIZESW:
        image = "bottom_left_corner";
        break;
      case VTK_CURSOR_SIZESE:
        image = "bottom_right_corner";
        break;
      case VTK_CURSOR_SIZENS:
        image = "sb_v_double_arrow";
        break;
      case VTK_CURSOR_SIZEWE:
        image = "sb_h_double_arrow";
        break;
      case VTK_CURSOR_SIZEALL:
        image = "hand2";
        break;
      case VTK_CURSOR_HAND:
        image = "hand2";
        break;
      }
    this->Script("%s config -cursor %s", 
                 this->GetPVWindow()->GetWidgetName(), image);
    } 
}

//----------------------------------------------------------------------------
void vtkPVRenderView::Enable3DWidget(vtkInteractorObserver *o)
{
  vtkRenderer *ren = this->GetRenderer2D();
  if (!ren)
    {
    return;
    }
  
  vtkRendererCollection *savedRens = vtkRendererCollection::New();
  vtkRenderWindow *renWin = this->GetRenderWindow();
  vtkRendererCollection *rens = renWin->GetRenderers();
  
  vtkRenderer *current;
  
  int i, numRens = rens->GetNumberOfItems();
  int renId = rens->IsItemPresent(ren) - 1;
  rens->InitTraversal();
  for (i = 0; i < numRens; i++)
    {
    current = rens->GetNextItem();
    if (current != ren)
      {
      savedRens->AddItem(current);
      renWin->RemoveRenderer(current);
      }
    }
  
  o->SetEnabled(1);  

  // put the renderers back in the correct order
  renWin->RemoveRenderer(ren);
  savedRens->InitTraversal();
  for (i = 0 ; i < numRens; i++)
    {
    if (i == renId)
      {
      renWin->AddRenderer(ren);
      }
    else
      {
      renWin->AddRenderer(savedRens->GetNextItem());
      }
    }      
  savedRens->Delete();
}

//----------------------------------------------------------------------------
void vtkPVRenderView::SetOrientationAxesVisibility(int val)
{
  if (this->OrientationAxesCheck->GetState() != val)
    {
    this->AddTraceEntry("$kw(%s) SetOrientationAxesVisibility %d",
                        this->GetTclName(), val);
    this->OrientationAxesCheck->SetState(val);
    }
  
  if (!this->OrientationAxes->GetInteractor())
    {
    this->OrientationAxes->SetInteractor(this->GetPVWindow()->GetInteractor());
    }
  if (!this->OrientationAxes->GetParentRenderer())
    {
    this->OrientationAxes->SetParentRenderer(this->GetRenderer());
    }

  this->OrientationAxes->SetEnabled(val);
  this->EventuallyRender();
}

//----------------------------------------------------------------------------
void vtkPVRenderView::OrientationAxesCheckCallback()
{
  int val = this->OrientationAxesCheck->GetState();
  this->AddTraceEntry("$kw(%s) SetOrientationAxesVisibility %d",
                      this->GetTclName(), val);
  this->SetOrientationAxesVisibility(val);
  
  this->GetApplication()->SetRegisteryValue(2, "RunTime",
                                       "OrientationAxesVisibility",
                                       (val ? "1" : "0"));
}

//----------------------------------------------------------------------------
void vtkPVRenderView::SetOrientationAxesInteractivity(int val)
{
  if (this->OrientationAxesInteractiveCheck->GetState() != val)
    {
    this->AddTraceEntry("$kw(%s) SetOrientationAxesInteractivity %d",
                        this->GetTclName(), val);
    this->OrientationAxesInteractiveCheck->SetState(val);
    }
  
  this->OrientationAxes->SetInteractive(val);
}

//----------------------------------------------------------------------------
void vtkPVRenderView::OrientationAxesInteractiveCallback()
{
  int val = this->OrientationAxesInteractiveCheck->GetState();
  this->AddTraceEntry("$kw(%s) SetOrientationAxesInteractivity %d",
                      this->GetTclName(), val);
  this->SetOrientationAxesInteractivity(val);
  this->GetApplication()->SetRegisteryValue(2, "RunTime",
                                       "OrientationAxesInteractivity",
                                       (val ? "1" : "0"));
}

//----------------------------------------------------------------------------
void vtkPVRenderView::SetOrientationAxesOutlineColor(double r, double g, double b)
{
  double *color = this->OrientationAxesOutlineColor->GetColor();
  if (r != color[0] || g != color[1] || b != color[2])
    {
    this->OrientationAxesOutlineColor->SetColor(r, g, b);
    }
  this->AddTraceEntry("$kw(%s) SetOrientationAxesOutlineColor %lf %lf %lf",
                      this->GetTclName(), r, g, b);
  this->OrientationAxes->SetOutlineColor(r, g, b);
  this->GetPVWindow()->SaveColor(2, "OrientationAxesOutline",
                                 this->OrientationAxes->GetOutlineColor());
}

//----------------------------------------------------------------------------
void vtkPVRenderView::SetOrientationAxesTextColor(double r, double g, double b)
{
  double *color = this->OrientationAxesTextColor->GetColor();
  if (r != color[0] || g != color[1] || b != color[2])
    {
    this->OrientationAxesTextColor->SetColor(r, g, b);
    }
  this->AddTraceEntry("$kw(%s) SetOrientationAxesTextColor %lf %lf %lf",
                      this->GetTclName(), r, g, b);
  this->OrientationAxes->SetAxisLabelColor(r, g, b);
  this->GetPVWindow()->SaveColor(2, "OrientationAxesText",
                                 this->OrientationAxes->GetAxisLabelColor());
  this->EventuallyRender();
}

//----------------------------------------------------------------------------
void vtkPVRenderView::SetRenderWindowSize(int x, int y)
{
  this->GetRenderWindow()->SetSize(x,y);
}

//----------------------------------------------------------------------------
void vtkPVRenderView::EditCopy()
{
  vtkWindowToImageFilter *w2i = vtkWindowToImageFilter::New();
  w2i->SetInput(this->GetRenderWindow());
  w2i->ShouldRerenderOff();
  w2i->Update();

#ifdef _WIN32
  // get the pointer to the data
  unsigned char *ptr = 
    (unsigned char *)(w2i->GetOutput()->GetScalarPointer());
  
  LPBITMAPINFOHEADER  lpbi;       // pointer to BITMAPINFOHEADER
  DWORD               dwLen;      // size of memory block
  HANDLE              hDIB = NULL;  // handle to DIB, temp handle
  int *size = this->GetVTKWindow()->GetSize();
  int dataWidth = ((size[0]*3+3)/4)*4;
  int srcWidth = size[0]*3;
  
  if (::OpenClipboard((HWND)this->GetVTKWindow()->GetGenericWindowId()))
    {
    EmptyClipboard();
    
    dwLen = sizeof(BITMAPINFOHEADER) + dataWidth*size[1];
    hDIB = ::GlobalAlloc(GHND, dwLen);
    lpbi = (LPBITMAPINFOHEADER) ::GlobalLock(hDIB);
    
    lpbi->biSize = sizeof(BITMAPINFOHEADER);
    lpbi->biWidth = size[0];
    lpbi->biHeight = size[1];
    lpbi->biPlanes = 1;
    lpbi->biBitCount = 24;
    lpbi->biCompression = BI_RGB;
    lpbi->biClrUsed = 0;
    lpbi->biClrImportant = 0;
    lpbi->biSizeImage = dataWidth*size[1];
    
    // copy the data to the clipboard
    unsigned char *dest = (unsigned char *)lpbi + lpbi->biSize;
    int i,j;
    for (i = 0; i < size[1]; i++)
      {
      for (j = 0; j < size[0]; j++)
        {
        *dest++ = ptr[2];
        *dest++ = ptr[1];
        *dest++ = *ptr;
        ptr += 3;
        }
      dest = dest + (dataWidth - srcWidth);
      }
    
    SetClipboardData (CF_DIB, hDIB);
    ::GlobalUnlock(hDIB);
    CloseClipboard();
    }           
#endif
  w2i->Delete();
}

//----------------------------------------------------------------------------
void vtkPVRenderView::PrintView()
{
#ifdef _WIN32
  this->Superclass::PrintView();
#else
  
  vtkWindowToImageFilter *w2i = vtkWindowToImageFilter::New();
  float DPI=0;
  if (this->GetParentWindow())
    {
    // Is this right? Should DPI be int or float?
    DPI = this->GetParentWindow()->GetPrintTargetDPI();
    }

  if (DPI >= 150.0)
    {
    w2i->SetMagnification(2);
    }
  if (DPI >= 300.0)
    {
    w2i->SetMagnification(3);
    }

  w2i->SetInput(this->GetRenderWindow());
  w2i->Update();
  
  this->Script("tk_getSaveFile -title \"Save Postscript\" -filetypes {{{Postscript} {.ps}}}");
  char* path = 
    strcpy(new char[strlen(this->GetApplication()->GetMainInterp()->result)+1], 
           this->GetApplication()->GetMainInterp()->result);
  if (strlen(path) != 0)
    {
    vtkPostScriptWriter *psw = vtkPostScriptWriter::New();
    psw->SetInput(w2i->GetOutput());
    psw->SetFileName(path);
    psw->Write();
    psw->Delete();

    vtkKWMessageDialog *dlg = vtkKWMessageDialog::New();
    dlg->SetMasterWindow(this->ParentWindow);
    dlg->Create(this->GetApplication(),"");
    dlg->SetText(
      "A postscript file has been generated. You will need to\n"
      "print this file using a print command appropriate for\n"
      "your system. Typically this command is lp or lpr. For\n"
      "additional information on printing a postscript file\n"
      "please contact your system administrator.");
    dlg->Invoke();
    }
  w2i->Delete();
#endif
  this->Printing = 0;
}

void vtkPVRenderView::UpdateEnableState()
{
  this->Superclass::UpdateEnableState();

  this->PropagateEnableState(this->StandardViewsFrame);
  this->PropagateEnableState(this->XMaxViewButton);
  this->PropagateEnableState(this->XMinViewButton);
  this->PropagateEnableState(this->YMaxViewButton);
  this->PropagateEnableState(this->YMinViewButton);
  this->PropagateEnableState(this->ZMaxViewButton);
  this->PropagateEnableState(this->ZMinViewButton);
  this->PropagateEnableState(this->RenderParametersFrame);
  this->PropagateEnableState(this->TriangleStripsCheck);
  this->PropagateEnableState(this->ParallelProjectionCheck);
  this->PropagateEnableState(this->ImmediateModeCheck);
  this->PropagateEnableState(this->RenderModuleUI);
  this->PropagateEnableState(this->InterfaceSettingsFrame);
  this->PropagateEnableState(this->Display3DWidgets);
  this->PropagateEnableState(this->OrientationAxesFrame);
  this->PropagateEnableState(this->OrientationAxesCheck);
  this->PropagateEnableState(this->OrientationAxesInteractiveCheck);
  this->PropagateEnableState(this->OrientationAxesOutlineColor);
  this->PropagateEnableState(this->OrientationAxesTextColor);
  this->PropagateEnableState(this->SplitFrame);
  this->PropagateEnableState(this->NavigationFrame);
  this->PropagateEnableState(this->NavigationWindow);
  this->PropagateEnableState(this->SelectionWindow);
  this->PropagateEnableState(this->NavigationWindowButton);
  this->PropagateEnableState(this->SelectionWindowButton);
  this->PropagateEnableState(this->TopLevelRenderWindow);
  this->PropagateEnableState(this->ManipulatorControl2D);
  this->PropagateEnableState(this->ManipulatorControl3D);
  this->PropagateEnableState(this->CameraControlFrame);
  this->PropagateEnableState(this->CameraIconsFrame);
  int cc;
  for ( cc = 0; cc < 6; cc ++ )
    {
    this->PropagateEnableState(this->CameraIcons[cc]);
    }
  this->PropagateEnableState(this->CameraControlFrame);
  this->PropagateEnableState(this->CameraControl);
  this->PropagateEnableState(this->PropertiesButton);
}

//----------------------------------------------------------------------------
void vtkPVRenderView::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os,indent);
  os << indent << "ImmediateModeCheck: " 
     << this->GetImmediateModeCheck() << endl;
  os << indent << "SplitFrame: " 
     << this->GetSplitFrame() << endl;
  os << indent << "NavigationFrame: " 
     << this->GetNavigationFrame() << endl;
  os << indent << "TriangleStripsCheck: " 
     << this->GetTriangleStripsCheck() << endl;
  os << indent << "ParallelProjectionCheck: "
     << this->GetParallelProjectionCheck();
  os << indent << "ManipulatorControl2D: " 
     << this->ManipulatorControl2D << endl;
  os << indent << "ManipulatorControl3D: " 
     << this->ManipulatorControl3D << endl;
  os << indent << "RenderModuleUI: " << this->RenderModuleUI << endl;
  os << indent << "CameraControl: " << this->CameraControl << endl;
  os << indent << "CameraControlFrame: " << this->CameraControlFrame << endl;
  os << indent << "StandardViewsFrame: " << this->StandardViewsFrame << endl;
  os << indent << "CameraIconsFrame: " << this->CameraIconsFrame << endl;
  os << indent << "OrientationAxes: " << this->OrientationAxes << endl;
  os << indent << "OrientationAxesFrame: " << this->OrientationAxesFrame
     << endl;
  os << indent << "SourceNotebook: ";
  if( this->SourceNotebook )
    {
    this->SourceNotebook->PrintSelf(os << endl, indent.GetNextIndent() );
    }
  else
    {
    os << "(none)" << endl;
    }
}
