#!/bin/bash
__doc__="
Script to help building podman in a docker image
https://podman.io/getting-started/installation
"

install_cni_plugins(){
    # Define known SHA256 hashes for CNI plugins by version and arch
    # See: https://github.com/containernetworking/plugins/releases
    declare -A CNI_INSTALL_KNOWN_HASHES=(
        # Format: "${VERSION}-${ARCH}" = "<sha256sum>"
        ["v1.3.0-amd64"]="754a71ed60a4bd08726c3af705a7d55ee3df03122b12e389fdba4bea35d7dd7e"
        ["v1.3.0-arm64"]="bc3e72cf8c3c4ef78742697dc96b3466c92b060f4193a171eaa04cfa97087e4c"
        # Add more if needed
    )

    CNI_VERSION="${CNI_VERSION:-v1.3.0}"
    CNI_OS="linux"
    CNI_ARCH="$(uname -m)"

    # Normalize arch names to match release naming
    case "$CNI_ARCH" in
        x86_64)  CNI_ARCH="amd64" ;;
        aarch64) CNI_ARCH="arm64" ;;
        *) echo "Unsupported arch: $CNI_ARCH" && exit 1 ;;
    esac

    CNI_KEY="${CNI_VERSION}-${CNI_ARCH}"
    EXPECTED_SHA256="${CNI_INSTALL_KNOWN_HASHES[$CNI_KEY]}"

    if [[ -z "$EXPECTED_SHA256" ]]; then
        echo "No known hash for CNI_VERSION '$CNI_VERSION' on arch '$CNI_ARCH'"
        exit 1
    fi

    FILENAME="cni-plugins-${CNI_OS}-${CNI_ARCH}-${CNI_VERSION}.tgz"
    DOWNLOAD_URL="https://github.com/containernetworking/plugins/releases/download/${CNI_VERSION}/${FILENAME}"
    DOWNLOAD_PATH="/bootstrap/${FILENAME}"

    mkdir -p /bootstrap

    echo "Downloading CNI plugins from: $DOWNLOAD_URL"
    curl -LsSf "$DOWNLOAD_URL" -o "$DOWNLOAD_PATH"

    report_bad_checksum() {
        echo "❌ Unexpected checksum:"
        sha256sum "$DOWNLOAD_PATH"
        exit 1
    }

    echo "$EXPECTED_SHA256  $DOWNLOAD_PATH" | sha256sum --check || report_bad_checksum

    echo "✅ Checksum verified: extracting plugins..."
    mkdir -p /opt/cni/bin
    tar -C /opt/cni/bin -xzf "$DOWNLOAD_PATH"

    echo "✅ CNI plugins installed to /opt/cni/bin"

}

setup_configs(){
    # Optional: set CNI path (Podman will look here)
    mkdir -p /etc/containers
cat <<EOF > /etc/containers/containers.conf
[network]
cni_plugin_dirs = ["/opt/cni/bin"]
EOF

cat <<EOF > /etc/containers/registries.conf
[registries.search]
registries = ['docker.io', 'quay.io', 'gitlab.kitware.com']

[registries.insecure]
registries = []

[registries.block]
registries = []
EOF
}

install_go(){
    # Remove old version if needed
    apt-get remove golang-go -y

    # Install Go 1.19+ from official source
    wget https://go.dev/dl/go1.20.13.linux-amd64.tar.gz
    tar -C /usr/local -xzf go1.20.13.linux-amd64.tar.gz

    # Add to path (can go in Dockerfile or build script)
    export PATH=/usr/local/go/bin:$PATH

}

_podman_build_script(){
    set -e
    # Build deps for podman
    #apt-get update
    #DEBIAN_FRONTEND=noninteractive apt-get install btrfs-progs git golang-go go-md2man iptables libassuan-dev libbtrfs-dev libc6-dev libdevmapper-dev libglib2.0-dev libgpgme-dev libgpg-error-dev libprotobuf-dev libprotobuf-c-dev libseccomp-dev libselinux1-dev libsystemd-dev pkg-config runc uidmap gcc g++ build-essential  curl git iptables aufs-tools python3-pip libgl1-mesa-glx tree wget gnupg2 libapparmor-dev -y
    local _PODMAN_TAG="${PODMAN_VERSION:-v3.3}"
    local _CONMON_TAG="${CONMAN_VERSION:-v2.0.24}"
    echo "_CONMON_TAG = $_CONMON_TAG"
    echo "_PODMAN_TAG = $_PODMAN_TAG"

    install_cni_plugins

    setup_configs

    install_go

    CODE_DPATH=$HOME/code

    mkdir -p "$CODE_DPATH"
    mkdir -p /etc/containers

    # CNI Networking
    # TODO: check hashes
    #echo "==========="
    #echo "Grab network config"
    #echo "==========="
    #curl -L -o /etc/containers/registries.conf https://src.fedoraproject.org/rpms/containers-common/raw/main/f/registries.conf
    #curl -L -o /etc/containers/policy.json https://src.fedoraproject.org/rpms/containers-common/raw/main/f/default-policy.json

    # Do all the network stuff first
    # Grab conmon and podman source
    echo "==========="
    echo "Clone source"
    echo "==========="
    echo "_PODMAN_TAG = $_PODMAN_TAG"
    echo "_CONMON_TAG = $_CONMON_TAG"
    git clone -b "$_PODMAN_TAG" --single-branch https://github.com/containers/podman/ "$CODE_DPATH/podman"

    git clone -b "$_CONMON_TAG" --single-branch https://github.com/containers/conmon "$CODE_DPATH/conmon"

    # Build conmon
    echo "==========="
    echo "Build conmon"
    echo "==========="
    cd "$CODE_DPATH/conmon"
    local _GOCACHE="$(mktemp -d)"
    GOCACHE=$_GOCACHE make
    GOCACHE=$_GOCACHE make podman

    echo "==========="
    echo "Build podman"
    echo "==========="
    cd "$CODE_DPATH/podman"
    #make
    # Do install
    #make BUILDTAGS=""
    make BUILDTAGS="seccomp"

    make install

    echo "Cleaning up source directories and bootstrap data..."
    rm -rf "$CODE_DPATH/podman"
    rm -rf "$CODE_DPATH/conmon"
    rm -rf /bootstrap
    rm -rf "$_GOCACHE"

    #make BUILDTAGS="selinux seccomp"
    #make install PREFIX=/usr

    #sh -c "echo 'deb http://download.opensuse.org/repositories/devel:/kubic:/libcontainers:/stable/xUbuntu_${VERSION_ID}/ /' > /etc/apt/sources.list.d/devel:kubic:libcontainers:stable.list
    ## Install podman deps
    #RUN apt-get update \
    #    && DEBIAN_FRONTEND=noninteractive apt install curl git iptables aufs-tools podman python3-pip libgl1-mesa-glx tree wget gnupg2 -y \
    #    && rm -rf /var/lib/apt/lists/*
    ## Install the stable podman
    #RUN apt-get update \
    #    && DEBIAN_FRONTEND=noninteractive apt install podman -y \
    #    && rm -rf /var/lib/apt/lists/*
}

_podman_build_script $@
