#!/bin/bash
__doc__="
Helper script to build and push uv containers for different Python versions
"
cd ~/code/ci-docker

# Relative path within the repo
DOCKERFILE_PATH=uv.dockerfile
IMAGE_NAME=uv
BASE_IMAGE=nvidia/cuda:12.4.1-cudnn-devel-ubuntu22.04
PYTHON_VERSIONS=(
    "3.14"
    "3.13"
    "3.12"
    "3.11"
    "3.10"
    "3.9"
    "3.8"
)

# List of remotes to push images to
REMOTE_NAMES=(
    "gitlab.kitware.com:4567/computer-vision/ci-docker"
    "docker.io/erotemic"
)


make_vcs_ref(){
    local VCS_REF="$(git rev-parse HEAD)"
    # Is the repo dirty? (unstaged, staged, or untracked)
    local is_dirty=false
    git diff --quiet || is_dirty=true
    git diff --cached --quiet || is_dirty=true
    # Untracked files
    if [ -n "$(git ls-files --others --exclude-standard)" ]; then is_dirty=true; fi
    VCS_REF_FULL="$VCS_REF"
    $is_dirty && VCS_REF_FULL="${VCS_REF}-dirty"
    echo "$VCS_REF_FULL"
}

get_dockerfile_arg_default(){
    __doc__="
    Helper to get the default value of an arg out of a dockerfile.
    This lets use explicitly use it in the build command and importantly in the tag name.
    "
    local _ARGNAME=$1
    local _DOCKERFILE_PATH=$2
    grep -E "^ARG $_ARGNAME=" "$_DOCKERFILE_PATH" | head -n1 | cut -d= -f2-
}

get_repo_url(){
    local _RAW_URL="$(git config --get remote.origin.url)"
    local _REPO_URL="$_RAW_URL"
    if [[ "$_RAW_URL" =~ ^git@([^:]+):(.+)\.git$ ]]; then
      _REPO_URL="https://${BASH_REMATCH[1]}/${BASH_REMATCH[2]}"
    elif [[ "$_RAW_URL" =~ ^https?://.*\.git$ ]]; then
      _REPO_URL="${_RAW_URL%.git}"
    fi
    echo "$_REPO_URL"
}
REPO_URL="$(get_repo_url)"



# Use the default UV version in the Dockerfile
UV_VERSION="$(get_dockerfile_arg_default UV_VERSION "$DOCKERFILE_PATH")"
VCS_REF=$(make_vcs_ref)
#CREATED="$(date -u +%Y-%m-%dT%H:%M:%SZ)"   # RFC3339-ish UTC


echo "
IMAGE_NAME = $IMAGE_NAME
BASE_IMAGE = $BASE_IMAGE
DOCKERFILE_PATH = $DOCKERFILE_PATH
UV_VERSION = $UV_VERSION
VCS_REF = $VCS_REF
REPO_URL = $REPO_URL
"

REMOTE_TAGS=()
for PYTHON_VERSION in "${PYTHON_VERSIONS[@]}"; do
    # shellcheck disable=SC2001
    export BASE_TAG=$(echo "$BASE_IMAGE" | sed "s#.*/##; s/://g")
    export IMAGE_TAG=${UV_VERSION}-python${PYTHON_VERSION}-$BASE_TAG
    export IMAGE_QUALNAME=$IMAGE_NAME:$IMAGE_TAG

    echo "=== Building image for Python ${PYTHON_VERSION} ==="
    DOCKER_BUILDKIT=1 docker build --progress=plain \
        -t "$IMAGE_QUALNAME" \
        --build-arg UV_VERSION="$UV_VERSION" \
        --build-arg PYTHON_VERSION="$PYTHON_VERSION" \
        --build-arg BASE_IMAGE=$BASE_IMAGE \
        --build-arg VCS_REF="$VCS_REF" \
        --build-arg REPO_URL="$REPO_URL" \
        --build-arg DOCKERFILE_PATH="$DOCKERFILE_PATH" \
        -f "$DOCKERFILE_PATH" . || break
        #--build-arg CREATED="$CREATED" \

    # Tag local aliases
    ALIASES=(
        "$IMAGE_NAME:latest-python${PYTHON_VERSION}"
        "$IMAGE_NAME:latest"
    )
    for ALIAS in "${ALIASES[@]}"; do
        docker tag "$IMAGE_QUALNAME" "$ALIAS"
    done

    # Tag remote names and aliases
    for REMOTE_NAME in "${REMOTE_NAMES[@]}"; do
        REMOTE_TAGS+=(
            "$REMOTE_NAME/$IMAGE_QUALNAME"
        )
        docker tag "$IMAGE_QUALNAME" "$REMOTE_NAME/$IMAGE_QUALNAME"
        for ALIAS in "${ALIASES[@]}"; do
            REMOTE_TAGS+=(
                "$REMOTE_NAME/$ALIAS"
            )
            docker tag "$IMAGE_QUALNAME" "$REMOTE_NAME/$ALIAS"
        done
    done
done

inspect_notes(){

    docker inspect "$IMAGE_QUALNAME"
    docker image history "$IMAGE_QUALNAME" --no-trunc
    docker image history "$IMAGE_QUALNAME"
}


docker_login_notes(){
    # Note: create a personal access token
    # https://gitlab.kitware.com/-/user_settings/personal_access_tokens
    # setup
    # GITLAB_KITWARE_USERNAME=<gitlab-username>
    # GITLAB_KITWARE_TOKEN=<personal-accses-token-value>
    load_secrets
    echo "$GITLAB_KITWARE_TOKEN" | docker login gitlab.kitware.com:4567 -u "$GITLAB_KITWARE_USERNAME" --password-stdin
}

docker login gitlab.kitware.com:4567

for REMOTE_TAG in "${REMOTE_TAGS[@]}"; do
    echo "REMOTE_TAG = $REMOTE_TAG"
done
for REMOTE_TAG in "${REMOTE_TAGS[@]}"; do
    docker push "$REMOTE_TAG"
done


test_new_images(){
    https://hub.docker.com/layers/nvidia/cuda/12.9.1-cudnn-devel-ubuntu24.04/images/sha256-1cb02388491c05441f8963d42bc38b6ec5462fba95a8e3bb2cb8e6091f0738c8

    # Build the image with version-specific tags
    export IMAGE_NAME=uv
    export UV_VERSION=0.8.4
    export PYTHON_VERSION=3.13

    #export BASE_IMAGE=nvidia/cuda:12.4.1-cudnn-devel-ubuntu22.04
    export BASE_IMAGE=nvidia/cuda:12.9.1-cudnn-devel-ubuntu24.04

    # shellcheck disable=SC2001
    export BASE_TAG=$(echo "$BASE_IMAGE" | sed "s#.*/##; s/://g")
    export IMAGE_TAG=${UV_VERSION}-python${PYTHON_VERSION}-$BASE_TAG
    export IMAGE_QUALNAME=$IMAGE_NAME:$IMAGE_TAG

    echo "IMAGE_QUALNAME=$IMAGE_QUALNAME"

    DOCKER_BUILDKIT=1 docker build --progress=plain \
        -t "$IMAGE_QUALNAME" \
        --build-arg UV_VERSION=$UV_VERSION \
        --build-arg PYTHON_VERSION=$PYTHON_VERSION \
        --build-arg BASE_IMAGE=$BASE_IMAGE \
        -f uv.dockerfile .
}
