#ifndef _cmbNucPinCell_h_
#define _cmbNucPinCell_h_

#include <QObject>
#include <QSet>

#include "cmbNucPart.h"
#include "cmbNucMaterial.h"
#include "vtkBoundingBox.h"

class PinSubPart;
class PinCell;

class PinSubPart
{
public:
  enum End{BOTTOM=0, TOP=1};
  PinSubPart(double z1 = 0, double z2= 10);
  virtual ~PinSubPart();
  virtual QString getTitle(){ return "PinPart"; }

  QPointer<cmbNucMaterial> GetMaterial(int i);
  void SetMaterial(int i, QPointer<cmbNucMaterial> material);
  void SetNumberOfLayers(int numLayers);
  std::size_t GetNumberOfLayers() const;
  QSet< cmbNucMaterial* > getMaterials();
  QSet< cmbNucMaterial* > getOuterMaterials(QPointer<cmbNucMaterial> blMat);

  enum partType{CYLINDER, FRUSTUM};
  virtual PinSubPart::partType GetType() const = 0;

  //sets others values here if they are different.
  //If values have changed, we return true.
  virtual bool fill(PinSubPart const* other);
  virtual PinSubPart * clone() const = 0;

  virtual double getNormalizedThickness(int layer, PinSubPart::End end) = 0;
  virtual void setNormalizedThickness(int layer, PinSubPart::End end, double thick) = 0;
  virtual double getRadius(int layer, PinSubPart::End end) = 0;
  virtual double getRadius(PinSubPart::End end) const = 0;
  virtual void setRadius(PinSubPart::End end, double r) = 0;
  virtual double length() const { return z2 - z1; }
  virtual void reverseRadii() {};

  cmbNucMaterialLayer const& getMaterialLayer(int i) const
  { return *Materials[i]; }
  //Takes ownership
  void setMaterialLayer(int i, cmbNucMaterialLayer * m);

  virtual bool equal(PinSubPart const* other) const;

  virtual std::vector<PinSubPart *> split(std::vector<double>::const_iterator b,
                                          std::vector<double>::const_iterator end) = 0;

  double getZ1() const
  {
    return this->z1;
  }
  double getZ2() const
  {
    return this->z2;
  }

  void setZ1(double v)
  {
    this->z1 = v;
  }
  void setZ2(double v)
  {
    this->z2 = v;
  }

  double x;
  double y;

  void setConnection(cmbNucPartConnection *);
  void clearConnection();

protected:
  void setConnection(cmbNucMaterialLayer * layer);
  double z1;
  double z2;
  std::vector< cmbNucMaterialLayer * > Materials;
  cmbNucPartConnection * connection;
private:
  PinSubPart( const PinSubPart & )
  {}
};

class Cylinder : public PinSubPart
{
public:
  Cylinder(double rin, double z1, double z2);
  Cylinder(PinSubPart const* other);
  virtual ~Cylinder() {}
  virtual PinSubPart::partType GetType() const;
  double getNormalizedThickness(int layer);
  void setNormalizedThickness(int layer, double thick);
  double getRadius(int layer);

  PinSubPart * clone() const;

  virtual double getNormalizedThickness(int layer, PinSubPart::End /*end*/)
  {
    return this->getNormalizedThickness(layer);
  }

  virtual void setNormalizedThickness(int layer, PinSubPart::End /*end*/, double thick)
  {
    this->setNormalizedThickness(layer, thick);
  }

  virtual double getRadius(int layer, PinSubPart::End /*end*/)
  {
    return this->getRadius(layer);
  }

  virtual double getRadius(PinSubPart::End) const
  {
    return r;
  }

  virtual void setRadius(PinSubPart::End, double rin)
  {
    this->r = rin;
  }

  std::vector<PinSubPart *> split(std::vector<double>::const_iterator b,
                                  std::vector<double>::const_iterator end);

  double r;
};

class Frustum : public PinSubPart
{
public:
  Frustum(double const* rin, double z1, double z2);
  Frustum(PinSubPart const* other);
  virtual ~Frustum(){}
  virtual PinSubPart::partType GetType() const;
  double getNormalizedThickness(int layer, PinSubPart::End end);
  void setNormalizedThickness(int layer, PinSubPart::End end, double thick);
  double getRadius(int layer, PinSubPart::End end);
  virtual void reverseRadii()
  {
    double t = r[0];
    r[0] = r[1];
    r[1] = t;
  }


  std::vector<PinSubPart *> split(std::vector<double>::const_iterator b,
                                  std::vector<double>::const_iterator end);

  PinSubPart * clone() const;

  virtual double getRadius(PinSubPart::End end) const
  {
    return r[end];
  }

  virtual void setRadius(PinSubPart::End end, double rin)
  {
    this->r[end] = rin;
  }

  double r[2];
};

// Represents a single pin cell. Pin cells can have multiple
// sections which are either cylinders (constant radius) or
// frustums (with start and end radii) aka truncated cones.
// Pin cells also have multiple layers specified by their thickness
// which can be assigned different material properties.
//
// Pin cells also have names (strings) and labels (usually two character
// strings). In other parts of the code (e.g. cmbNucAssembly) pin cells
// are refered to by their label.

class PinCell : public cmbNucPart
{
public:
  PinCell();

  virtual ~PinCell();

  enumNucPartsType GetType() const;

  void RemoveSection(PinSubPart* obj);

  double Radius(int idx) const;
  virtual double getRadius() const;
  QPointer<cmbNucMaterial> Material(int layer);

  void SetRadius(int idx, double radius);
  void SetMaterial(int idx, QPointer<cmbNucMaterial> material);

  void InsertLayer(int layer);
  void DeleteLayer(int layer);

  int GetNumberOfLayers();

  void SetNumberOfLayers(int numLayers);

  bool fill(PinCell const* other);

  //These take ownership
  void AddPart(PinSubPart * part);

  PinSubPart* GetPart(int i) const;
  size_t GetNumberOfParts() const;

  QSet< cmbNucMaterial* > getMaterials();

  virtual QString getTitle()
  { return "PinCell: " + cmbNucPart::getTitle(); }

  QPointer<cmbNucMaterial> getCellMaterial();
  void setCellMaterial(QPointer<cmbNucMaterial> material);
  bool cellMaterialSet() const;

  vtkBoundingBox computeBounds( bool isHex );

  bool cutaway;

  std::vector<double> getPinLayers() const;

  void splitPin(std::vector<double> const& layers);

  void setHeight(double nh);

  double getZ0() const;

  bool equal(PinCell const& other) const;
  bool sameShape(PinCell const& other) const;
  virtual bool sameShape(cmbNucPart const* other) const;
  virtual bool equal(cmbNucPart const* other) const;

  void removeFakeBoundaryLayer(QString blname);

  cmbNucPart * clone();

protected:
  std::vector<PinSubPart*> Parts;
  cmbNucMaterialLayer CellMaterial;

  void sort();
};

#endif
