//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#ifndef smtk_simulation_windtunnel_qt_qtOpenFoamRunner_h
#define smtk_simulation_windtunnel_qt_qtOpenFoamRunner_h

#include "smtk/simulation/windtunnel/qt/Exports.h"

#include <QObject>
#include <QString>
#include <QStringList>
#include <QTextStream>

class QDialog;
class QDir;
class QFile;
class QPlainTextEdit;
class QWidget;

namespace smtk
{
namespace simulation
{
namespace windtunnel
{

/**\brief Singleton class for launching OpenFOAM apps as external processes. */

class SMTKWINDTUNNELQTEXT_EXPORT qtOpenFoamRunner : public QObject
{
  Q_OBJECT
  using Superclass = QObject;

public:
  static qtOpenFoamRunner* instance(QObject* parent = nullptr);
  void setParentWidget(QWidget* widget);

  enum ProcessState
  {
    NotRunning = 0,
    Starting,
    Running
  };

  void useOpenFoamDocker(bool enabled) { m_useDocker = enabled; }

  /** \brief Launch OpenFOAM application; return boolean indicating success.
     *
     * Runs the specified application as a detached process, setting up log
     * file, polling for completion, and notifying app when process has finished.
     */
  qint64 start(
    const QString& application,
    const QString& projectFolder,
    const QString& caseFolder,
    const QStringList& arguments);

  ProcessState state() const { return m_state; }

  void showLogFile(const QString& path);

Q_SIGNALS:
  void finished(qint64 pid, int exitCode);
  void started(qint64);

public Q_SLOTS:

protected Q_SLOTS:
  void checkStatus();
  void checkLogFile();
  void onLogDialogClosed();
  void onProcessStarted();

protected:
  qtOpenFoamRunner(QObject* parent = nullptr);
  ~qtOpenFoamRunner() override;

  QString createRunScript(
    const QString& application,
    const QStringList& arguments,
    const QString& caseFolder,
    const QString& logPath,
    const QString& exitcodePath) const;

  void createRefineMeshRunScript(
    const QString& caseFolder,
    const QString& logfilePath,
    const QString& exitcodePath,
    QTextStream& qs) const;

  bool createRunInfo(const QString& application, const QString& caseFolder, const QString& logPath)
    const;

  QWidget* m_parentWidget = nullptr;
  QDir* m_processDir = nullptr;
  ProcessState m_state = ProcessState::NotRunning;
  qint64 m_pid = 0;
  bool m_logFilePollingOn = false;

  // Classes for displaying the logfile
  QDialog* m_logDialog = nullptr;
  QFile* m_logFile = nullptr;
  QPlainTextEdit* m_textEdit = nullptr;

  bool m_useDocker = false;
};

} // namespace windtunnel
} // namespace simulation
} // namespace smtk

#endif
