# =============================================================================
#
#  Copyright (c) Kitware, Inc.
#  All rights reserved.
#  See LICENSE.txt for details.
#
#  This software is distributed WITHOUT ANY WARRANTY; without even
#  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the above copyright notice for more information.
#
# =============================================================================

"""SetupPaths operation

Sets two variables on smtk for use by other other operations
    smtk.workflows_folder   file system location of simulation-workflows folder
    smtk.operations_folder  file system location of python operation source files

Applications should run this operation to set workflows_path and operations_path
*before* running the other operations
"""

import os
import sys

import smtk
import smtk.attribute
import smtk.io
import smtk.operation

template_string = """
<?xml version="1.0" encoding="utf-8" ?>
<SMTK_AttributeResource Version="5">
  <Definitions>
    <AttDef Type="setup" BaseType="operation">
      <ItemDefinitions>
        <Directory Name="workflows_folder" NumberOfRequiredValues="1" />
        <Directory Name="operations_folder" NumberOfRequiredValues="1" />
      </ItemDefinitions>
    </AttDef>

    <AttDef Type="result(setup)" BaseType="result" />
  </Definitions>
</SMTK_AttributeResource>
"""


class SetupPaths(smtk.operation.Operation):
    """Adds global smtk.workflows_folder and smtk.operations_folders."""

    def __init__(self):
        smtk.operation.Operation.__init__(self)

    def name(self):
        return "SetupPaths"

    def createSpecification(self):
        spec = self.createBaseSpecification()

        reader = smtk.io.AttributeReader()
        hasErr = reader.readContents(spec, template_string, self.log())
        if hasErr:
            message = 'Error loading SetPaths specification'
            self.log().addError(message)
            raise RuntimeError(message)
        return spec

    def operateInternal(self):
        """"""
        workflows_item = self.parameters().findDirectory('workflows_folder')
        workflows_folder = workflows_item.value()
        if not os.path.exists(workflows_folder):
            self.log().addError('workflows_folder not found: {}'.format(workflows_folder))
            return self.createResult(smtk.operation.Operation.Outcome.FAILED)
        smtk.workflows_folder = workflows_folder
        self.log().addDebug('workflows_folder: {}'.format(smtk.workflows_folder))

        operations_item = self.parameters().findDirectory('operations_folder')
        operations_folder = operations_item.value()
        if not os.path.exists(operations_folder):
            self.log().addError('operations_folder not found: {}'.format(operations_folder))
            return self.createResult(smtk.operation.Operation.Outcome.FAILED)
        smtk.operations_folder = operations_folder
        self.log().addDebug('operations_folder: {}'.format(smtk.operations_folder))

        return self.createResult(smtk.operation.Operation.Outcome.SUCCEEDED)
