# =============================================================================
#
#  Copyright (c) Kitware, Inc.
#  All rights reserved.
#  See LICENSE.txt for details.
#
#  This software is distributed WITHOUT ANY WARRANTY; without even
#  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the above copyright notice for more information.
#
# =============================================================================
"""MergeMeshes operation"""

import os
import shutil
import sys

import smtk
import smtk.attribute
import smtk.io
import smtk.operation
import smtk.project

OP_SUCCEEDED = int(smtk.operation.Operation.Outcome.SUCCEEDED)

# Make sure __file__ is set when using modelbuilder
import inspect
source_file = os.path.abspath(inspect.getfile(inspect.currentframe()))
__file__ = source_file

# Make sure this folder is in sys.path
source_dir = os.path.abspath(os.path.dirname(__file__))
if source_dir not in sys.path:
    sys.path.insert(0, source_dir)

from foam_mixin import FoamMixin
from control_dict_writer import ControlDictWriter
from process_status import ProcessStatus


class MergeMeshes(smtk.operation.Operation, FoamMixin):
    """Configures and optionally runs mergeMeshes and topoSet.

    Uses predefined foam directories:
        background: the background mesh
        overset: the moving object mesh
        analysis: the resulting (merged) mesh
    Uses foam_operation.sbt for template.
    """

    def __init__(self):
        smtk.operation.Operation.__init__(self)
        FoamMixin.__init__(self)
        # Do NOT store any smtk resources as member data (causes memory leak)

    def name(self):
        return "Setup and run mergeMeshes and topoSet"

    def createSpecification(self):
        spec = self._create_specification(app='mergeMeshes')
        return spec

    def operateInternal(self):
        """"""
        project = self._get_project()
        if project is None:
            return self.createResult(smtk.operation.Operation.Outcome.UNABLE_TO_OPERATE)

        att_resource = self._get_attribute_resource(project)
        # print('att_resource:', att_resource)
        if att_resource is None:
            return self.createResult(smtk.operation.Operation.Outcome.UNABLE_TO_OPERATE)

        # Check attributes
        att_names = ['controlDict']
        if not self._check_attributes(att_resource, att_names):
            return self.createResult(smtk.operation.Operation.Outcome.UNABLE_TO_OPERATE)

        solver = self.cd_writer.get_application_name(att_resource)
        if solver != 'overInterDyMFoam':
            message = 'This operation was not designed for {}'.format(solver)
            self.log().addWarning(message)

        project_dir = os.path.abspath(os.path.dirname(project.location()))
        foam_dir = os.path.join(project_dir, 'foam')
        case_dir = os.path.join(foam_dir, 'analysis')

        # Delete current analysis folder
        if os.path.exists(case_dir):
            shutil.rmtree(case_dir)

        # Copy background case to analysis
        from_dir = os.path.join(foam_dir, 'background')
        to_dir = os.path.join(foam_dir, 'analysis')
        shutil.copytree(from_dir, to_dir)

        # Remove background.foam file
        background_foamfile = os.path.join(to_dir, 'background.foam')
        if os.path.exists(background_foamfile):
            os.remove(background_foamfile)

        root_dir = os.path.abspath(os.path.join(source_dir, *[os.pardir] * 4))
        to_solver_dir = 'simulation-workflows/internal/foam/{}'.format(solver)
        solver_dir = os.path.join(root_dir, to_solver_dir)

        # Copy constant folder
        from_dir = os.path.join(solver_dir, 'constant')
        to_dir = os.path.join(case_dir, 'constant')
        shutil.copytree(from_dir, to_dir, dirs_exist_ok=True)

        # Generate controlDict file
        if not self.cd_writer.generate_controlDict(case_dir, att_resource):
            return self.createResult(smtk.operation.Operation.Outcome.FAILED)

        # Check "Run" option
        run_item = self.parameters().findString('run')
        if run_item.isEnabled():
            args = ['./analysis', './overset', '-overwrite']
            log_dir = os.path.join(case_dir, 'logs')
            if not self._run_openfoam(
                    'mergeMeshes',
                    foam_dir,
                    run_item,
                    args_list=args,
                    log_dir=log_dir):
                return self.createResult(smtk.operation.Operation.Outcome.FAILED)

            # Manually add foamfile to case dir
            foamfile_path = os.path.join(case_dir, 'merged.foam')
            with open(foamfile_path, 'a'):
                os.utime(foamfile_path, None)

        result = self._create_result()
        return result
