"""
Class using girder to track simulation resources uploaded to remote machines.
SALI = Simulation Asset Location Index

This first implementation creates a folder "SALI" under the user's Private folder,
and creates items to represent assets on the remote machine. The item name is set
to the local filename, and item meta data is used to store md5 sums and locations
for the local and remote files. An example metadata is:

{
    "cori: {
        "md5": "acada04583414ca8e5e646899e70e06d
        "path": "/scratch2/scratchdirs/johnt/cw18/pillbox4.ncdf"
    },
    "sources": [
        {
            "hostname": "turtleland4",
            "md5": "c3fe41261f6276ed81b6652103d02e9b",
            "path": "/home/john/projects/slac/git/smtk/data/model/3d/genesis/pillbox4.gen"
        }
    ]
}

"""

import argparse
import hashlib
import json
import os
import socket

import requests

import girder_client
from girder_client import GirderClient, HttpError

NEWT_URL = 'https://newt.nersc.gov/newt'


class SimulationAssetLocationIndex():
    """Utility for tracking simulation resources uploaded to remote machines at NERSC.

    Stores info in girder instance.
    """
    def __init__(self, girder_client, newt_sessionid):
        """"""
        self._girder_client = girder_client
        self._sali_folder_id = None
        self._newt_requests = requests.Session()
        self._newt_requests.cookies.update(dict(newt_sessionid=newt_sessionid))

        # Get user's private folder
        user = self._girder_client.get('user/me')
        user_id = user['_id']
        gen = self._girder_client.listFolder(user_id, 'user', name='Private')
        private_folder = self._next_item(gen)
        if private_folder is None:
            raise RuntimeError('Failed to find Private folder for this user')
        private_folder_id = private_folder['_id']
        # print('private_folder_id', private_folder_id)

        # Get SALI folder (create if needed)
        gen = self._girder_client.listFolder(private_folder_id, name='SALI')
        sali_folder = self._next_item(gen)
        if sali_folder is None:
            # Create folder now
            sali_folder = self._girder_client.createFolder(private_folder_id, 'SALI',
                description='Simulation Asset Location Index - first prototype')
            print('Created SALI folder, id {}'.format(sali_folder['_id']))
        self._sali_folder_id = sali_folder['_id']

    def query(self, local_location, remote_machine='cori', verify=True, return_all_metadata=False):
        """Checks for item in the SALI folder with the given model name.

        Return value is one of the following:
            * None if the asset is not found in the index.
            * None if the asset is found, and the "verify" argument is True, and either the local
              or remote asset fails verification.
            * The remote location of the asset if it is found and the "verify" argument is False.
            * The remote location of that asset if it is found, and the "verify" argument is True,
              and both local and remote assets pass verification.
            * The internal SALI meta data for the assets if the "return_all_metadata" argument is set
              and either the "verify" argument is False or the "verify" argument is True and both
              local and remote assets pass verfication. The "return_all_metadata" argument is for
              test and debug.
        """
        filename = os.path.basename(local_location)
        # print('query name:', filename)
        gen = self._girder_client.listItem(self._sali_folder_id, name=filename)
        item = self._next_item(gen)
        if item is None:
            return None

        if verify:
            if not self._verify_local_asset(item, local_location):
                return None

            if not self._verify_remote_asset(item):
                return None

        if return_all_metadata:
            return item.get('meta')
        # (else)
        remote_location = item.get('meta').get(remote_machine, {}).get('path')
        return remote_location

    def put(self, local_location, remote_location, remote_machine='cori', can_replace=False):
        """Adds item to SALI folder.

        Returns boolean indicating success.
        """
        # Make sure that local file exists
        if not os.path.exists(local_location):
            print('Local file not found at {}'.format(local_location))
            return False
        filename = os.path.basename(local_location)

        # Check if item for this filename is already in girder
        gen = self._girder_client.listItem(self._sali_folder_id, name=filename)
        item = self._next_item(gen)
        if item is not None and not can_replace:
            print('Item already exits for {}. Use can_replace flag to overwrite'.format(filename))
            return False

        # Get local md5 and hostname
        local_md5 = self._get_local_md5(local_location)
        if local_md5 is None:
            return False
        hostname = socket.gethostname()
        print('local hostname:', hostname)

        # Check for remote file
        url = '{}/command/{}'.format(NEWT_URL, remote_machine)
        data = {
            'executable': '/usr/bin/ls {}'.format(remote_location),
            'loginenv': 'true'
        }
        r = self._newt_requests.post(url, data=data)
        print('ls command returned', r.json())
        ls_result = r.json()
        if ls_result['error']:
            print(ls_result['error'])
            return False

        # Get md5 for remote file
        remote_md5 = self._get_remote_md5(remote_location, remote_machine)
        if remote_md5 is None:
            return False

        # Build metadata object
        source = dict(hostname=hostname, path=local_location, md5=local_md5)
        remote = dict(path=remote_location, md5=remote_md5)
        metadata = dict(sources=[source])
        metadata[remote_machine] = remote

        item = self._girder_client.createItem(
            self._sali_folder_id, filename, reuseExisting=True, metadata=metadata)
        print('createItem returned', item)
        return True

    def _get_local_md5(self, local_location):
        """"""
        local_md5 = None
        with open(local_location, 'rb') as fp:
            local_md5 = hashlib.md5(fp.read()).hexdigest()
        if local_md5 is None:
            print('Error getting model file md5')
        # print('local md5:', local_md5)
        return local_md5

    def _get_remote_md5(self, remote_location, remote_machine='cori'):
        """Runs md5sum command on remote machine"""
        url = '{}/command/{}'.format(NEWT_URL, remote_machine)
        data = {
            'executable': '/usr/bin/md5sum {}'.format(remote_location),
            'loginenv': 'true'
        }
        r = self._newt_requests.post(url, data=data)
        print('md5sum command returned', r.json())
        md5_result = r.json()
        if md5_result['error']:
            print(md5_result['error'])
            return None

        # Valid output is of the form "<md5sum> <path>"
        output = md5_result['output']
        remote_md5 = output.split(' ')[0]
        return remote_md5

    def _next_item(self, gen):
        """Returns next item from generator, or None if empty"""
        try:
            entry = next(gen)
        except StopIteration:
            return None
        return entry

    def _verify_local_asset(self, sali_item, local_location):
        """Checks that the file at local_location matches the source information in the SALI item"""
        sources_meta = sali_item.get('meta', {}).get('sources')
        if sources_meta is None:
            print('Internal error: Invalid SALI item {} - no meta.sources list'.format(sali_item[_id]))
            return False

        expected_md5 = sources_meta[0].get('md5')
        if expected_md5 is None:
            print('Internal error: Missing sources[0].md5 in SALI item {}'.format(sali_item[_id]))
            return False

        local_md5 = self._get_local_md5(local_location)
        if local_md5 is None:
            return False

        # Verify if the 2 md5's match, of course
        return local_md5 == expected_md5

    def _verify_remote_asset(self, sali_item):
        """Checks that the file at local_location matches the source information in the SALI item"""
        sources_meta = sali_item.get('meta', {}).get('sources')
        if sources_meta is None:
            print('Internal error: Invalid SALI item {} - no meta.sources list'.format(sali_item[_id]))
            return False

        expected_md5 = sources_meta[0].get('md5')
        if expected_md5 is None:
            print('Internal error: Missing sources[0].md5 in SALI item {}'.format(sali_item[_id]))
            return False

        remote_md5 = this._get_remote_md5(remote_location)
        if remote_md5 is None:
            return False

        return remote_md5 == expected_md5
