//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#include "pqCumulusJobsPanel.h"

#include "smtk/cumulus/jobspanel/cumuluswidget.h"
#include "smtk/cumulus/jobspanel/job.h"

#include "pqApplicationCore.h"

#include <QAction>
#include <QDebug>
#include <QFormLayout>
#include <QLineEdit>
#include <QMessageBox>
#include <QPushButton>
#include <QSslSocket>
#include <QString>
#include <QVBoxLayout>
#include <QWidget>
#include <QtGlobal>

class pqCumulusJobsPanel::pqCumulusJobsPanelInternal
{
public:
  pqCumulusJobsPanelInternal();

  QWidget* MainWidget;
  QVBoxLayout* MainLayout;

  QWidget* FirstWidget;
  QLineEdit* CumulusUrlEdit;
  QLineEdit* SessionIdEdit;
  QPushButton* AuthenticateButton;

  cumulus::CumulusWidget* CumulusWidget;

  // Used to provide NEWT session info to other plugins
  QObject* JobsPanelObject;
};

pqCumulusJobsPanel::pqCumulusJobsPanelInternal::pqCumulusJobsPanelInternal()
  : MainWidget(nullptr)
  , MainLayout(nullptr)
  , FirstWidget(nullptr)
  , CumulusUrlEdit(nullptr)
  , CumulusWidget(nullptr)
  , JobsPanelObject(nullptr)
{
}

pqCumulusJobsPanel::pqCumulusJobsPanel(QWidget* parent)
  : QDockWidget(parent)
{
  this->setObjectName("Jobs Panel");
  this->setWindowTitle("Cumulus Jobs");
  this->Internal = new pqCumulusJobsPanelInternal;

  // Initialize main widget
  this->Internal->MainWidget = new QWidget(parent);
  this->Internal->MainLayout = new QVBoxLayout;

  // Instantiate first page (displayed at startup)
  this->Internal->FirstWidget = new QWidget(this->Internal->MainWidget);
  QFormLayout* firstLayout = new QFormLayout;

  // Cumulus URL
  this->Internal->CumulusUrlEdit = new QLineEdit(this->Internal->FirstWidget);
  // this->Internal->CumulusUrlEdit->setText("http://localhost:8080/api/v1");
  QString spinUrl("https://ace3p.kitware.com/api/v1");
  this->Internal->CumulusUrlEdit->setText(spinUrl);
  firstLayout->addRow("Cumulus URL", this->Internal->CumulusUrlEdit);

  // Login button and NEWT session id
  this->Internal->AuthenticateButton = new QPushButton("NERSC Login", this->Internal->FirstWidget);
  this->Internal->SessionIdEdit = new QLineEdit(this->Internal->FirstWidget);
  this->Internal->SessionIdEdit->setPlaceholderText("NERSC/NEWT Session Id");
  this->Internal->SessionIdEdit->setReadOnly(true);
  firstLayout->addRow(this->Internal->AuthenticateButton, this->Internal->SessionIdEdit);

  this->Internal->FirstWidget->setLayout(firstLayout);
  this->Internal->MainLayout->addWidget(this->Internal->FirstWidget);

  // Instantiate cumulus-monitoring objects
  this->Internal->CumulusWidget = new cumulus::CumulusWidget(this);
  this->Internal->MainLayout->addWidget(this->Internal->CumulusWidget);
  connect(this->Internal->CumulusWidget, &cumulus::CumulusWidget::info, this,
    &pqCumulusJobsPanel::infoSlot);
  connect(this->Internal->CumulusWidget, &cumulus::CumulusWidget::newtSessionId, this,
    &pqCumulusJobsPanel::receivedNewtSessionId);

  // Add context menu item to load results
  QAction* action = new QAction("Load Simulation Results", this);
  this->Internal->CumulusWidget->addContextMenuAction("downloaded", action);
  QObject::connect(
    action, &QAction::triggered, this, &pqCumulusJobsPanel::requestSimulationResults);

  QObject::connect(this->Internal->AuthenticateButton, &QPushButton::clicked, this,
    &pqCumulusJobsPanel::authenticateHPC);
  connect(this->Internal->CumulusWidget, SIGNAL(resultDownloaded(const QString&)), this,
    SIGNAL(resultDownloaded(const QString&)));

  // Finish main widget
  this->Internal->MainWidget->setLayout(this->Internal->MainLayout);
  this->setWidget(this->Internal->MainWidget);

  // Register JobsPanelInfo as a paraview manager, for use by other plugins
  auto pqCore = pqApplicationCore::instance();
  if (!pqCore)
  {
    qWarning() << "pqCumulusJobsPanel missing pqApplicationCore";
    return;
  }
  this->Internal->JobsPanelObject = new QObject;
  pqCore->registerManager(QString("jobs_panel"), this->Internal->JobsPanelObject);
}

pqCumulusJobsPanel::~pqCumulusJobsPanel()
{
  delete this->Internal;
}

void pqCumulusJobsPanel::infoSlot(const QString& msg)
{
  // Would like to emit signal to main window status bar
  // but that is not currently working. Instead, qInfo()
  // messages are sent to the CMB Log Window:
  qInfo() << "JobsPanel:" << msg;
}

void pqCumulusJobsPanel::authenticateHPC()
{
  // Check for SSL
  if (!QSslSocket::supportsSsl())
  {
    QMessageBox::critical(NULL, QObject::tr("SSL support"),
      QObject::tr("SSL support is required, you must rebuild Qt with SSL support."));
    return;
  }

  // Get cumulus/girder url
  QString cumulusUrl = this->Internal->CumulusUrlEdit->text();
  this->Internal->CumulusWidget->girderUrl(cumulusUrl);

  // Check for cumulus server
  if (!this->Internal->CumulusWidget->isGirderRunning())
  {
    QString msg = QString("Cumulus server NOT FOUND at %1").arg(cumulusUrl);
    QMessageBox::critical(NULL, QObject::tr("Cumulus Server Not Found"), msg);
    return;
  }

  // Open NERSC login dialog
  this->Internal->CumulusWidget->showLoginDialog();
}

void pqCumulusJobsPanel::requestSimulationResults()
{
  QAction* action = qobject_cast<QAction*>(sender());
  if (!action)
  {
    QMessageBox::critical(this, "Error", "Not connected to QAction?");
    return;
  }

  cumulus::Job job = action->data().value<cumulus::Job>();
  // QString message;
  // QTextStream qs(&message);
  // qs << "The slot named test() was called for job " << job.id() << ".\n\n"
  //    << "Download folder: " << job.downloadFolder();
  // QMessageBox::information(this, "Test", message);

  emit this->loadSimulationResults(job.downloadFolder());
}

void pqCumulusJobsPanel::receivedNewtSessionId(const QString& sessionId, int lifetime)
{
  this->Internal->JobsPanelObject->setProperty("newt_sessionid", sessionId);
  // Todo add expiration datetime based on lifetime seconds

  this->Internal->SessionIdEdit->setText(sessionId);
  bool enable = sessionId.isEmpty();
  this->Internal->AuthenticateButton->setEnabled(enable);
}
