"""
Script to query girder instance for items corresponding to simulation assets.
SALI = Simulation Asset Location Index

"""

import argparse
import hashlib
import json
import os
import socket

import requests

import girder_client
from girder_client import GirderClient, HttpError

class SALIClient():
    """"""
    def __init__(self, girder_url, apikey):
        """"""
        self._girder_client = None
        self._girder_url = girder_url
        self._sali_folder_id = None
        self._session = requests.Session()

        # Initialize girder client
        url = '%s/api/v1' % self._girder_url
        self._girder_client = GirderClient(apiUrl=url)
        self._girder_client.authenticate(apiKey=apikey)

        # Get user's private folder
        user = self._girder_client.get('user/me')
        #print('user', user)
        user_id = user['_id']
        gen = self._girder_client.listFolder(user_id, 'user', name='Private')
        private_folder = self._next_item(gen)
        if private_folder is None:
            raise RuntimeError('Failed to find Private folder for this user')
        private_folder_id = private_folder['_id']
        # print('private_folder_id', private_folder_id)

        # Get SALI folder (create if needed)
        gen = self._girder_client.listFolder(private_folder_id, name='SALI')
        sali_folder = self._next_item(gen)
        if sali_folder is None:
            # Create folder now
            sali_folder = self._girder_client.createFolder(private_folder_id, 'SALI',
                description='Simulation Asset Location Index - first prototype')
            print('Created SALI folder, id {}'.format(self._sali_folder['_id']))
        self._sali_folder_id = sali_folder['_id']

    def query(self, local_location, remote_machine='cori', return_all_metadata=False):
        """Checks for item in the SALI folder with the given model name."""
        filename = os.path.basename(local_location)
        # print('query name:', filename)
        gen = self._girder_client.listItem(self._sali_folder_id, name=filename)
        item = self._next_item(gen)
        if return_all_metadata:
            return item.get('meta')
        # (else)
        remote_location = item.get('meta').get(remote_machine, {}).get('path')
        return remote_location

    def put(self, local_location, remote_machine, remote_location, can_replace=False):
        """Adds item to SALI folder."""
        # Make sure that input file exists
        if not os.path.exists(local_location):
            raise RuntimeError('Local file not found at {}'.format(local_location))
        filename = os.path.basename(local_location)

        # TODO Check for current item

        # Get md5 and hostname
        md5 = None
        with open(local_location, 'rb') as fp:
            md5 = hashlib.md5(fp.read()).hexdigest()
        if md5 is None:
            raise RuntimeError('Error getting model file md5')
        print('md5:', md5)

        hostname = socket.gethostname()
        print('hostname:', hostname)

        source = dict(hostname=hostname, path=local_location, md5=md5)
        remote = dict(path=remote_location)
        metadata = dict(sources=[source])
        metadata[remote_machine] = remote

        item = self._girder_client.createItem(
            self._sali_folder_id, filename, reuseExisting=True, metadata=metadata)
        print('createItem returned', item)

    def _next_item(self, gen):
        """Returns next item from generator, or None if empty"""
        try:
            entry = next(gen)
        except StopIteration:
            return None
        return entry
