//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#include "smtk/simulation/ace3p/operations/Export.h"

#include "smtk/simulation/ace3p/Metadata.h"
#include "smtk/simulation/ace3p/Project.h"

// SMTK includes
#include "smtk/attribute/Attribute.h"
#include "smtk/attribute/FileItem.h"
#include "smtk/attribute/IntItem.h"
#include "smtk/attribute/Item.h"
#include "smtk/attribute/ReferenceItem.h"
#include "smtk/attribute/DirectoryItem.h"
#include "smtk/attribute/Resource.h"
#include "smtk/attribute/ResourceItem.h"
#include "smtk/attribute/SearchStyle.h"
#include "smtk/attribute/StringItem.h"
#include "smtk/io/Logger.h"
#include "smtk/model/Resource.h"
#include "smtk/operation/Manager.h"
#include "smtk/operation/operators/ImportPythonOperation.h"
#include "smtk/project/Project.h"

// Build dir includes
#include "smtk/simulation/ace3p/Export_xml.h"

#include <boost/filesystem.hpp>

#include <iostream>
#include <string>

namespace
{
const int OP_SUCCEEDED = static_cast<int>(smtk::operation::Operation::Outcome::SUCCEEDED);
}

namespace smtk
{
namespace simulation
{
namespace ace3p
{

smtk::operation::Operation::Result Export::operateInternal()
{
  auto& logger = this->log();

  // Make sure the python operation (ACE3P.py) exists.
  if (Metadata::WORKFLOWS_DIRECTORY.empty())
  {
    smtkErrorMacro(logger, "Internal Error: Metadata::WORKFLOWS_DIRECTORY not defined.");
    return this->createResult(smtk::operation::Operation::Outcome::FAILED);
  }

  boost::filesystem::path workflowDir(Metadata::WORKFLOWS_DIRECTORY);
  boost::filesystem::path pyPath = workflowDir / "ACE3P.py";
  if (!boost::filesystem::exists(pyPath))
  {
    smtkErrorMacro(logger, "Error: expecing python file at " << pyPath.string());
    return this->createResult(smtk::operation::Operation::Outcome::FAILED);
  }

  // Import the python op
  smtk::operation::ImportPythonOperation::Ptr importPythonOp =
    this->manager()->create<smtk::operation::ImportPythonOperation>();
  if (!importPythonOp)
  {
    smtkErrorMacro(logger, "Unable to create ImportPythonOperation");
    return this->createResult(smtk::operation::Operation::Outcome::FAILED);
  }
  importPythonOp->parameters()->findFile("filename")->setValue(pyPath.string());
  auto loadResult = importPythonOp->operate();
  if (loadResult->findInt("outcome")->value() != OP_SUCCEEDED)
  {
    smtkErrorMacro(logger, "Failed to load export script " << pyPath.string());
    // Copy the outcome
    int loadOutcome = loadResult->findInt("outcome")->value();
    auto thisOutcome = static_cast<smtk::operation::Operation::Outcome>(loadOutcome);
    return this->createResult(thisOutcome);
  }

  // Access the unique name assigned to the operation
  std::string operationName = loadResult->findString("unique_name")->value();

  // Instantiate the operation
  smtk::operation::Operation::Ptr pythonOp = this->manager()->create(operationName);
  if (pythonOp == nullptr)
  {
    smtkErrorMacro(logger, "Failed to instantate operation from export script ");
    return this->createResult(smtk::operation::Operation::Outcome::FAILED);
  }

  // Configure the operation
  auto refItem = this->parameters()->associations();
  auto project = refItem->valueAs<smtk::simulation::ace3p::Project>();

  std::cout << "\npythonOp:\n";
  for(int i = 0; i < pythonOp->parameters()->numberOfItems(); ++i) {
    std::cout << "item "<< i << ": " << pythonOp->parameters()->item(i)->name() << "\n";
  }

  std::vector<std::string> itemNames = {"OutputFolder", "OutputFilePrefix", "MeshFile"};
  for(auto item : itemNames)
  {
    smtk::attribute::ConstItemPtr sourceItem =
      this->parameters()->find(item, smtk::attribute::SearchStyle::RECURSIVE_ACTIVE);
    auto targetItem =
      pythonOp->parameters()->find(item, smtk::attribute::SearchStyle::RECURSIVE_ACTIVE);
    if(sourceItem != nullptr && targetItem != nullptr)
    {
      targetItem->assign(sourceItem);
    }
  }

  smtk::model::ResourcePtr modelResource =
    project->resources().getByRole<smtk::model::Resource>("analysis-mesh");
  assert(modelResource != nullptr);
  auto resItem = pythonOp->parameters()->findResource("model");
  assert(resItem != nullptr);
  assert(resItem->setValue(modelResource));

  smtk::resource::ResourcePtr attResource = this->parameters()->findResource("analysis")->value();
  assert(attResource != nullptr);
  assert(pythonOp->parameters()->findResource("attributes")->setValue(attResource));

  // Get mesh location from the project metadata
  Metadata mdata;
  mdata.getFromResource(project);

  // Analysis Transfer Mesh
  bool anMeshEnabled = !mdata.NativeAnalysisMeshLocation.empty();
  auto anMeshItem = pythonOp->parameters()->findFile("MeshFile");
  assert(anMeshItem != nullptr);
  anMeshItem->setIsEnabled(anMeshEnabled);
  if (anMeshEnabled)
  {
    anMeshItem->setValue(mdata.NativeAnalysisMeshLocation);
  }

  assert(pythonOp->ableToOperate());
  smtk::attribute::AttributePtr exportResult = pythonOp->operate();
  if (exportResult == nullptr)
  {
    std::cout << "WARNING: exportResult is null" << std::endl;
    return this->createResult(smtk::operation::Operation::Outcome::FAILED);
  }

  // Copy the outcome
  int exportOutcome = exportResult->findInt("outcome")->value();
  auto thisOutcome = static_cast<smtk::operation::Operation::Outcome>(exportOutcome);

  this->manager()->unregisterOperation(operationName);
  
  return this->createResult(thisOutcome);
}

smtk::operation::Operation::Specification Export::createSpecification()
{
  Specification spec = this->smtk::operation::XMLOperation::createSpecification();
  //std::cout << "Export Op:\n" << this->xmlDescription() << std::endl;

  return spec;
}

const char* Export::xmlDescription() const
{
  return Export_xml;
}
}
}
}
