#=============================================================================
#
#  Copyright (c) Kitware, Inc.
#  All rights reserved.
#  See LICENSE.txt for details.
#
#  This software is distributed WITHOUT ANY WARRANTY; without even
#  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the above copyright notice for more information.
#
#=============================================================================

import configparser
import datetime
import os
import warnings
print('loading', os.path.basename(__file__))

import smtk
import smtk.attribute
import smtk.io
# import smtk.model

# ---------------------------------------------------------------------
class AWSWriter:
    """"""
    def __init__(self, scope):
        """"""
        # Todo Check OS? (only support linux?)

        # Check for genesis/exodus model files, which aren't supported
        for path in scope.files_to_upload:
            basename, ext = os.path.splitext(path)
            if ext in ['.gen', '.exo', '.ex2']:
                msg = 'AWS does not Genesis/Exodus models: {}'.format(ext)
                print(msg)
                RuntimeError(msg)

        self.aws_item = None
        self.scope = scope

    def export_ace3p(self, aws_item):
        self.aws_item = aws_item

        # Create job folder
        project_folder_item = self.scope.export_att.findDirectory('ProjectFolder')
        if project_folder_item is None or \
                not project_folder_item.isEnabled() or \
                not project_folder_item.isSet():
            msg = 'AWSWriter Error: ProjectFolder is not specified'
            print(msg)
            raise RuntimeError(msg)

        dt = datetime.datetime.now()
        folder_name = dt.strftime('aws-%y%m%d-%H%M%S')

        # Generate cluster configuration, starting with specified default file
        config_file_item = self.scope.export_att.itemAtPath('AWSSimulation/AWSConfigFile')
        if config_file_item is None or not config_file_item.isSet():
            msg = 'AWSWriter ERROR: AWS config file not specified'
            print(msg)
            raise RuntimeError(msg)
        config_file_path = config_file_item.value()
        if not os.path.exists(config_file_path):
            msg = 'AWSWriter ERROR: AWS config file not found: {}'.format(config_file_path)
            print(msg)
            raise RuntimeError(msg)

        # Get the cluster section to use
        cluster_key_item = self.scope.export_att.itemAtPath('AWSSimulation/ClusterSection')
        cluster_key = cluster_key_item.value()

        # Load config file and set instance type
        config = configparser.ConfigParser()
        config.read_file(open(config_file_path))
        instance_type_item = self.scope.export_att.itemAtPath('AWSSimulation/InstanceType')
        # There must be one active child
        if instance_type_item.numberOfActiveChildrenItems() != 1:
            n = instance_type_item.numberOfActiveChildrenItems()
            msg = 'AWSWriter ERROR: instance type should have 1 active child item, not {}'.format(n)
            print(msg)
            raise RuntimeError(msg)
        instance_item = instance_type_item.activeChildItem(0)
        config[cluster_key]['compute_instance_type'] = instance_item.value()

        # Todo change master instance type to a compute node?
        # config[cluster_key]['master_instance_type'] = instance_item.value()

        # Todo Generate arguments for aws submit script, using:
        #   - NumberOfNodes item
        #   - Timeout (min) item

        # Create job folder
        folder_path = os.path.join(project_folder_item.value(), 'jobs', folder_name)
        os.makedirs(folder_path)

        # Write list of files to upload
        upload_info_path = os.path.join(folder_path, 'files_to_upload.txt')
        with open(upload_info_path, 'w') as upload_info_file:
            for path in self.scope.files_to_upload:
                upload_info_path.write(path)
                upload_info_path.write('\n')

        # Todo copy the ace3p input file (?)

        # Write the config file
        config_path = os.path.join(folder_path, 'parallelcluster.config')
        with open(config_path, 'w') as configfile:
            config.write(configfile)

        # Write export attributes to the job folder
        att_writer = smtk.io.AttributeWriter()
        logger = smtk.io.Logger()
        export_atts_filename = 'export.sbi'
        export_atts_path = os.path.join(folder_path, export_atts_filename)
        err = att_writer.write(self.scope.export_att.attributeResource(), export_atts_path, logger)
        if err:
            msg = 'AWSWriter Error: Unable to write export attribute file {}'.format(export_atts_path)
            print(msg)
            raise RuntimeError(msg)

        # Create stamp subdirectory and touch "export" file
        stamp_path = os.path.join(folder_path, 'stamp')
        os.makedirs(stamp_path)
        export_path = os.path.join(stamp_path, 'export')
        with open(export_path, 'w') as st:
            pass

        print('Created job folder {}'.format(folder_path))

        return True
