//=========================================================================
//  Copyright (c) Kitware, Inc.
//  All rights reserved.
//  See LICENSE.txt for details.
//
//  This software is distributed WITHOUT ANY WARRANTY; without even
//  the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
//  PURPOSE.  See the above copyright notice for more information.
//=========================================================================
#include "plugins/postprocessing-mode/pqCMBPostProcessingModeBehavior.h"

// Client side
#include "pqApplicationCore.h"
#include "pqCoreUtilities.h"
#include "pqInterfaceTracker.h"
#include "pqOutputPort.h"
#include "pqPVApplicationCore.h"
#include "pqParaViewMenuBuilders.h"

// Qt
#include <QAction>
#include <QDir>
#include <QMainWindow>
#include <QMenu>
#include <QMenuBar>
#include <QMetaObject>
#include <QToolBar>
#include <QWidget>

// Qt generated UI
#include "ui_pqCMBPostProcessingModeBehavior.h"

#include <iostream>
#include <vector>

static pqCMBPostProcessingModeBehavior* s_postMode = nullptr;

class pqCMBPostProcessingModeBehavior::pqInternal
{
public:
  Ui::pqCMBPostProcessingModeBehavior Actions;
  QAction* ModeAction;
  QWidget ActionsOwner;
  QMenu* MenuSources;
  QMenu* MenuFilters;
  std::vector<QToolBar*> PostProcessingToolBars;
  std::vector<QAction*> PostProcessingActions;
};

pqCMBPostProcessingModeBehavior::pqCMBPostProcessingModeBehavior(QObject* parent)
  : Superclass(parent)
  , m_menusBuilt(false)
{
  m_p = new pqInternal;
  m_p->Actions.setupUi(&m_p->ActionsOwner);

  m_p->ModeAction = m_p->Actions.actionPostProcessingMode;

  if (!s_postMode)
  {
    s_postMode = this;
  }

  // By default, the button is off. Register them in this state.
  this->addAction(m_p->ModeAction);
  this->setExclusive(false);

  QObject::connect(this, SIGNAL(triggered(QAction*)), this, SLOT(switchModes(QAction*)));
}

pqCMBPostProcessingModeBehavior::~pqCMBPostProcessingModeBehavior()
{
  if (s_postMode == this)
  {
    s_postMode = nullptr;
  }
}

pqCMBPostProcessingModeBehavior* pqCMBPostProcessingModeBehavior::instance()
{
  return s_postMode;
}

void pqCMBPostProcessingModeBehavior::switchModes(QAction* a)
{
  this->buildMenus();

  for (auto act : m_p->PostProcessingActions)
  {
    if (act)
    {
      act->setVisible(a->isChecked());
    }
  }
  if (a->isChecked())
  {
    m_p->MenuSources->setEnabled(true);
    m_p->MenuFilters->setEnabled(true);
    m_p->MenuSources->menuAction()->setVisible(true);
    m_p->MenuFilters->menuAction()->setVisible(true);
    m_p->Actions.pipelineBrowserDock->show();
    for (auto tbar : m_p->PostProcessingToolBars)
    {
      if (tbar)
      {
        tbar->show();
      }
    }
  }
  else
  {
    m_p->MenuSources->menuAction()->setVisible(false);
    m_p->MenuFilters->menuAction()->setVisible(false);
    m_p->MenuSources->setEnabled(false);
    m_p->MenuFilters->setEnabled(false);
    m_p->Actions.pipelineBrowserDock->hide();
    for (auto tbar : m_p->PostProcessingToolBars)
    {
      if (tbar)
      {
        tbar->hide();
      }
    }
    // TODO: We could do something fancy here like:
    // for each view v
    //   for each representation r
    //     if !pipline-source(r).isA('pqSMTKResource') then r.setVisibility(false)
    // to disable all post-processing filter representations.
  }
}

static QAction* findMenuAction(QMenuBar* bar, const QString& name)
{
  foreach (QAction* act, bar->actions())
  {
    if (act->text() == name)
    {
      return act;
    }
  }
  return nullptr;
}

static QToolBar* findToolBar(QMainWindow* mw, const QString& name)
{
  foreach (QToolBar* bar, mw->findChildren<QToolBar*>())
  {
    if (bar->windowTitle() == name)
    {
      return bar;
    }
  }
  return nullptr;
}

static QAction* findActionByName(QMainWindow* mw, const QString& name)
{
  if (mw)
  {
    foreach (QAction* act, mw->findChildren<QAction*>())
    {
      if (act->objectName() == name)
      {
        return act;
      }
    }
  }
  return nullptr;
}

static QAction* findAction(QToolBar* toolbar, const QString& name)
{
  if (toolbar)
  {
    foreach (QAction* bar, toolbar->findChildren<QAction*>())
    {
      if (bar->text() == name)
      {
        return bar;
      }
    }
  }
  return nullptr;
}

void pqCMBPostProcessingModeBehavior::buildMenus()
{
  if (m_menusBuilt)
  {
    return;
  }

  if (QMainWindow* mw = dynamic_cast<QMainWindow*>(pqCoreUtilities::mainWidget()))
  {
    m_menusBuilt = true;

    // Build the source and filter menus:
    m_p->MenuSources = mw->menuBar()->addMenu("&Sources");
    pqParaViewMenuBuilders::buildSourcesMenu(*m_p->MenuSources, mw);

    m_p->MenuFilters = mw->menuBar()->addMenu("&Filters");
    pqParaViewMenuBuilders::buildFiltersMenu(*m_p->MenuFilters, mw);

    QAction* toolsMenu = findMenuAction(mw->menuBar(), "&Tools");
    mw->menuBar()->insertMenu(toolsMenu, m_p->MenuSources);
    mw->menuBar()->insertMenu(toolsMenu, m_p->MenuFilters);

    // Move the pipeline browser widget from our UI file to the
    // main window and dock it on the left, like ParaView:
    m_p->Actions.pipelineBrowserDock->setParent(mw);
    mw->addDockWidget(Qt::LeftDockWidgetArea, m_p->Actions.pipelineBrowserDock);
    QDockWidget* propDock = mw->findChild<QDockWidget*>("proxyTabDock");
    if (propDock)
    {
      // Be tricky here: we just added the pipeline browser panel, so it is not
      // located anywhere we have control over. But the user has placed the propDock
      // panel where he wants it and that position is saved across restarts (unlike
      // the pipeline browser). So, first, split the propDock widget (which moves
      // the pipeline browser into a good place next to the propDoc, but leaves it
      // below the propDock, which is not nice). However, now we can split the
      // pipeline browser leaving the propDock beneath it. And if the user has tabbed
      // the propDock then the pipeline browser should now be tabbed with it.
      //
      // We only do this once at creation time. The rest of the time, the panel will
      // remember its location if the user moves it.
      mw->splitDockWidget(propDock, m_p->Actions.pipelineBrowserDock, Qt::Vertical);
      mw->splitDockWidget(m_p->Actions.pipelineBrowserDock, propDock, Qt::Vertical);
    }

    // Load the filter and source XML configurations like
    // ParaView's branded_paraview_initializer.cxx.in would:
    auto pvapp = pqPVApplicationCore::instance();
    QString root(":/cmbPostProcessingModePlugin/Configuration");
    QDir dir(root);
    QStringList files = dir.entryList(QDir::Files);
    foreach (QString file, files)
    {
      pvapp->loadConfiguration(root + QString("/") + file);
    }
    // Loading these configurations creates some toolbars
    // we want to disable when not in post-processing mode.
    // Collect them:
    m_p->PostProcessingToolBars.push_back(findToolBar(mw, "&Common"));
    m_p->PostProcessingToolBars.push_back(findToolBar(mw, "&Data Analysis"));
    m_p->PostProcessingToolBars.push_back(findToolBar(mw, "VCR Controls"));
    m_p->PostProcessingToolBars.push_back(findToolBar(mw, "Current Time Controls"));
    m_p->PostProcessingActions.push_back(
      findAction(findToolBar(mw, "Main Controls"), "Find data..."));
    m_p->PostProcessingActions.push_back(findActionByName(mw, "actionFileLoadServerState"));
    m_p->PostProcessingActions.push_back(findActionByName(mw, "actionFileSaveServerState"));
  }

  if (!m_menusBuilt)
  {
    std::cerr << "Warning: Could not add post-processing menu items. Sorry.\n";
  }
}
