/* Distributed under the OSI-approved BSD 3-Clause License.  See accompanying
   file Copyright.txt or https://cmake.org/licensing for details.  */
#include "cmFindProgramCommand.h"

#include <algorithm>
#include <string>
#include <utility>
#include <version>
#include <mutex>

#if _MSVC_LANG == 201703L || __cplusplus == 201703L
#define CMFINDPROGRAMCOMMAND_EXECUTION
#endif

#ifdef CMFINDPROGRAMCOMMAND_EXECUTION
#include <execution>
#endif

#include "cmMakefile.h"
#include "cmMessageType.h"
#include "cmPolicies.h"
#include "cmStateTypes.h"
#include "cmStringAlgorithms.h"
#include "cmSystemTools.h"
#include "cmValue.h"
#include "cmWindowsRegistry.h"

class cmExecutionStatus;

#if defined(__APPLE__)
#  include <CoreFoundation/CoreFoundation.h>
#endif

enum struct CMP0109Warning: uint8_t {
  NoWarning = 0,
  NotReadable = 1,
  NotExecutable = 2,
};

struct cmFindProgramHelper
{
  cmFindProgramHelper(std::string debugName, cmMakefile* makefile,
                      cmFindBase const* base)
    : DebugSearches(std::move(debugName), base)
    , Makefile(makefile)
    , FindBase(base)
    , PolicyCMP0109(makefile->GetPolicyStatus(cmPolicies::CMP0109))
  {
#if defined(_WIN32) || defined(__CYGWIN__) || defined(__MINGW32__)
    // Consider platform-specific extensions.
    this->Extensions.push_back(".com");
    this->Extensions.push_back(".exe");
#endif
    // Consider original name with no extensions.
    this->Extensions.emplace_back();
  }

  // List of valid extensions.
  std::vector<std::string> Extensions;

  // Current names under consideration.
  std::vector<std::string> Names;

  // Debug state
  cmFindBaseDebugState DebugSearches;
  cmMakefile* Makefile;
  cmFindBase const* FindBase;

  cmPolicies::PolicyStatus PolicyCMP0109;

  void AddName(std::string const& name) { this->Names.push_back(name); }
  void SetName(std::string const& name)
  {
    this->Names.clear();
    this->AddName(name);
  }
  void AddCompoundNames(std::vector<std::string>& paths)
  {
    for (auto const& name : this->Names) {
      // Only perform search relative to current directory
      // if the file name contains a directory separator.
      if (name.find('/') == std::string::npos) {
        continue;
      }
      this->AddDirectoryForName("", name, paths);
    }
  }
  void AddDirectory(std::string const& path, std::vector<std::string>& paths)
  {
    for (auto const& name : this->Names) {
      this->AddDirectoryForName(path, name, paths);
    }
  }
  void AddDirectoryForName(std::string const& path, std::string const& name,
                           std::vector<std::string>& paths)
  {
    for (auto const& extension : this->Extensions) {
      if (!extension.empty() && cmHasSuffix(name, extension)) {
        continue;
      }
      auto test_name_ext = cmStrCat(name, extension);
      paths.emplace_back(cmSystemTools::CollapseFullPath(test_name_ext, path));
    }
  }

  bool FileIsValid(std::string const& file) const
  {
    if (!this->FileIsExecutableCMP0109(file)) {
      return false;
    }
#ifdef _WIN32
    // Pretend the Windows "python" app installer alias does not exist.
    if (cmSystemTools::LowerCase(file).find("/windowsapps/python") !=
        std::string::npos) {
      std::string dest;
      if (cmSystemTools::ReadSymlink(file, dest) &&
          cmHasLiteralSuffix(dest, "\\AppInstallerPythonRedirector.exe")) {
        return false;
      }
    }
#endif
    return this->FindBase->Validate(file);
  }
  bool FileIsExecutableCMP0109(std::string const& file) const
  {
    switch (this->PolicyCMP0109) {
      case cmPolicies::OLD:
        return cmSystemTools::FileExists(file, true);
      case cmPolicies::NEW:
      case cmPolicies::REQUIRED_ALWAYS:
      case cmPolicies::REQUIRED_IF_USED:
        return cmSystemTools::FileIsExecutable(file);
      default:
        break;
    }
    bool const isExeOld = cmSystemTools::FileExists(file, true);
    bool const isExeNew = cmSystemTools::FileIsExecutable(file);
    if (isExeNew == isExeOld) {
      return isExeNew;
    }
    if (isExeNew) {
      this->Makefile->IssueMessage(
        MessageType::AUTHOR_WARNING,
        cmStrCat(cmPolicies::GetPolicyWarning(cmPolicies::CMP0109),
                 "\n"
                 "The file\n"
                 "  ",
                 file,
                 "\n"
                 "is executable but not readable.  "
                 "CMake is ignoring it for compatibility."));
    } else {
      this->Makefile->IssueMessage(
        MessageType::AUTHOR_WARNING,
        cmStrCat(cmPolicies::GetPolicyWarning(cmPolicies::CMP0109),
                 "\n"
                 "The file\n"
                 "  ",
                 file,
                 "\n"
                 "is readable but not executable.  "
                 "CMake is using it for compatibility."));
    }
    return isExeOld;
  }
};

static bool FileIsExecutableCMP0109(std::string const& file,
                                    cmPolicies::PolicyStatus policy_cmp_0109,
                                    CMP0109Warning& cmp_warning)
{
  switch (policy_cmp_0109) {
    case cmPolicies::OLD:
      return cmSystemTools::FileExists(file, true);
    case cmPolicies::NEW:
    case cmPolicies::REQUIRED_ALWAYS:
    case cmPolicies::REQUIRED_IF_USED:
      return cmSystemTools::FileIsExecutable(file);
    default:
      break;
  }
  bool const isExeOld = cmSystemTools::FileExists(file, true);
  bool const isExeNew = cmSystemTools::FileIsExecutable(file);
  if (isExeNew == isExeOld) {
    return isExeNew;
  }

  if (isExeNew) {
    cmp_warning = CMP0109Warning::NotReadable;
  } else {
    cmp_warning = CMP0109Warning::NotExecutable;
  }

  return isExeOld;
}

static bool FileIsValid(std::string const& file,
                        cmPolicies::PolicyStatus policy_cmp_0109,
                        std::mutex& validate_mtx,
                        cmFindBase const* FindBase,
                        CMP0109Warning& cmp_warning)
{
  if (!FileIsExecutableCMP0109(file, policy_cmp_0109, cmp_warning)) {
    return false;
  }
#ifdef _WIN32
  // Pretend the Windows "python" app installer alias does not exist.
  if (cmSystemTools::LowerCase(file).find("/windowsapps/python") !=
      std::string::npos) {
    std::string dest;
    if (cmSystemTools::ReadSymlink(file, dest) &&
        cmHasLiteralSuffix(dest, "\\AppInstallerPythonRedirector.exe")) {
      return false;
    }
  }
#endif

  std::unique_lock validate_lg(validate_mtx);
  return FindBase->Validate(file);
}

cmFindProgramCommand::cmFindProgramCommand(cmExecutionStatus& status)
  : cmFindBase("find_program", status)
{
  this->NamesPerDirAllowed = true;
  this->VariableDocumentation = "Path to a program.";
  this->VariableType = cmStateEnums::FILEPATH;
  // Windows Registry views
  // When policy CMP0134 is not NEW, rely on previous behavior:
  if (this->Makefile->GetPolicyStatus(cmPolicies::CMP0134) !=
      cmPolicies::NEW) {
    if (this->Makefile->GetDefinition("CMAKE_SIZEOF_VOID_P") == "8") {
      this->RegistryView = cmWindowsRegistry::View::Reg64_32;
    } else {
      this->RegistryView = cmWindowsRegistry::View::Reg32_64;
    }
  } else {
    this->RegistryView = cmWindowsRegistry::View::Both;
  }
}

// cmFindProgramCommand
bool cmFindProgramCommand::InitialPass(std::vector<std::string> const& argsIn)
{

  this->CMakePathName = "PROGRAM";

  // call cmFindBase::ParseArguments
  if (!this->ParseArguments(argsIn)) {
    return false;
  }
  this->DebugMode = this->ComputeIfDebugModeWanted(this->VariableName);

  if (this->AlreadyDefined) {
    this->NormalizeFindResult();
    return true;
  }

  std::string const result = this->FindProgram();
  this->StoreFindResult(result);
  return true;
}

std::string cmFindProgramCommand::FindProgram()
{
  std::string program;

  if (this->SearchAppBundleFirst || this->SearchAppBundleOnly) {
    program = this->FindAppBundle();
  }
  if (program.empty() && !this->SearchAppBundleOnly) {
    program = this->FindNormalProgram();
  }

  if (program.empty() && this->SearchAppBundleLast) {
    program = this->FindAppBundle();
  }
  return program;
}

std::string cmFindProgramCommand::FindNormalProgram()
{
  if (this->NamesPerDir) {
    return this->FindNormalProgramNamesPerDir();
  }
  return this->FindNormalProgramDirsPerName();
}

std::string cmFindProgramCommand::FindNormalProgramImpl(
  std::vector<std::string> const& paths)
{
  auto policy_cmp_0109 = this->Makefile->GetPolicyStatus(cmPolicies::CMP0109);
  std::mutex validate_mtx;
  std::vector<CMP0109Warning> file_cmp_warnings;
  file_cmp_warnings.resize(paths.size());

  //printf("FindNormalProgramImpl paths: %zu\n", paths.size());

  auto best_path = std::find_if(
#ifdef CMFINDPROGRAMCOMMAND_EXECUTION
    std::execution::par,
#endif
    paths.begin(), paths.end(),
    [this, &paths, &file_cmp_warnings, policy_cmp_0109,
     &validate_mtx](std::string const& path) {
      auto path_index = paths.data() - &path;
      CMP0109Warning cmp_warning = CMP0109Warning::NoWarning;
      auto file_valid =
        FileIsValid(path, policy_cmp_0109, validate_mtx, this, cmp_warning);
      if (cmp_warning != CMP0109Warning::NoWarning) {
        file_cmp_warnings[path_index] = cmp_warning;
      }
      return file_valid;
    });

  cmFindBaseDebugState debug_searches(this->FindCommandName, this);

  for (auto path_it = paths.begin(); path_it != best_path; ++path_it) {
    auto path_idx = std::distance(paths.begin(), path_it);
    if (file_cmp_warnings[path_idx] == CMP0109Warning::NotReadable) {
      this->Makefile->IssueMessage(
        MessageType::AUTHOR_WARNING,
        cmStrCat(cmPolicies::GetPolicyWarning(cmPolicies::CMP0109),
                 "\n"
                 "The file\n"
                 "  ",
                 *path_it,
                 "\n"
                 "is executable but not readable.  "
                 "CMake is ignoring it for compatibility."));
    } else if (file_cmp_warnings[path_idx] == CMP0109Warning::NotExecutable) {
      this->Makefile->IssueMessage(
        MessageType::AUTHOR_WARNING,
        cmStrCat(cmPolicies::GetPolicyWarning(cmPolicies::CMP0109),
                 "\n"
                 "The file\n"
                 "  ",
                 *path_it,
                 "\n"
                 "is readable but not executable.  "
                 "CMake is using it for compatibility."));
    }
    debug_searches.FailedAt(*path_it);
  }

  if (best_path != paths.end()) {
    auto best_path_idx = std::distance(paths.begin(), best_path);
    debug_searches.FoundAt(*best_path);
    return *best_path;
  }

  // Couldn't find the program.
  return "";
}

std::string cmFindProgramCommand::FindNormalProgramNamesPerDir()
{
  // Search for all names in each directory.
  cmFindProgramHelper helper(this->FindCommandName, this->Makefile, this);
  for (std::string const& n : this->Names) {
    helper.AddName(n);
  }

  std::vector<std::string> paths;

  // Check for the names themselves if they contain a directory separator.
  helper.AddCompoundNames(paths);

  // Search every directory.
  for (std::string const& sp : this->SearchPaths) {
    helper.AddDirectory(sp, paths);
  }

  return FindNormalProgramImpl(paths);
}

std::string cmFindProgramCommand::FindNormalProgramDirsPerName()
{
  // Search the entire path for each name.
  cmFindProgramHelper helper(this->FindCommandName, this->Makefile, this);
  std::vector<std::string> paths;

  for (std::string const& n : this->Names) {
    // Switch to searching for this name.
    helper.SetName(n);

    // Check for the names themselves if they contain a directory separator.
    helper.AddCompoundNames(paths);

    // Search every directory.
    for (std::string const& sp : this->SearchPaths) {
      helper.AddDirectory(sp, paths);
    }
  }

  return FindNormalProgramImpl(paths);
}

std::string cmFindProgramCommand::FindAppBundle()
{
  for (std::string const& name : this->Names) {

    std::string appName = name + std::string(".app");
    std::string appPath =
      cmSystemTools::FindDirectory(appName, this->SearchPaths, true);

    if (!appPath.empty()) {
      std::string executable = this->GetBundleExecutable(appPath);
      if (!executable.empty()) {
        return cmSystemTools::CollapseFullPath(executable);
      }
    }
  }

  // Couldn't find app bundle
  return "";
}

std::string cmFindProgramCommand::GetBundleExecutable(
  std::string const& bundlePath)
{
  std::string executable;
  (void)bundlePath;
#if defined(__APPLE__)
  // Started with an example on developer.apple.com about finding bundles
  // and modified from that.

  // Get a CFString of the app bundle path
  // XXX - Is it safe to assume everything is in UTF8?
  CFStringRef bundlePathCFS = CFStringCreateWithCString(
    kCFAllocatorDefault, bundlePath.c_str(), kCFStringEncodingUTF8);

  // Make a CFURLRef from the CFString representation of the
  // bundle’s path.
  CFURLRef bundleURL = CFURLCreateWithFileSystemPath(
    kCFAllocatorDefault, bundlePathCFS, kCFURLPOSIXPathStyle, true);

  // Make a bundle instance using the URLRef.
  CFBundleRef appBundle = CFBundleCreate(kCFAllocatorDefault, bundleURL);

  // returned executableURL is relative to <appbundle>/Contents/MacOS/
  CFURLRef executableURL = CFBundleCopyExecutableURL(appBundle);

  if (executableURL != nullptr) {
    const int MAX_OSX_PATH_SIZE = 1024;
    UInt8 buffer[MAX_OSX_PATH_SIZE];

    if (CFURLGetFileSystemRepresentation(executableURL, false, buffer,
                                         MAX_OSX_PATH_SIZE)) {
      executable = bundlePath + "/Contents/MacOS/" +
        std::string(reinterpret_cast<char*>(buffer));
    }
    // Only release CFURLRef if it's not null
    CFRelease(executableURL);
  }

  // Any CF objects returned from functions with "create" or
  // "copy" in their names must be released by us!
  CFRelease(bundlePathCFS);
  CFRelease(bundleURL);
  CFRelease(appBundle);
#endif

  return executable;
}

bool cmFindProgram(std::vector<std::string> const& args,
                   cmExecutionStatus& status)
{
  return cmFindProgramCommand(status).InitialPass(args);
}
