/*=========================================================================

  Program:   Visualization Toolkit
  Module:    vtkPythonInterpreter.cxx

  Copyright (c) Ken Martin, Will Schroeder, Bill Lorensen
  All rights reserved.
  See Copyright.txt or http://www.kitware.com/Copyright.htm for details.

     This software is distributed WITHOUT ANY WARRANTY; without even
     the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
     PURPOSE.  See the above copyright notice for more information.

=========================================================================*/
#include "vtkPython.h" // this must be the first include.
#include "vtkPythonInterpreter.h"
#include "vtkPythonInterpreterConfigure.h"

#include "vtkCommand.h"
#include "vtkObjectFactory.h"
#include "vtkWeakPointer.h"
#include "vtkPythonStdStreamCaptureHelper.h"

#include <vtksys/SystemTools.hxx>

#include <algorithm>
#include <string>
#include <vector>
#include <signal.h>

#if defined(__linux)
# ifndef _GNU_SOURCE
#   define _GNU_SOURCE
# endif
#   include <dlfcn.h>
#elif defined(__unix)
# include <dlfcn.h>
#endif

#if PY_VERSION_HEX >= 0x03000000
#if defined(__APPLE__) && PY_VERSION_HEX < 0x03050000
extern "C" {
extern wchar_t*
_Py_DecodeUTF8_surrogateescape(const char *s, Py_ssize_t size);
}
#endif
#endif

/* Python major.minor version string.  */
#define VTK_PYTHON_TO_STRING(x) VTK_PYTHON_TO_STRING0(x)
#define VTK_PYTHON_TO_STRING0(x) VTK_PYTHON_TO_STRING1(x)
#define VTK_PYTHON_TO_STRING1(x) #x
#define VTK_PYTHON_VERSION VTK_PYTHON_TO_STRING(PY_MAJOR_VERSION.PY_MINOR_VERSION)

namespace
{
  class StringPool
    {
  public:
    std::vector<char*> Strings;
    ~StringPool()
      {
      for (size_t cc=0; cc < this->Strings.size(); cc++)
        {
        delete [] this->Strings[cc];
        }
      }
    };

  static std::vector<vtkWeakPointer<vtkPythonInterpreter> > GlobalInterpreters;
  static std::vector<std::string> PythonPaths;

  void NotifyInterpreters(unsigned long eventid, void *calldata=NULL)
    {
    std::vector<vtkWeakPointer<vtkPythonInterpreter> >::iterator iter;
    for (iter = GlobalInterpreters.begin(); iter != GlobalInterpreters.end();
      ++iter)
      {
      if (iter->GetPointer())
        {
        iter->GetPointer()->InvokeEvent(eventid, calldata);
        }
      }
    }

  inline void vtkPrependPythonPath(const char* pathtoadd)
    {
    vtkPythonScopeGilEnsurer gilEnsurer;
    PyObject* path = PySys_GetObject(const_cast<char*>("path"));
#if PY_VERSION_HEX >= 0x03000000
    PyObject* newpath = PyUnicode_FromString(pathtoadd);
#else
    PyObject* newpath = PyString_FromString(pathtoadd);
#endif
    PyList_Insert(path, 0, newpath);
    Py_DECREF(newpath);
    }

  std::string GetVTKPythonLibDir(void)
  {
#if defined(__linux) || defined(__unix)
    // Use the library location of VTK's python wrapping as a prefix to search
    void *handle = dlsym(RTLD_NEXT, "PyVTKObject_New");
    if(!handle)
    {
      return "";
    }

    Dl_info di;
    int ret = dladdr(handle, &di);
    if(ret == 0 || !di.dli_saddr || !di.dli_fname)
    {
      return "";
    }

    return vtksys::SystemTools::GetFilenamePath(di.dli_fname);

#elif defined(WIN32)
    // Use the path to the running exe as the start of a search prefix
    HMODULE handle = GetModuleHandle(NULL);
    if(!handle)
    {  // Can't find ourselves????? this shouldn't happen
      return "";
    }

    TCHAR path[MAX_PATH];
    if(!GetModuleFileName(handle, path, MAX_PATH))
    {
      return "";
    }

    return vtksys::SystemTools::GetFilenamePath(path);
#endif
  }


  void LocateVTKPythonModule(const std::string &start_dir,
    std::vector<std::string> &module_paths)
  {
    // Check for paths in a build tree
    const char* build_dirs[] = {
      "/../Wrapping/Python",
      "/../VTK/Wrapping/Python",
      0 };

    for (const char** build_dir = build_dirs; *build_dir; ++build_dir)
    {
      std::string package_dir = start_dir;
#if defined(CMAKE_INTDIR)
      package_dir += "/..";
#endif
      package_dir += (*build_dir);
      package_dir = vtksys::SystemTools::CollapseFullPath(package_dir);

      // Try to locate the directory containing vtk python module files.
      if(vtksys::SystemTools::FileIsDirectory(package_dir+"/vtk"))
      {
        module_paths.push_back(package_dir);
        module_paths.push_back(VTK_PYTHON_LIBRARY_DIR);
        return;
      }
    }

    // Check for paths in an install tree
    const char* inst_dirs[] = {
      // *NIX paths
      "/lib/python" VTK_PYTHON_VERSION "/site-packages",
      "/python" VTK_PYTHON_VERSION "/site-packages",
      "/site-packages",
      "/lib/python",

      // Apple
      "/../Library/Python/" VTK_PYTHON_VERSION "/site-packages",

      // Windows
      "/Lib/site-packages",
      "/Lib",
      "/site-packages",
      0
    };

    std::string prefix = vtksys::SystemTools::GetFilenamePath(start_dir);
    for(const char** dir = inst_dirs; *dir; ++dir)
    {
      std::string package_dir = prefix;
      package_dir += *dir;
      package_dir = vtksys::SystemTools::CollapseFullPath(package_dir);
      if(vtksys::SystemTools::FileIsDirectory(package_dir+"/vtk"))
      {
        module_paths.push_back(package_dir);
        return;
      }
    }
  }
}

bool vtkPythonInterpreter::InitializedOnce = false;
bool vtkPythonInterpreter::CaptureStdin = false;
bool vtkPythonInterpreter::ConsoleBuffering = false;
std::string vtkPythonInterpreter::StdErrBuffer;
std::string vtkPythonInterpreter::StdOutBuffer;

vtkStandardNewMacro(vtkPythonInterpreter);
//----------------------------------------------------------------------------
vtkPythonInterpreter::vtkPythonInterpreter()
{
  GlobalInterpreters.push_back(this);
}

//----------------------------------------------------------------------------
vtkPythonInterpreter::~vtkPythonInterpreter()
{
  std::vector<vtkWeakPointer<vtkPythonInterpreter> >::iterator iter;
  for (iter = GlobalInterpreters.begin(); iter != GlobalInterpreters.end();
    ++iter)
    {
    if (*iter == this)
      {
      GlobalInterpreters.erase(iter);
      break;
      }
    }
}

//----------------------------------------------------------------------------
bool vtkPythonInterpreter::IsInitialized()
{
  return (Py_IsInitialized() != 0);
}

//----------------------------------------------------------------------------
void vtkPythonInterpreter::PrintSelf(ostream& os, vtkIndent indent)
{
  this->Superclass::PrintSelf(os, indent);
}

//----------------------------------------------------------------------------
bool vtkPythonInterpreter::Initialize(int initsigs /*=0*/)
{
  if (Py_IsInitialized() == 0)
  {
#if defined(__linux) || defined(__unix)
    // On *Nix, use the path to the python library to initialize
    {
      void *handle = dlsym(RTLD_NEXT, "Py_SetProgramName");
      if(handle)
      {
        Dl_info di;
        int ret = dladdr(handle, &di);
        if (ret != 0 && di.dli_saddr && di.dli_fname)
        { 
          Py_SetProgramName(const_cast<char *>(di.dli_fname));
        }
      }
    }
#endif


#if (VTK_PYTHON_MAJOR_VERSION > 2) ||\
    (VTK_PYTHON_MAJOR_VERSION == 2 && VTK_PYTHON_MINOR_VERSION >= 4)
    Py_InitializeEx(initsigs);
#else
    (void)initsigs;
    Py_Initialize();
#endif

#ifdef SIGINT
    // Put default SIGINT handler back after Py_Initialize/Py_InitializeEx.
    signal(SIGINT, SIG_DFL);
#endif
  }

  if (!vtkPythonInterpreter::InitializedOnce)
  {
    vtkPythonInterpreter::InitializedOnce = true;

#ifdef VTK_PYTHON_FULL_THREADSAFE
    int threadInit = PyEval_ThreadsInitialized();
    PyEval_InitThreads(); // safe to call this multiple time
    if(!threadInit)
    {
      PyEval_SaveThread(); // release GIL
    }
#endif

    // HACK: Calling PyRun_SimpleString for the first time for some reason results in
    // a "\n" message being generated which is causing the error dialog to
    // popup. So we flush that message out of the system before setting up the
    // callbacks.
    vtkPythonInterpreter::RunSimpleString("");

    // Setup handlers for stdout/stdin/stderr.
    vtkPythonStdStreamCaptureHelper* wrapperOut =
      NewPythonStdStreamCaptureHelper(false);
    vtkPythonStdStreamCaptureHelper* wrapperErr =
      NewPythonStdStreamCaptureHelper(true);

    // Redirect Python's stdout and stderr and stdin - GIL protected operation
    {
      vtkPythonScopeGilEnsurer gilEnsurer;
      PySys_SetObject(const_cast<char*>("stdout"),
        reinterpret_cast<PyObject*>(wrapperOut));
      PySys_SetObject(const_cast<char*>("stderr"),
        reinterpret_cast<PyObject*>(wrapperErr));
      PySys_SetObject(const_cast<char*>("stdin"),
        reinterpret_cast<PyObject*>(wrapperOut));
      Py_DECREF(wrapperOut);
      Py_DECREF(wrapperErr);
    }

    std::string start_dir = GetVTKPythonLibDir();
    if(!start_dir.empty())
    {
      LocateVTKPythonModule(start_dir, PythonPaths);
    }

    for (size_t cc=0; cc < PythonPaths.size(); cc++)
    {
      vtkPrependPythonPath(PythonPaths[cc].c_str());
    }

    NotifyInterpreters(vtkCommand::EnterEvent);
    return true;
  }

  return false;
}

//----------------------------------------------------------------------------
void vtkPythonInterpreter::Finalize()
{
  if (Py_IsInitialized() != 0)
    {
    NotifyInterpreters(vtkCommand::ExitEvent);
    vtkPythonScopeGilEnsurer gilEnsurer(false, true); 
    // Py_Finalize will take care of relasing gil
    Py_Finalize();
    }
}

//----------------------------------------------------------------------------
void vtkPythonInterpreter::SetProgramName(const char* programname)
{
  if (vtkPythonInterpreter::InitializedOnce ||
    Py_IsInitialized() != 0)
    {
    return;
    }

  if (programname)
    {
    static StringPool pool;
    pool.Strings.push_back(vtksys::SystemTools::DuplicateString(programname));

    // From Python Docs: The argument should point to a zero-terminated character
    // string in static storage whose contents will not change for the duration of
    // the program's execution. No code in the Python interpreter will change the
    // contents of this storage.
#if PY_VERSION_HEX >= 0x03000000
    wchar_t *argv0;
    const std::string& av0 = pool.Strings.back();
#if PY_VERSION_HEX >= 0x03050000
    argv0 = Py_DecodeLocale(av0.c_str(), NULL);
#elif defined(__APPLE__)
    argv0 = _Py_DecodeUTF8_surrogateescape(av0.data(), av0.length());
#else
    argv0 = _Py_char2wchar(av0.c_str(), NULL);
#endif
    if (argv0 == 0)
      {
      fprintf(stderr, "Fatal vtkpython error: "
                      "unable to decode the program name\n");
      static wchar_t empty[1] = { 0 };
      argv0 = empty;
      }
    Py_SetProgramName(argv0);
#else
    Py_SetProgramName(pool.Strings.back());
#endif
    }
}

//----------------------------------------------------------------------------
void vtkPythonInterpreter::PrependPythonPath(const char* dir)
{
  if (!dir) { return; }

  std::string out_dir = dir;

#if defined(_WIN32) && !defined(__CYGWIN__)
  // Convert slashes for this platform.
  std::replace(out_dir.begin(), out_dir.end(), '/', '\\');
#endif

  // save path for future use.
  PythonPaths.push_back(out_dir);

  if (Py_IsInitialized() == 0)
    {
    return;
    }

  // Append the path to the python sys.path object.
  vtkPrependPythonPath(out_dir.c_str());
}

//----------------------------------------------------------------------------
int vtkPythonInterpreter::PyMain(int argc, char** argv)
{
  if (!vtkPythonInterpreter::InitializedOnce && Py_IsInitialized() == 0 &&
    argc > 0)
    {
    vtkPythonInterpreter::SetProgramName(argv[0]);
    }
  vtkPythonInterpreter::Initialize(1);

#if PY_VERSION_HEX >= 0x03000000
  wchar_t *argv0;
#if PY_VERSION_HEX >= 0x03050000
  argv0 = Py_DecodeLocale(argv[0], NULL);
#elif defined(__APPLE__)
  argv0 = _Py_DecodeUTF8_surrogateescape(argv[0], strlen(argv[0]));
#else
  argv0 = _Py_char2wchar(argv[0], NULL);
#endif
    if (argv0 == 0)
      {
      static wchar_t empty[1] = { 0 };
      argv0 = empty;
      }
  // Need two copies of args, because programs might modify the first
  wchar_t **argvWide = new wchar_t *[argc];
  wchar_t **argvWide2 = new wchar_t *[argc];
  for (int i = 0; i < argc; i++)
    {
#if PY_VERSION_HEX >= 0x03050000
    argvWide[i] = Py_DecodeLocale(argv[i], NULL);
#elif defined(__APPLE__)
    argvWide[i] = _Py_DecodeUTF8_surrogateescape(argv[i], strlen(argv[i]));
#else
    argvWide[i] = _Py_char2wchar(argv[i], NULL);
#endif
    argvWide2[i] = argvWide[i];
    if (argvWide[i] == 0)
      {
      fprintf(stderr, "Fatal vtkpython error: "
                      "unable to decode the command line argument #%i\n",
                      i + 1);
      for (int k = 0; k < i; k++)
        {
        PyMem_Free(argvWide2[i]);
        }
      PyMem_Free(argv0);
      delete [] argvWide;
      delete [] argvWide2;
      return 1;
      }
    }
  vtkPythonScopeGilEnsurer gilEnsurer;
  int res = Py_Main(argc, argvWide);
  PyMem_Free(argv0);
  for (int i = 0; i < argc; i++)
    {
    PyMem_Free(argvWide2[i]);
    }
  delete [] argvWide;
  delete [] argvWide2;
  return res;
#else

  vtkPythonScopeGilEnsurer gilEnsurer(false, true);
  return Py_Main(argc, argv);
#endif
}

//----------------------------------------------------------------------------
int vtkPythonInterpreter::RunSimpleString(const char* script)
{
  vtkPythonInterpreter::Initialize(1);
  vtkPythonInterpreter::ConsoleBuffering = true;

  // The embedded python interpreter cannot handle DOS line-endings, see
  // http://sourceforge.net/tracker/?group_id=5470&atid=105470&func=detail&aid=1167922
  std::string buffer = script ? script : "";
  buffer.erase(std::remove(buffer.begin(), buffer.end(), '\r'), buffer.end());

  // The cast is necessary because PyRun_SimpleString() hasn't always been const-correct
  int pyReturn;
    {
    vtkPythonScopeGilEnsurer gilEnsurer;
    pyReturn = PyRun_SimpleString(const_cast<char*>(buffer.c_str()));
    }

  vtkPythonInterpreter::ConsoleBuffering = false;
  if (! vtkPythonInterpreter::StdErrBuffer.empty())
    {
    NotifyInterpreters(vtkCommand::ErrorEvent, const_cast<char*>(
                         vtkPythonInterpreter::StdErrBuffer.c_str()));
    vtkPythonInterpreter::StdErrBuffer.clear();
    }
  if (! vtkPythonInterpreter::StdOutBuffer.empty())
    {
    NotifyInterpreters(vtkCommand::SetOutputEvent, const_cast<char*>(
                         vtkPythonInterpreter::StdOutBuffer.c_str()));
    vtkPythonInterpreter::StdOutBuffer.clear();
    }

  return pyReturn;
}

//----------------------------------------------------------------------------
void vtkPythonInterpreter::SetCaptureStdin(bool val)
{
  vtkPythonInterpreter::CaptureStdin = val;
}

//----------------------------------------------------------------------------
bool vtkPythonInterpreter::GetCaptureStdin()
{
  return vtkPythonInterpreter::CaptureStdin;
}

//----------------------------------------------------------------------------
void vtkPythonInterpreter::WriteStdOut(const char* txt)
{
  cout << txt;
  if (vtkPythonInterpreter::ConsoleBuffering)
    {
    vtkPythonInterpreter::StdOutBuffer += std::string(txt);
    }
  else
    {
    NotifyInterpreters(vtkCommand::SetOutputEvent, const_cast<char*>(txt));
    }
}

//----------------------------------------------------------------------------
void vtkPythonInterpreter::FlushStdOut()
{
    cout.flush();
}

//----------------------------------------------------------------------------
void vtkPythonInterpreter::WriteStdErr(const char* txt)
{
  cerr << txt;
  if (vtkPythonInterpreter::ConsoleBuffering)
    {
    vtkPythonInterpreter::StdErrBuffer += std::string(txt);
    }
  else
    {
    NotifyInterpreters(vtkCommand::ErrorEvent, const_cast<char*>(txt));
    }
}

//----------------------------------------------------------------------------
void vtkPythonInterpreter::FlushStdErr()
{
    cerr.flush();
}

//----------------------------------------------------------------------------
vtkStdString vtkPythonInterpreter::ReadStdin()
{
  if (!vtkPythonInterpreter::CaptureStdin)
    {
    vtkStdString string;
    cin >> string;
    return string;
    }
  vtkStdString string;
  NotifyInterpreters(vtkCommand::UpdateEvent, &string);
  return string;
}
